
/*

    File: ip-lib.c

    Copyright (C) 1999,2004-2007  Wolfgang Zekoll  <wzk@quietsche-entchen.de>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */
 
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include <signal.h>
#include <syslog.h>

#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <net/if.h>
#include <sys/ioctl.h>
#include <netdb.h>
#include <errno.h>

#include "lib.h"
#include "ip-lib.h"
#include "tcpproxy.h"


int issock(int pfd)
{
	unsigned int size;
	struct sockaddr_in saddr;

	size = sizeof(saddr);
	if (getsockname(pfd, (struct sockaddr *) &saddr, &size) < 0)
		return (1);

	return (0);
}

unsigned int get_interface_info(int pfd, peer_t *sock)
{
	unsigned int size;
	struct sockaddr_in saddr;

	size = sizeof(saddr);
	if (getsockname(pfd, (struct sockaddr *) &saddr, &size) < 0)
		printerror(1, "-ERR", "can't get sockname, error= %s", strerror(errno));

	copy_string(sock->ipnum, (char *) inet_ntoa(saddr.sin_addr), sizeof(sock->ipnum));
	sock->port = ntohs(saddr.sin_port);
	copy_string(sock->name, sock->ipnum, sizeof(sock->name));

	return (sock->port);
}

int get_client_info(int pfd, peer_t *peer, int ipnumonly)
{
	unsigned int port;
	unsigned int size;
	struct sockaddr_in saddr;

	*peer->ipnum = 0;
	*peer->name  = 0;
	size = sizeof(saddr);
	if (getpeername(pfd, (struct sockaddr *) &saddr, &size) < 0)
		printerror(1, "-ERR", "can't get peername, error= %s", strerror(errno));

	copy_string(peer->ipnum, (char *) inet_ntoa(saddr.sin_addr), sizeof(peer->ipnum));
	peer->port = ntohs(saddr.sin_port);
	copy_string(peer->name, peer->ipnum, sizeof(peer->name));

	if (ipnumonly == 0) {
		struct in_addr *addr;
		struct hostent *hostp = NULL;

		addr = &saddr.sin_addr;
		hostp = gethostbyaddr((char *) addr,
				sizeof (saddr.sin_addr.s_addr), AF_INET);

		if (hostp != NULL) {
			copy_string(peer->name, hostp->h_name, sizeof(peer->name));
			strlwr(peer->name);
			}
		}

	port = ntohs(saddr.sin_port);
	return (port);
}


int openip(char *host, unsigned int port, char *srcip, unsigned int srcport,
			peer_t *peer)
{
	int	socketd;
	char	*p;
	struct sockaddr_in server;
	struct hostent *hostp, *gethostbyname();

	socketd = socket(AF_INET, SOCK_STREAM, 0);
	if (socketd < 0)
		return (-1);
  
  	if (srcip != NULL  &&  *srcip != 0) {
		struct sockaddr_in laddr;

		if (srcport != 0) {
			int	one;

			one = 1;
	 		setsockopt (socketd, SOL_SOCKET, SO_REUSEADDR, (int *) &one, sizeof(one));
			}
 
 		/*
         	 * Bind local socket to srcport and srcip
         	 */

 		memset(&laddr, 0, sizeof(laddr));
 		laddr.sin_family = AF_INET;
 		laddr.sin_port   = htons(srcport);

 		if (srcip == NULL  ||  *srcip == 0)
 			srcip = "0.0.0.0";
 		else {
 			struct hostent *ifp;
 
 			ifp = gethostbyname(srcip);
 			if (ifp == NULL) {
				close (socketd);
				return (-1);
 				}
 
 			memcpy(&laddr.sin_addr, ifp->h_addr, ifp->h_length);
 	 	 	}
 
 		if (bind(socketd, (struct sockaddr *) &laddr, sizeof(laddr))) {
			close (socketd);
			return (-2);
			}
		}


	server.sin_family = AF_INET;
	p = host;
	while (*(p = skip_ws(p)) != 0) {
		get_quoted(&p, ',', peer->name, sizeof(peer->name));
		noctrl(peer->name);
		if (*peer->name == 0)
			continue;

		peer->port = get_port(peer->name, port);
		
		hostp = gethostbyname(peer->name);
		if (hostp == NULL) {
			printerror(0 | ERR_INFO, "-INFO", "can't lookup server %s:%u, error= %s",
					peer->name, peer->port, strerror(errno));
			continue;
			}

		memcpy(&server.sin_addr, hostp->h_addr, hostp->h_length);
		server.sin_port = htons(peer->port);
		copy_string(peer->ipnum, (char *) inet_ntoa(server.sin_addr), sizeof(peer->ipnum));

		if (connect(socketd, (struct sockaddr *) &server, sizeof(server)) >= 0)
			return (socketd);
		else {
			printerror(0 | ERR_INFO, "-INFO", "can't connect to server %s:%u, error= %s",
					peer->name, peer->port, strerror(errno));
			}
		}

	close (socketd);
 	return (-1);
}	

unsigned int getportnum(char *name)
{
	unsigned int port;
	struct servent *portdesc;
	
	if (isdigit(*name) != 0)
		port = atol(name);
	else {
		portdesc = getservbyname(name, "tcp");
		if (portdesc == NULL) {
			fprintf (stderr, "%s: service not found: %s\n", program, name);
			exit (1);
			}

		port = ntohs(portdesc->s_port);
		if (port == 0) {
			fprintf (stderr, "%s: port error: %s\n", program, name);
			exit (1);
			}
		}
	
	return (port);
}

unsigned int get_port(char *server, unsigned int def_port)
{
	unsigned int port;
	char	*p;

	if ((p = strchr(server, ':')) == NULL)
		return (def_port);

	*p++ = 0;
	port = getportnum(p);

	return (port);
}

int bind_to_port(char *interface, unsigned int port)
{
	struct sockaddr_in saddr;
	int	sock;
	char	*ipnum;

	if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		fprintf (stderr, "%s: can't create socket\n", program);
		exit (1);
		}
	else {
		int	opt;

		opt = 1;
		setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(opt));
		}


	memset(&saddr, 0, sizeof(saddr));
	saddr.sin_family = AF_INET;
	saddr.sin_port   = htons(port);

	ipnum = interface;
	if (ipnum == NULL  ||  *ipnum == 0)
		ipnum = "0.0.0.0";
	else {
		struct hostent *ifp;

		ifp = gethostbyname(ipnum);
		if (ifp == NULL)
			printerror(1, "-ERR", "can't resolve name: %s, error= %s", ipnum, strerror(errno));

		memcpy(&saddr.sin_addr, ifp->h_addr, ifp->h_length);
		}
		
		
	if (bind(sock, (struct sockaddr *) &saddr, sizeof(saddr)))
		printerror(1, "-ERR", "can't bind to %s:%u, error= %s", ipnum, port, strerror(errno));

	if (listen(sock, 5) < 0)
		printerror(1, "-ERR", "can't listen on %s:%u, error= %s", ipnum, port, strerror(errno));

	return (sock);
}


int get_phyint_addr(char *interface, char *ipnum, int size)
{
	int	sock;
	struct ifreq ifrr;

	ifrr.ifr_addr.sa_family = AF_INET;
	copy_string(ifrr.ifr_name, interface, sizeof(ifrr.ifr_name));

	if ((sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_IP)) < 0)
		printerror(1, "", "can't allocate socket, error= %s", strerror(errno));

	if (ioctl(sock, SIOCGIFADDR, &ifrr) < 0) {
		close (sock);
		return (-1);
		}

	close (sock);
	copy_string(ipnum, (char *) inet_ntoa(((struct sockaddr_in *) &ifrr.ifr_addr)->sin_addr), size - 2);

	return (0);
}



unsigned long atoip(char *string, char **r, int *error)
{
	int	i, k;
	unsigned long num;
	char	*p;

	*error = 0;
	num    = 0;

	p = string;
	for (i=0; i < 4  &&  *p != 0; i++) {
		k = (unsigned) strtol(p, &p, 10);
		if (k < 0  ||  k > 255) {
			*error = 1;
			return (1);
			}

		num = (num << 8) + k;
		if (*p == '.') {
			p++;
			if (isdigit(*p) == 0) {
				*error = 1;
				return (1);
				}
			}
		else if (i != 3) {
			*error = 1;
			return (1);
			}
		}

	if (r != NULL)
		*r = p;

	return (num);
}

int checkacl(char *acl, char *ipnum, int rc)
{
	unsigned long ip, adr, mask, weight;
	int	allowdeny, error, len;
	char	*r, *p, word[80];

/* printerror(0, "+DEBUG", "acl= %s", acl); */
	if (*acl == 0)
		return (rc);

	weight = 0;
	ip = atoip(ipnum, &p, &error);
	r = acl;
	while (*get_word(&r, word, sizeof(word)) != 0) {
		p = word;
		if (*p == '+') {
			allowdeny = ACL_ALLOW;
			p++;
			}
		else if (*p == '-') {
			allowdeny = ACL_DENY;
			p++;
			}
		else
			allowdeny = ACL_ALLOW;

		adr = atoip(p, &p, &error);
/* printerror(0, "+DEBUG", "word= %s, adr= %08X, p= %s",
 *				word, adr, p);
 */
		if (error != 0)
			return (ACL_DENY);

		mask = 0xFFFFFFFF;
		if (*p == '/') {
			p++;
			if (strchr(p, '.') != NULL)
				mask = atoip(p, &p, &error);
			else {
				len = strtoul(p, &p, 10);
				mask = 0xFFFFFFFF << (32 - len);
				}

/* printerror(0, "+DEBUG", "word= %s, adr= %08X, mask= %08X, ip= %08X, weight= %08X",
 *				word, adr, mask, ip, weight);
 */

			if (*p != 0  ||  error != 0)
				return (ACL_DENY);
			}

		if (mask > weight) {
			if ((ip & mask) == (adr & mask))
				rc = allowdeny;
			}
		}

	return (rc);
}

