
/*

    File: config.c

    Copyright (C) 2005  Wolfgang Zekoll  <wzk@quietsche-entchen.de>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */



#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include <errno.h>
#include <ctype.h>
#include <pwd.h>
#include <grp.h>


#include "lib.h"
#include "ip-lib.h"
#include "procinfo.h"
#include "tcpproxy.h"


proxy_t *new_configuration(config_t *config, proxy_t *def)
{
	proxy_t *x;

	x = allocate(sizeof(proxy_t));
	if (def != NULL)
		memmove(x, def, sizeof(proxy_t));

	if (config->first == NULL) {
		config->first = x;
		config->last  = x;
		}
	else {
		config->last->next = x;
		config->last = x;
		}

	config->proxy_count++;
	return (x);
}

proxy_t *find_configuration(config_t *config, char *interface, unsigned int port)
{
	proxy_t *x, *global;

	global = NULL;
	x = config->first;
	while (x != NULL) {
		if (x->port == port) {
			if (*x->interface == 0  ||  strcmp(x->interface, "0.0.0.0") == 0)
				global = x;
			else if (strcmp(x->interface, interface) == 0)
				break;
			}
			
		x = x->next;
		}

	if (x == NULL) {

		/*
		 * Check if we have a configuration for a matching
		 * interface name.
		 */

		x = config->first;
		while (x != NULL) {
			if (x->port == port  &&  isalpha(*x->interface)) {
				char	ipnum[80];

				if (get_phyint_addr(x->interface, ipnum, sizeof(ipnum)) < 0)
					;
				else if (strcmp(interface, ipnum) == 0)
					break;
				}

			x = x->next;
			}
		}

	if (x == NULL  &&  global != NULL)
		x = global;
		
	return (x);
}


int _check_interface(proxy_t *x, char *par, char *filename, int lineno)
{
	if (*x->interface == 0)
		printerror(1, "-ERR", "parameter requires proxy configuration: %s, %s:%d", par, filename, lineno);

	return (0);
}

int _check_proxytype(proxy_t *x, int lineno)
{
	if (x->proxytype != 0) {
		printerror(ERR_STDERR, "-ERR", "line %d: service redefinition", lineno);
		exit (1);
		}
	
	return (0);
}


int addsrcip(proxy_t *x, char *par)
{
	int	k;
	char	ip[40];

/*	if (x->proxytype == PROXY_PROGRAM)
 *		return (0);
 */

	copy_string(ip, par, sizeof(ip));
	if (x->srcip.count >= x->srcip.max) {
		x->srcip.max += 10;
		x->srcip.srcip = reallocate(x->srcip.srcip, x->srcip.max * sizeof(char *));
		}

	k = x->srcip.count++;
	x->srcip.srcip[k] = strdup(ip);

	return (0);
}

char *iptoa(unsigned long ip, char *str, int size)
{
	unsigned char *p;

	p = (unsigned char *) &ip;
	snprintf (str, size-2, "%u.%u.%u.%u", p[3], p[2], p[1], p[0]);

	return (0);
}

int addsrc(proxy_t *x, char *ip, char *filename, int lineno)
{
	int	error;
	char	*p;
	unsigned long ipnum;

	ipnum = atoip(ip, &p, &error);
	if (error != 0)
		printerror(1, "-ERR", "error in IP number: %s, %s:%d", ip, filename, lineno);

/* fprintf (stderr, "p= %s c= %d\n", p, *p); */
	if (*p == 0)
		addsrcip(x, ip);
	else if (*p == '-') {
		unsigned int i, start, end;
		unsigned long num;
		char	adr[20];

		p++;
		start = ipnum & 0xFF;
		end = strtoul(p, &p, 10) & 0xFF;
		if (*p != 0)
			printerror(1, "-ERR", "error in IP range: %s, %s:%d", ip, filename, lineno);

/*		if (start > end) {
 *			i= end;
 *			end = start;
 *			start = i;
 *			}
 */
		for (i=start; i<=end; i++) {
			num = (ipnum & 0xFFFFFF00) | i;
			iptoa(num, adr, sizeof(adr));
			addsrcip(x, adr);
			}
		}
	else {
		printerror(1, "-ERR", "error in IP number: %s, %s:%d", ip, filename, lineno);
		}

	return (0);
}




#define	printyesno(x, y)	(y != 0? printf ("%s: %s\n", x, (y == 0)? "no": "yes"): 0)
#define printnum(x, y)		((y > 0)? printf ("%s: %u\n", x, y): 0)
#define printoct(x, y)		((y > 0)? printf ("%s: %03o\n", x, y): 0)
#define	printstring(x, y)	((y != NULL  &&  *y != 0)? printf("%s: %s\n", x, y): 0)


static int check_parameter(char **from, char *par, char *filename, int lineno)
{
	*from = skip_ws(*from);
	if (**from == 0)
		printerror(1, "-ERR", "missing parameter: %s, %s:%d", par, filename, lineno);

	return (0);
}

static int get_yesno(char **from, char *par, char *filename, int lineno)
{
	char	word[80];

	if (**from == 0)
		printerror(1, "-ERR", "missing parameter: %s, %s:%d", par, filename, lineno);

	get_word(from, word, sizeof(word));
	if (strcmp(word, "yes") == 0)
		return (1);
	else if (strcmp(word, "no") == 0)
		return (0);
	else
		printerror(1, "-ERR", "bad parameter value: %s, parameter= %s, %s:%d", word, par, filename, lineno);

	return (0);
}

static char *get_wordpar(char **from, char *par, char *word, int size,
		char *filename, int lineno)
{
	if (**from == 0)
		printerror(1, "-ERR", "missing parameter: %s, %s:%d", par, filename, lineno);

	get_word(from, word, size);
	if (strcmp(word, "-") == 0)
		*word = 0;

	*from = skip_ws(*from);
	if (**from != 0)
		printerror(1, "-ERR", "trailing characters: %s, parameter= %s, %s:%d", *from, par, filename, lineno);

	return (word);
}

static char *get_parameter(char **from, char *par, char *value, int size,
		char *filename, int lineno)
{
	if (**from == 0)
		printerror(1, "-ERR", "missing parameter: %s, %s:%d", par, filename, lineno);

	copy_string(value, *from, size);
	if (strcmp(value, "-") == 0)
		*value = 0;

	return (value);
}

static long get_number(char **from, char *par, char *filename, int lineno)
{
	long	num;
	char	*p, word[80];

	if (**from == 0)
		printerror(1, "-ERR", "missing parameter: %s, %s:%d", par, filename, lineno);

	get_word(from, word, sizeof(word));
	*from = skip_ws(*from);
	if (**from != 0)
		printerror(1, "-ERR", "bad number: %s, %s:%d", par, filename, lineno);

	num = strtol(word, &p, 10);
	if (*p != 0)
		printerror(1, "-ERR", "not a number: %s, parameter= %s, %s:%d", word, par, filename, lineno);

	return (num);
}


int set_redirmode(proxy_t *x, char *word, char *filename, int lineno)
{
	if (strcmp(word, "none") == 0  ||  strcmp(word, "off") == 0  ||
	    strcmp(word, "no") == 0) {
		x->redirmode = REDIR_NONE;
		}
	else if (strcmp(word, "accept") == 0  ||  strcmp(word, "redirect") == 0)
		x->redirmode = REDIR_ACCEPT;
	else if (strcmp(word, "forward") == 0)
		x->redirmode = REDIR_FORWARD;
	else if (strcmp(word, "forward-only") == 0)
		x->redirmode = REDIR_FORWARD_ONLY;
	else
		printerror(1, "-ERR", "bad keyword: %s, %s:%d", word, filename, lineno);

	return (0);
}

int set_username(proxy_t *x, char *username, char *filename, int lineno)
{
	char	*p, *q, word[40];
	struct passwd *pw;
	struct group *gr;

	p = username;
	get_quoted(&p, '.', word, sizeof(word));
	if (isdigit(*word) != 0) {
		x->uid = strtoul(word, &q, 10);
		if (*q != 0)
			printerror(1, "-ERR", "bad user id: %s, %s:%d", word, filename, lineno);
		}
	else if (*word != 0) {
		if ((pw = getpwnam(word)) == NULL) {
			printerror(1, "-ERR", "can't read passwd entry: %s, error= %s",
					word, strerror(errno));
			}

		x->uid = pw->pw_uid;
		if (*p == 0)
			x->gid = pw->pw_gid;
		}

	if (*p == '.') {
		p++;
		if (isdigit(*p) != 0) {
			x->gid = strtoul(p, &q, 10);
			if (*q != 0)
				printerror(1, "-ERR", "bad group id: %s, %s:%d", p, filename, lineno);
			}
		else if (*p != 0) {
			if ((gr = getgrnam(p)) == NULL) {
				printerror(1, "-ERR", "can't read group entry: %s, error= %s",
						p, strerror(errno));
				}

			x->gid = gr->gr_gid;
			}
		}

	return (0);
}


char *uncomma(char *string)
{
	int	c;
	char	*p;

	for (p = string; (c = *p) != 0; p++) {
		if (c == ',')
			*p = ' ';
		}

	p = skip_ws(string);
	return (p);
}

int read_configuration(config_t *config, char *filename)
{
	int	lineno;
	unsigned int port;
	char	*p, word[80], par[200], line[300];
	FILE	*fp;
	proxy_t *x;

	/*
	 * Open and read the configuration file.
	 */
	 
	if ((fp = fopen(filename, "r")) == NULL)
		printerror(1, "-ERR", "can't read config file: %s, error= %s", filename, strerror(errno));


	/*
	 * All parameters before the first `port' go into the default
	 * configuration profile.
	 */

	if (config->def == NULL)
		config->def = allocate(sizeof(proxy_t));

	x = config->def;

	
	lineno = 0;
	port   = 0;
	while (fgets(line, sizeof(line), fp) != NULL) {
		lineno++;

		p = skip_ws(noctrl(line));
		if (*p == 0  ||  *p == '#')
			continue;

		get_word(&p, word, sizeof(word));
		strlwr(word);
		p = skip_ws(p);

		/*
		 * Global options
		 */
		 
		if (strcmp(word, "pidfile") == 0) {
			char	pidfile[200];

			get_wordpar(&p, word, pidfile, sizeof(pidfile), filename, lineno);
			setpidfile(pidfile);
			}
		else if (strcmp(word, "standalone") == 0)
			config->standalone = get_yesno(&p, word, filename, lineno);
		else if (strcmp(word, "logname") == 0) {

			/*
			 * The logname can be configuration specific.
			 */

			if (config->binds == 0)
				get_parameter(&p, word, config->logname, sizeof(config->logname), filename, lineno);
			else
				get_parameter(&p, word, x->logname, sizeof(x->logname), filename, lineno);
			}


		/*
		 * Globale Vorgaben, bzw. Serviceparameter
		 */

		else if (strcmp(word, "timeout") == 0) {
			x->timeout = get_number(&p, word, filename, lineno);
			if (x->timeout < 1)
				x->timeout = 60;
			}
		else if (strcmp(word, "setenv") == 0) {
			get_parameter(&p, word, varname, sizeof(varname), filename, lineno);
			if (*varname == 0)
				strcpy(varname, "PROXY_");
			}
		else if (strcmp(word, "acp") == 0)
			get_parameter(&p, word, x->acp, sizeof(x->acp), filename, lineno);
		else if (strcmp(word, "ctp") == 0)
			get_parameter(&p, word, x->ctp, sizeof(x->ctp), filename, lineno);


		/*
		 * For compatibility, use new `user' option instead.
		 */

		else if (strcmp(word, "uid") == 0)
			x->uid = get_number(&p, word, filename, lineno);
		else if (strcmp(word, "gid") == 0)
			x->gid = get_number(&p, word, filename, lineno);

		else if (strcmp(word, "user") == 0) {
			char	username[40];

			get_wordpar(&p, word, username, sizeof(username), filename, lineno);
			set_username(x, username, filename, lineno);
			}

		/*
		 * server configuration
		 */
		 
		else if (strcmp(word, "port") == 0  ||  strcmp(word, "bind") == 0) {
			char	portname[40];

			if (config->binds >= MAX_PORT)
				printerror(1, "-ERR", "too many ports %s:%d", filename, lineno);

			get_wordpar(&p, "word", portname, sizeof(portname), filename, lineno);
			port = getportnum(portname);

			/*
			 * The bindport computation is done before we bind
			 * to the ports.
			 * 
			 * config->bindport[config->binds].port  = port;
			 * config->bindport[config->binds].count = 0;
			 * config->binds++;
			 */
			}
		else if (strcmp(word, "interface") == 0) {
/*			if (config->binds == 0  ||  port == 0) */
			if (port == 0)
				printerror(1, "-ERR", "no port specification: %s:%d", filename, lineno);

			x = new_configuration(config, config->def);
			x->port = port;
			get_wordpar(&p, word, x->interface, sizeof(x->interface), filename, lineno);
			if (strcmp(x->interface, "*") == 0  ||  strcmp(x->interface, "any") == 0)
				strcpy(x->interface, "0.0.0.0");
			else if (isalpha(*x->interface) != 0) {

				/*
				 * An interface name.  Check if this interface exists.
				 */

				}

			/*
			 * This is now done before we bind to the configured
			 * ports.
			 *
			 * copy_string(config->bindport[config->binds - 1].ip, x->interface, sizeof(config->bindport[config->binds - 1].ip));
			 * config->bindport[config->binds - 1].count++;
			 */
			}
		else if (strcmp(word, "server") == 0) {
			_check_interface(x, word, filename, lineno);

			get_parameter(&p, word, x->server, sizeof(x->server), filename, lineno);
			if (*x->server == '/')
				printerror(1, "-ERR", "bad server specification: %s", x->server);
			}
		else if (strcmp(word, "srcip") == 0) {
			char	ipnum[40];

			_check_interface(x, word, filename, lineno);

			get_wordpar(&p, word, ipnum, sizeof(ipnum), filename, lineno);
			addsrcip(x, ipnum);
			}
		else if (strcmp(word, "rotate") == 0) {
			_check_interface(x, word, filename, lineno);

			check_parameter(&p, word, filename, lineno);
			while (*get_word(&p, word, sizeof(word)) != 0)
				addsrc(x, word, filename, lineno);
			}
		else if (strcmp(word, "exec") == 0) {
			_check_interface(x, word, filename, lineno);

			x->proxytype = PROXY_PROGRAM;
			get_parameter(&p, word, x->server, sizeof(x->server), filename, lineno);
			if (*x->server != '/')
				printerror(1, "-ERR", "bad program specification: %s", x->server);
			}
		else if (strcmp(word, "writefile") == 0  ||  strcmp(word, "debug") == 0) {
			_check_interface(x, word, filename, lineno);
			get_wordpar(&p, word, x->writefile, sizeof(x->writefile), filename, lineno);
			if (strcmp(word, "debug") == 0)
				debug = 1;
			}

		else if (strcmp(word, "redirect-mode") == 0) {
			char	type[40];

			get_wordpar(&p, word, type, sizeof(type), filename, lineno);
			set_redirmode(x, type, filename, lineno);
			}
		else if (strcmp(word, "acl") == 0) {
			char	acl[200];

			get_parameter(&p, word, acl, sizeof(acl), filename, lineno);
			if (*(p = acl) == '=') {
				p++;
				clearstr(&x->acl);
				}

			p = skip_ws(p);
			append(&x->acl, x->acl.len > 0? " ": "", p);
			}
		else if (strcmp(word, "statdir") == 0)
			get_wordpar(&p, word, statdir, sizeof(statdir), filename, lineno);

		else if (strcmp(word, "name") == 0)
			get_wordpar(&p, word, x->name, sizeof(x->name), filename, lineno);

		else if (strcmp(word, "extended-info") == 0)
			x->extendedinfo = get_yesno(&p, word, filename, lineno);

		else if (strcmp(word, "sessiondir") == 0)
			get_wordpar(&p, word, sessiondir, sizeof(sessiondir), filename, lineno);
		else if (strcmp(word, "session-mode") == 0) {
			char	mode[40];

			get_wordpar(&p, word, mode, sizeof(mode), filename, lineno);
			set_sessionmode(mode, filename, lineno);
			}
		else if (strcmp(word, "exit-handler") == 0  ||  strcmp(word, "error-handler") == 0) {
			get_parameter(&p, word, par, sizeof(par), filename, lineno);
			set_exithandler(par);
			}
		else if (strcmp(word, "exit-codes") == 0  ||  strcmp(word, "error-types") == 0) {
			unsigned int error, mask;
			char	list[200];

			get_parameter(&p, word, list, sizeof(list), filename, lineno);
			p = uncomma(list);
			mask = 0;
			while (*get_word(&p, word, sizeof(word)) != 0) {
				error = geterrcode(word);
				if (error == 0)
					printerror(0 | ERR_INFO, "-INFO", "bad keyword: %s, %s:%d", word, filename, lineno);
				else if (error == 1)
					mask = 0;
				else
					mask |= error;

				set_exitcodes(mask);
				}
			}

		else {
			printerror(ERR_STDERR, "-ERR", "line %d: unkown configuration directive %s", lineno, word);
			exit (1);
			}
		}

	fclose (fp);
	return (0);
}


