%{
/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */
#include <stdio.h>
#include <string.h>

/** This is a global brace counter used for brace matching.  */
/* int g_cbrace; */

/** A hack for dealing with commas.  */
int g_firstArg;

/** A flag indicating an optional argument was scanned. */
int g_oarg;

/**
 * Keeps track of the current class name.  This is needed for correct
 * output of constructors and destructors using C++ syntax.
 */
char *g_className = NULL;


/** 
 * Finds the index of the second hash character in @p text.
 * 
 * @param text what to scan
 * @param len length in characters of @p text
 * 
 * @return the index of the second '#' in @p text.
 * 
 * @return if @p text does not contain two '#' characters @p len is
 * returned.
 */
int getHashIndex(const char *text, int len);

/** 
 * Outputs a comma only if necessary.
 *
 * @post side-effect: g_firstArg is mofidied.
 */
void handleArgumentComma();
%}

%option nounput 

%x CBLOCK CLASS PROC PROC_OPEN PROC_ARGS OPTIONAL_ARG 
%x VAR CLASS_INH INH NAMESPACE UNSUPPORTED_CMD

ID    [[:alpha:]_]+[[:alnum:]_]*
WS    [ \t]

%%


<INITIAL>{
 ##{WS}*$ {
     /* need to strip leading hashes */
     int hashIx = getHashIndex(yytext, yyleng);

     fprintf(yyout, "/** %s", &yytext[hashIx]);
     BEGIN(CBLOCK);
 }

 #.*$ {
     /* Change TCL comments that are not in doxygen format into C
      * comments. */
   int hashIx = getHashIndex(yytext, yyleng);
   fprintf(yyout, "/* %s*/", &yytext[hashIx]);
 }



 "class" {
    ECHO;
    BEGIN(CLASS);
 } 

 "constructor" {
    fprintf(yyout, "public:\n %s", g_className);
    BEGIN(PROC_OPEN);
 }

 "destructor" {
    fprintf(yyout, "public:\n ~%s()", g_className);
 }

 "proc" {
    /* If we are inside of a class then the proc is static. */
    if(g_className)
    {
       fprintf(yyout, "%s", "static string ");
    }
    else
    {
       fprintf(yyout, "%s", "string ");
    }
    BEGIN(PROC);
 }

 "method" {
    fprintf(yyout, "%s", "string ");
    BEGIN(PROC);
 }

 "namespace eval" {
    fprintf(yyout, "%s", "namespace");
    BEGIN(NAMESPACE);
 }

 "public"|"private"|"protected" {
    ECHO;
    fprintf(yyout, "%s", ":\n");
 }

 "variable"|"common"|"global"|"array set"|"set" {
    fprintf(yyout, "%s", "type ");
    BEGIN(VAR);
 }


 \} {

     /* We don't want to gobble up braces.  Doxygen should be OK with
      * the braces as long as they are balanced. */
     ECHO;
    /* Check for the end of the class */
    if(g_className)
    {
       free(g_className);
       g_className = NULL;
       fprintf(yyout, "%s", ";");
    }
 }



 \{ {
     /* We don't want to gobble up braces.  Doxygen should be OK with
      * the braces as long as they are balanced. */
     ECHO;
 }

 . {
     /* eat up anything else */
 }
}




<CBLOCK>{
   ^{WS}*#.*$ {
      /* change first hash mark to a star */
       while(*yytext){
         if(*yytext == '#')
         {
            *yytext = '*';
            break;
         }
         ++yytext;
      }
       /* pad with a space for readability */
       fprintf(yyout, " %s", yytext);
   }
   ^{WS}*[^#] {
      /* Close the comment block and then start scanning in the
       * initial state. */
      fprintf(yyout, "%s\n", " */");
      yyless(0);
      BEGIN(INITIAL);
   }
}


<VAR>{

   {ID}{WS}* {
      ECHO;
      fprintf(yyout, "%s", ";");
      BEGIN(INITIAL);
   }

   {ID}.+$ {
      /* replace the first space or tab with an = character */
     /* Replace ( and ) with [ and ] and , with _ for Tcl arrays
      * because doxygen may get confused and think that array
      * references are function calls.
      */
     int i;
     for(i=0; i<yyleng; ++i)
     {
         switch(yytext[i])
         {
         case '(':
             yytext[i] = '[';
             break;
         case ')':
             yytext[i] = ']';
             break;
         case ',':
             yytext[i] = '_';
             break;
         case ' ':
         case '\t':
            yytext[i] = '=';
            break;
         case '{':
         case '}':
             yytext[i] = '\"';
             break;
         }
     }

      ECHO;
      fprintf(yyout, ";");
      BEGIN(INITIAL);
   }
}

<PROC>{
   (::)?{ID} {
      /* Remove leading : in proc declarations
       */
      while(*yytext == ':')
      {
	  ++yytext;
	  --yyleng;
      }
      ECHO;
      BEGIN(PROC_OPEN);
   }
}

<PROC_OPEN>\{ {
   fprintf(yyout, "(");
   g_firstArg = 1;
   BEGIN(PROC_ARGS);
}


<OPTIONAL_ARG>{
   \} {
      g_oarg = 0;
      BEGIN(PROC_ARGS);
   }
   [^\n\} \t]+ {
      g_oarg++;
      if((g_oarg % 2) == 0){
         fprintf(yyout, "%s", "=");
      }
      ECHO;
   }
}

<PROC_ARGS>{
   \} {
      fprintf(yyout, "%s", ")");
      BEGIN(INITIAL);
   }

   {ID} {
      handleArgumentComma();
      fprintf(yyout, "type %s", yytext);
   }

   \{{WS}* {
      /* This is an optional argument */
      handleArgumentComma();
      fprintf(yyout, "optional ");
      BEGIN(OPTIONAL_ARG);
   }
}



<CLASS>{
   {ID} {
       /* make sure white space occurs between the class keyword and
        * the class name. */
      fprintf(yyout, " ");
      ECHO;
      /* save the class name for ctor dtor decls */
      if(g_className)
      {
         free(g_className);
      }
      g_className = strdup(yytext);
      fprintf(yyout, "\n{");
   }

   \{ {
      BEGIN(CLASS_INH);
   }

   . {
       /* gobble up anything else.  Hopefully this won't eat up too
        * much stuff.   */
   }
}

<CLASS_INH>{
   {ID} {
      if(strcmp("inherit", yytext) == 0){
         fprintf(yyout, ": ");
         BEGIN(INH);
      }else{
         fprintf(yyout, "%s", "\n\n");
         /* Put the token back and start in the initial state. */
         yyless(0);
         BEGIN(INITIAL);
      }
   }
   # {
       /* Put the characters back and start in the initial state */
       yyless(0);
       BEGIN(INITIAL);
   }
}

<INH>{
   {ID} {
      ECHO;
      fprintf(yyout, "%s", "\n{\n");
      BEGIN(INITIAL);
   }
}

<NAMESPACE>{
    /* This rather ugly regex matches things like 'namespace eval foo {' */
   (::)?{ID}(::{ID})*(::)?.+\{ {
      /* Skip leading :: if present */
      while(*yytext == ':')
      {
	  ++yytext;
	  --yyleng;
      }

      /* remove trailing :: if present */
      int i = yyleng;
      while (yytext[i] == ':') {
         yytext[i] = '\0';
         --yyleng;
         --i;
      }
      ECHO;
      BEGIN(INITIAL);
   }

}

%%


void handleArgumentComma()
{
   const char *comma = ", ";
   if(g_firstArg)
   {
      g_firstArg = 0;
      comma = "";
   }
   fprintf(yyout, "%s", comma);
}

int getHashIndex(const char *text, int len)
{
   int i=0;
   while((i <= len) &&
         (yytext[i] != '#'))
   {
      i++;
   }

   while((i <= len) &&
         (yytext[i] == '#'))
   {
      i++;
   }

/*    printf("hash index: %d, text: (%s) length: %d", i, text, len); */
   return i;
}


int yywrap(void)
{
    return 1;
}

int main(int argc, char *argv[])
{
/*    g_cbrace = 0; */
   g_firstArg = 0;
   g_oarg = 0;
   if(argc > 1){
      yyin = fopen(argv[1], "r");
      if(!yyin){
         perror("Couldn't open input");
         exit(EXIT_FAILURE);
      }
   }

   yylex();
   return 0;
}
