// $Id: SVN.cc 1635 2023-03-30 04:16:57Z peter $

/*
	Copyright (C) 2006 Jari Häkkinen
	Copyright (C) 2007, 2008 Jari Häkkinen, Peter Johansson
	Copyright (C) 2010, 2012, 2015, 2023 Peter Johansson

	This file is part of svndigest, http://dev.thep.lu.se/svndigest

	svndigest is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 3 of the License, or
	(at your option) any later version.

	svndigest is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with svndigest. If not, see <http://www.gnu.org/licenses/>.
*/

#include <config.h>

#include "SVN.h"

#include "utility.h"

#include <map>
#include <string>
#include <vector>

#include <cassert>

#include <apr_allocator.h>
#include <apr_hash.h>
#include <subversion-1/svn_client.h>
#include <subversion-1/svn_cmdline.h>
#include <subversion-1/svn_path.h>
#include <subversion-1/svn_pools.h>
#include <subversion-1/svn_wc.h>
#include <subversion-1/svn_subst.h>

#include <string>

#include <iostream>
namespace theplu {
namespace svndigest {


	SVNException::SVNException(const std::string& msg, svn_error_t* error)
		: std::runtime_error(""), error_(error)
	{
		ref_count_ = new int(1);
		if (error_) {
			apr_size_t bufsize=255;
			char buf[255];
			msg_ = svn_err_best_message(error_, buf, bufsize);
		}
		if (msg.size()) {
			msg_ += ". ";
			msg_ += msg;
		}
	}

	SVNException::SVNException(const SVNException& other)
		: std::runtime_error(other), error_(other.error_), msg_(other.msg_),
			ref_count_(other.ref_count_)
	{
		++(*ref_count_);
	}


	SVNException::~SVNException(void) throw()
	{
		--(*ref_count_);
		if (!(*ref_count_)) {
			delete ref_count_;
			ref_count_=NULL;
			svn_error_clear(error_);
			error_=SVN_NO_ERROR;
		}
	}


	const char* SVNException::what(void) const throw()
	{
		return msg_.c_str();
	}


	const svn_error_t* SVNException::error(void) const
	{
		return error_;
	}


	SVN* SVN::instance_=NULL;


	SVN::SVN(const std::string& path)
		: adm_access_(NULL), allocator_(NULL), context_(NULL), pool_(NULL)
	{
		svn_error_t* err=NULL;

		// From http://subversion.apache.org: Applications using the
		// Subversion libraries must call apr_initialize() before calling
		// any Subversion functions.
		apr_initialize();

		// The subversion API documentation says that svn_cmdline_init
		// will iSet up the locale for character conversion, and
		// initialize APR". The documentation goes on to say that this
		// function should only be called once before any other APR and
		// subversion functions. This of course is in conflict with the
		// apr_initialize() recommendation above.
		if (svn_cmdline_init("svndigest",stderr) != EXIT_SUCCESS)
			throw SVNException("SVN: svn_cmdline_init failed");

		/// create top-level pool
		if (apr_allocator_create(&allocator_))
			throw SVNException("SVN(_allocator_create failed");
		apr_allocator_max_free_set(allocator_,SVN_ALLOCATOR_RECOMMENDED_MAX_FREE);
		pool_ = svn_pool_create_ex(NULL, allocator_);
		apr_allocator_owner_set(allocator_, pool_);

		// initialize the repository access library
		if ((err=svn_ra_initialize(pool_)))
			cleanup_failed_init(err, "SVN: svn_ra_initialize failed");

		// Check that users .subversion exist. Should this really be done?
		// If this check is to be done, we might be forced to support a
		// command line option to change the location of the .subversion
		// stuff (compare with the svn binary).
		if ((err=svn_config_ensure(NULL, pool_)))
			cleanup_failed_init(err, "SVN: svn_config_ensure failed");

		// create a client context object
		if ((err=svn_client_create_context(&context_, pool_)))
			cleanup_failed_init(err, "SVN: svn_client_create_context failed");

		if ((err=svn_config_get_config(&(context_->config), NULL, pool_)))
			cleanup_failed_init(err, "SVN: svn_config_get_config failed");

		// set up authentication stuff
		if ((err=svn_cmdline_setup_auth_baton(&(context_->auth_baton), false, NULL,
																					NULL, NULL, false,
												static_cast<svn_config_t*>(apr_hash_get(context_->config,
																										SVN_CONFIG_CATEGORY_CONFIG,
																										APR_HASH_KEY_STRING)),
																					context_->cancel_func,
																					context_->cancel_baton, pool_)))
			cleanup_failed_init(err, "SVN: svn_cmdline_setup_auth_baton failed");

		// Set up svn administration area access. The whole structure is
		// setup, maybe this is unnecessary?
		const char* canonical_path=svn_path_internal_style(path.c_str(), pool_);
		abs_wc_root_path_ = absolute_path(canonical_path);
		if ((err=svn_wc_adm_open3(&adm_access_, NULL, canonical_path,
															false, -1, context_->cancel_func,
															context_->cancel_baton, pool_))){
			if (err->apr_err == SVN_ERR_WC_NOT_DIRECTORY)
				cleanup_failed_init(err, std::string(err->message));
			cleanup_failed_init(err, "SVN: svn_wc_adm_open3 failed");
		}

		// get a session to the repository
		struct url_receiver_baton urb;
		client_info(path, url_receiver, static_cast<void*>(&urb));
		if ((err=svn_client_open_ra_session(&ra_session_,
																				urb.root_url.c_str(),
																				context_, pool_)))
			cleanup_failed_init(err, "SVN: svn_client_open_ra_session failed");

		// extract and store URL relative to root URL
		relative_url_ = relative_path(urb.url, urb.root_url);
	}


	SVN::~SVN(void)
	{
		if (adm_access_)
			svn_error_clear(svn_wc_adm_close(adm_access_));
		svn_pool_destroy(pool_);
		pool_=NULL;
		// other apr resources acquired in svn_cmdline_init are destroyed
		// at program exit, ok since SVN is a singleton
	}


	void SVN::cleanup_failed_init(svn_error_t *err, const std::string& message)
	{
		assert(message.size()); // compatible with r1213
		svn_error_clear(err);
		if (pool_) {
			svn_pool_destroy(pool_);
			pool_=NULL;
		}
		throw SVNException(message);
	}


	void SVN::client_blame(const std::string& path,
												 svn_client_blame_receiver_t receiver,
												 void *baton, svn_revnum_t rev)
	{
		svn_opt_revision_t head;
		head.kind=svn_opt_revision_number;
		head.value.number=rev;
		client_blame_call(path, receiver, baton, head);
	}

	void SVN::client_blame(const std::string& path,
												 svn_client_blame_receiver_t receiver,
												 void *baton)
	{
		svn_opt_revision_t head;
		head.kind = ( svn_path_is_url(path.c_str()) ?
									svn_opt_revision_head : svn_opt_revision_base );
		client_blame_call(path, receiver, baton, head);
	}


	void SVN::client_blame_call(const std::string& path,
															svn_client_blame_receiver_t receiver,
															void *baton, svn_opt_revision_t& head)
	{
		// Setup to use all revisions
		svn_opt_revision_t peg, start;
		peg.kind=svn_opt_revision_unspecified;
		start.kind=svn_opt_revision_number;
		start.value.number=0;
		AprPool subpool(pool_);
		svn_error_t* err =
			svn_client_blame3(path.c_str(), &peg, &start, &head,
												svn_diff_file_options_create(subpool.get()),
												false, receiver, baton, context_,
												subpool.get());
		if (err)
			throw SVNException("SVN::client_blame: svn_client_blame3 failed", err);
	}


	void SVN::client_cat(const std::string& path, svn_revnum_t rev,
											 std::string& result)
	{
		AprPool subpool(pool_);
		AprPool scratchpool(pool_);
		svn_stringbuf_t* str = svn_stringbuf_create("", subpool.get());
		assert(str);
		svn_stream_t* out = svn_stream_from_stringbuf(str, subpool.get());
		assert(out);

		svn_error_t* err = NULL;

		svn_opt_revision_t peg;
		peg.kind = svn_opt_revision_unspecified;
		svn_opt_revision_t revision;
		revision.kind = svn_opt_revision_number;
		revision.value.number = rev;

		err = svn_client_cat3(NULL, out, path.c_str(), &peg, &revision,
													false, context_, subpool.get(), scratchpool.get());

		if (err)
			throw SVNException("client_cat failed", err);

		// copy to result
		result = std::string(str->data, str->len);
	}


	void SVN::client_info(const std::string& path, svn_info_receiver_t receiver,
												void *baton)
	{
		AprPool subpool(pool_);
		if (svn_error_t *err=svn_client_info(path.c_str(), NULL, NULL, receiver,
																				 baton, false, context_,subpool.get()))
			throw SVNException("repository: svn_client_info failed", err);
	}


	void SVN::client_log(const std::string& path,
											 svn_log_message_receiver_t receiver, void *baton)
	{
		// Allocate space in subpool to pool_ for apr_path (here a string).
		AprPool subpool(pool_);
		apr_array_header_t* apr_path=apr_array_make(subpool.get(),1,4);
		// Copy path to apr_path.
		(*((const char **) apr_array_push(apr_path))) =
			apr_pstrdup(subpool.get(),
									svn_path_internal_style(path.c_str(), subpool.get()));

		// Setup to retrieve all commit logs.
		svn_opt_revision_t peg, start, head;
		peg.kind=svn_opt_revision_unspecified;
		start.kind=svn_opt_revision_number;
		start.value.number=0;
		head.kind = ( svn_path_is_url(path.c_str()) ?
									svn_opt_revision_head : svn_opt_revision_base );
		svn_error_t* err=NULL;
		if ((err=svn_client_log3(apr_path, &peg, &start, &head, 0, false, false,
														 receiver, baton, context_, subpool.get())))
			throw SVNException("commit_dates: svn_client_log3 failed", err);
	}


	void SVN::client_proplist(const std::string& path,
														std::map<std::string, std::string>& property)
	{
		svn_opt_revision_t peg, revision;
		peg.kind=svn_opt_revision_unspecified;
		revision.kind = ( svn_path_is_url(path.c_str()) ?
											svn_opt_revision_head : svn_opt_revision_base );
		AprPool subpool(pool_);
		apr_array_header_t * properties;
		svn_error_t *err=svn_client_proplist2(&properties, path.c_str(), &peg,
																					&revision, false, context_,
																					subpool.get());
		if (err)
			throw SVNException("repository: svn_client_proplist2 failed", err);

    for (int j = 0; j < properties->nelts; ++j) {
      svn_client_proplist_item_t *item =
        ((svn_client_proplist_item_t **)properties->elts)[j];
      for (apr_hash_index_t *hi =apr_hash_first(subpool.get(),item->prop_hash);
					 hi; hi = apr_hash_next (hi)) {
        const void *key;
        void *val;
        apr_hash_this (hi, &key, NULL, &val);
				svn_string_t *value;
				err=svn_subst_detranslate_string(&value,
																				 static_cast<const svn_string_t*>(val),
																				 false, subpool.get());
				if (err)
					throw SVNException(path +
									" property: svn_subst_detranslate_string failed on key " +
														 static_cast<const char*>(key), err);
				property[static_cast<const char*>(key)]=value->data;
      }
		}
	}


	SVN* SVN::instance(void)
	{
		if (!instance_)
			throw SVNException("SVN::instance(void): SVN singleton not initialized");
		return instance_;
	}


	SVN* SVN::instance(const std::string& url)
	{
		if (!instance_) {
			instance_=new SVN(url);
		}
		return instance_;
	}


	svn_error_t* SVN::url_receiver(void *baton, const char *,
																 const svn_info_t *info, apr_pool_t*)
	{
		if (!info)
			throw SVNException(std::string("SVN::url_receriver: ") +
												 "Failed to acquire an svn info object");

		url_receiver_baton* rurb=
			static_cast<struct url_receiver_baton*>(baton);
		if (info->repos_root_URL)
			rurb->root_url = info->repos_root_URL;
		if (info->URL)
			rurb->url = info->URL;
		return SVN_NO_ERROR;
	}


	SVN::vc_status SVN::version_controlled(const std::string& path)
	{
		svn_wc_status2_t* status=NULL;
		AprPool subpool(pool_);
		if (svn_error_t *err=
				svn_wc_status2(&status,
											 svn_path_internal_style(path.c_str(), subpool.get()),
											 adm_access_, subpool.get()))
			throw SVNException("version_controlled(): svn_config_get_config failed",
												 err);

		if ((status->text_status==svn_wc_status_none) ||
				(status->text_status==svn_wc_status_unversioned))
			return unversioned;
		else if (status->text_status==svn_wc_status_normal)
			return uptodate;

		return unresolved;
	}


	SVN::AprPool::AprPool(apr_pool_t* pool)
		: pool_(NULL)
	{
		pool_ = svn_pool_create(pool);
		assert(pool_);
	}


	SVN::AprPool::~AprPool(void)
	{
		if (pool_)
			svn_pool_destroy(pool_);
	}


	apr_pool_t* SVN::AprPool::get(void)
	{
		return pool_;
	}

}} // end of namespace svndigest and namespace theplu
