from stgit.argparse import diff_opts_option, opt, patch_range
from stgit.commands.common import (
    DirectoryHasRepository,
    color_diff_flags,
    parse_patches,
)
from stgit.lib.git import RepositoryException
from stgit.pager import pager
from stgit.run import Run

__copyright__ = """
Copyright (C) 2006, Catalin Marinas <catalin.marinas@gmail.com>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License version 2 as
published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, see http://www.gnu.org/licenses/.
"""

help = 'Show the commit corresponding to a patch'
kind = 'patch'
usage = [
    '[-b] [-s] [-O] [--] [<patch1>] [<patch2>] [<patch3>..<patch4>]',
    '(--applied | --unapplied) [-b] [-s] [-O]',
]

description = """
Show the commit log and the diff corresponding to the given patches.
The output is similar to that generated by 'git show'."""

args = [patch_range('applied_patches', 'unapplied_patches', 'hidden_patches')]
options = [
    opt(
        '-b',
        '--branch',
        args=['stg_branches'],
        short='Use BRANCH instead of the default branch',
    ),
    opt(
        '-a',
        '--applied',
        action='store_true',
        short='Show the applied patches',
    ),
    opt(
        '-u',
        '--unapplied',
        action='store_true',
        short='Show the unapplied patches',
    ),
    opt(
        '-s',
        '--stat',
        action='store_true',
        short='Show a diffstat summary of the specified patches',
    ),
] + diff_opts_option()

directory = DirectoryHasRepository()


def func(parser, options, args):
    """Show commit log and diff"""
    if args and (options.applied or options.unapplied):
        parser.error('patches may not be given with --applied or --unapplied')
    elif options.applied and options.unapplied:
        parser.error('cannot use both --applied and --unapplied')

    repository = directory.repository
    stack = repository.get_stack(options.branch)
    patchorder = stack.patchorder

    if options.applied:
        commits = [stack.patches[pn] for pn in patchorder.applied]
    elif options.unapplied:
        commits = [stack.patches[pn] for pn in patchorder.unapplied]
    elif not args:
        commits = [stack.top]
    elif '..' in ' '.join(args):
        # patch ranges
        patch_names = parse_patches(
            args,
            patchorder.all,
            len(patchorder.applied),
        )
        commits = [stack.patches[pn] for pn in patch_names]
    else:
        commits = []
        for name in args:
            if name in stack.patches:
                commits.append(stack.patches[name])
            else:
                try:
                    commits.append(
                        repository.rev_parse(
                            name, object_type='commit', discard_stderr=True
                        )
                    )
                except RepositoryException:
                    raise RepositoryException(
                        '%s: Unknown patch or revision name' % name
                    )

    cmd = ['git', 'show']
    if options.stat:
        cmd.extend(['--stat', '--summary'])
    else:
        cmd.append('--patch')
        cmd.extend(options.diff_flags)
    cmd.extend(color_diff_flags())
    cmd.extend(commit.sha1 for commit in commits)
    pager(Run(*cmd).decoding(None).raw_output())
