// ----------------------------------------------------------------------------
// Scrollable zoomable drawings.
//
#include <math.h>		// use fabs()

#include "list.h"		// Use List
#include "memalloc.h"		// use new()
#include "num.h"		// Use min(), segment_distance()
#include "rectangle.h"
#include "spoint.h"		// Use SRegion
#include "uiepanel.h"		// Use Edge_Panel
#include "uiviewwin.h"
#include "utility.h"
#include "winsystem.h"		// use WinSys, ...

class ZScroll
{
public:
  ZScroll(WinSys &, View_Window *vw, CB_Proc scroll_cb, CB_Proc z_value_cb);
  Widget frame, value, text, sbar;
};

static Rectangle flatten(const SRegion &region, int xaxis, int yaxis);

// ----------------------------------------------------------------------------
//
View_Window::View_Window(WinSys &ws, Widget parent, int xsize, int ysize,
			 const IPoint &axis_order, const SPoint &center,
			 const SPoint &pixel_size, const SPoint &depth,
			 const SRegion &fullview, const SPoint &font_height,
			 const SPoint &scroll_step,
			 const SRegion &repaint_tile, double repaint_priority,
			 bool contour_graying, Widget popup_menu)
  : Drawing(ws, (framew = (parent ?
			   ws.create_frame(parent, "view") :
			   ws.create_dialog("view"))),
	    xsize, ysize,
	    center[axis_order[0]], center[axis_order[1]],
	    pixel_size[axis_order[0]], pixel_size[axis_order[1]],
	    ::flatten(fullview, axis_order[0], axis_order[1]),
	    ::flatten(repaint_tile, axis_order[0], axis_order[1]),
	    repaint_priority, contour_graying)
{
  this->axisorder = axis_order;
  this->drawing_reg = fullview;
  this->scroll_reg = fullview;
  this->lo_scale = fullview.min;
  this->hi_scale = fullview.max;
  this->centr = center;
  this->pixel_sz = pixel_size;
  this->dpth = depth;
  this->repaint_tile = repaint_tile;
  this->font_height = font_height; 
  this->scroll_step = scroll_step;
  this->popup_menu = popup_menu;

  this->hbar = 0;
  this->vbar = 0;

  position_view_widgets(true);

  if (is_top_level_window())
    ws.add_event_callback(destroy_query_event, frame(), destroy_query_cb, this);

  ws.add_event_callback(destroy_event, frame(), destroy_cb, this);
  ws.add_event_callback(mapping_event, frame(), map_cb, this);
  ws.add_event_callback(button_3_press_event, drawing_widget(),
		     popup_menu_cb, this);

  ws.set_keyboard_focus_child(drawing_widget());

  synchronize_other_views();
}

// ----------------------------------------------------------------------------
//
View_Window::~View_Window()
{
  if (is_top_level_window())
    ws.remove_event_callback(destroy_query_event, frame(),
			     destroy_query_cb, this);

  ws.remove_event_callback(destroy_event, frame(), destroy_cb, this);
  ws.remove_event_callback(mapping_event, frame(), map_cb, this);
  ws.remove_event_callback(button_3_press_event, drawing_widget(),
			   popup_menu_cb, this);

  ws.delete_widget(frame());

  for (int zi = 0 ; zi < z_scrollbars.size() ; ++zi)
    delete (ZScroll *) z_scrollbars[zi];

  framew = 0;
}

// ----------------------------------------------------------------------------
//
void View_Window::create_scrollbars()
{
  this->hbar = ws.create_scrollbar(frame(), "xScrollbar", true, scroll_cb, this);
  this->vbar = ws.create_scrollbar(frame(), "yScrollbar", false, scroll_cb, this);

  for (int k = 0 ; k+2 < dimension() ; ++k)
    {
      ZScroll *zs = new ZScroll(ws, this, scroll_cb, z_value_cb);
      this->z_scrollbars.append(zs);
    }
}

// ----------------------------------------------------------------------------
//
ZScroll::ZScroll(WinSys &ws, View_Window *vw,
		 CB_Proc scroll_cb, CB_Proc z_value_cb)
{
  Widget f = ws.create_frame(vw->frame(), "zScrollbar");
  this->frame = f;
  this->value = ws.edit_field(f, "value");
  this->text = ws.edit_field_text(value);
  ws.add_event_callback(enter_pressed_event, text, z_value_cb, vw);
  this->sbar = ws.create_scrollbar(f, "scrollbar", true, scroll_cb, vw);
  ws.row_attachments(sbar, value, sbar, END_OF_WIDGETS);
  ws.detach_top_and_bottom(sbar);
}

// ----------------------------------------------------------------------------
//
void View_Window::z_value_cb(Widget text, CB_Data vwindow, CB_Data)
{
  View_Window *vw = (View_Window *) vwindow;
  vw->entered_z_value(text);
}

// ----------------------------------------------------------------------------
//
void View_Window::entered_z_value(Widget text)
{
  double z = ws.numeric_text_field(text);
  int axis = find_z_text_axis(text);
  SPoint c = center();
  c[axis] = from_scale_value(z, axis);
  set_view(c, pixel_size());
}

// ----------------------------------------------------------------------------
//
int View_Window::find_z_text_axis(Widget text)
{
  for (int k = 0 ; k < z_scrollbars.size() ; ++k)
    if (text == ((ZScroll *) z_scrollbars[k])->text)
      return axis_order()[k+2];

  fatal_error("View_Window::find_z_text_axis(): Unknown text widget.\n");
  return -1;
}

// ----------------------------------------------------------------------------
//
void View_Window::position_view_widgets(bool resize_window)
{
  if (resize_window)
    set_window_size();

  ws.plot_form_layout(frame(), drawing_widget(), horz_widgets, vert_widgets);
}

// ----------------------------------------------------------------------------
//
void View_Window::set_window_size()
{
  int wpanels = 0, hpanels = 0;

  for (int wi = 0 ; wi < horz_widgets.size() ; ++wi)
    hpanels += ws.requested_height((Widget) horz_widgets[wi]);
  for (int wi = 0 ; wi < vert_widgets.size() ; ++wi)
    wpanels += ws.requested_width((Widget) vert_widgets[wi]);

  int wdrawing, hdrawing;
  screen_size(&wdrawing, &hdrawing);

  if (is_top_level_window())
    ws.resize_dialog(frame(), wdrawing + wpanels, hdrawing + hpanels);
}

// ----------------------------------------------------------------------------
// Allow derived class to veto a destroy request from window manager.
//
bool View_Window::ok_to_destroy() { return true; }	// virtual
void View_Window::destroy_query_cb(Widget, CB_Data viewwin, CB_Data)
{
  destroy_view_window((View_Window *) viewwin);
}

// ----------------------------------------------------------------------------
//
bool destroy_view_window(View_Window *vw)
{
  bool destroy = vw->ok_to_destroy();
  if (destroy)
    delete vw;
  return destroy;
}

// ----------------------------------------------------------------------------
//
void View_Window::destroy_cb(Widget, CB_Data viewwin, CB_Data)
{
  delete (View_Window *) viewwin;
}

// ----------------------------------------------------------------------------
// So derived class can track visibility status
//
void View_Window::mapped_or_unmapped() {}  // virtual
void View_Window::map_cb(Widget, CB_Data viewwin, CB_Data)
{
  ((View_Window *) viewwin)->mapped_or_unmapped();
}

// ----------------------------------------------------------------------------
//
void View_Window::popup_menu_cb(Widget, CB_Data viewwin, CB_Data event)
{
  View_Window *vw = (View_Window *) viewwin;

  vw->got_focus();
  if (vw->popup_menu)
    vw->ws.popup_menu(vw->popup_menu, event);
}

// ----------------------------------------------------------------------------
//
int View_Window::dimension() const
  { return drawing_reg.dimension(); }
IPoint View_Window::axis_order() const
  { return axisorder; }
SPoint View_Window::center() const
  { return centr; }
double View_Window::center(int axis) const
  { return centr[axis]; }
double View_Window::center(Axis axis) const
  { return centr[this->axis(axis)]; }
SPoint View_Window::pixel_size() const
  { return pixel_sz; }
double View_Window::pixel_size(int axis) const
  { return pixel_sz[axis]; }
double View_Window::pixel_size(Axis axis) const
  { return pixel_sz[this->axis(axis)]; }
double View_Window::aspect() const
  { return pixel_size(Y) / pixel_size(X); }

// ----------------------------------------------------------------------------
//
Widget View_Window::frame() const
  { return framew; }
bool View_Window::is_top_level_window() const
  { return ws.is_top_level(frame()); }

// ----------------------------------------------------------------------------
//
void View_Window::show_view(bool show)
{
  if (is_top_level_window())
    if (show)
      {
	ws.show_dialog(frame());
	ws.raise_widget(frame());
      }
    else
      ws.unshow_dialog(frame());
}

// ----------------------------------------------------------------------------
//
bool View_Window::shown() const
{
  return ws.is_viewable(frame());
}

// ----------------------------------------------------------------------------
//
void View_Window::screen_location(int *x, int *y)
{
  if (is_top_level_window())
    ws.dialog_position(frame(), x, y);
  else
    *x = *y = 0;
}

// ----------------------------------------------------------------------------
//
void View_Window::set_screen_location(int x, int y)
{
  if (is_top_level_window())
    ws.set_dialog_position(frame(), x, y);
}

// ----------------------------------------------------------------------------
//
void View_Window::show_edge_panels(const List &inside, const List &outside,
				   bool show_scrollbars, bool resize_window)
{
  bool changed = false;

  List epanels = edge_panels;
  for (int k = 0 ; k < epanels.size() ; ++k)
    if (!inside.contains(epanels[k]) && !outside.contains(epanels[k]))
      changed |= position_edge_panel((Edge_Panel) epanels[k], false, false);

  for (int k = 0 ; k < inside.size() ; ++k)
    changed |= position_edge_panel((Edge_Panel) inside[k], true, true);

  for (int k = 0 ; k < outside.size() ; ++k)
    changed |= position_edge_panel((Edge_Panel) outside[k], true, false);

  changed |= position_scrollbars(show_scrollbars);

  if (changed)
    position_view_widgets(resize_window);
}

// ----------------------------------------------------------------------------
//
bool View_Window::position_edge_panel(Edge_Panel ep, bool show, bool near)
{
  if (show)
    {
      if (!edge_panel_shown(ep))
	{
	  edge_panels.append(ep);
	  List &widgets = (ep->is_horizontal() ? horz_widgets : vert_widgets);
	  if (near)	widgets.append(ep->widget());
	  else		widgets.insert(ep->widget(), 0);
	  Axis a = (ep->is_horizontal() ? X : Y);
	  Rectangle vr = view_rectangle();
	  ep->set_range(vr.min(a), vr.max(a));
	  return true;
	}
    }
  else
    {
      if (edge_panel_shown(ep))
	{
	  edge_panels.erase(ep);
	  List &widgets = (ep->is_horizontal() ? horz_widgets : vert_widgets);
	  widgets.erase(ep->widget());
	  return true;
	}
    }
  return false;
}

// ----------------------------------------------------------------------------
//
bool View_Window::scrollbars_shown()
{
  return vbar != 0 && vert_widgets.contains(vbar);
}

// ----------------------------------------------------------------------------
//
bool View_Window::position_scrollbars(bool show)
{
  bool shown = scrollbars_shown();
  if (show && !shown)
    {
      if (vbar == 0)
	create_scrollbars();
      for (int zi = 0 ; zi < z_scrollbars.size() ; ++zi)
	horz_widgets.insert(((ZScroll *)z_scrollbars[zi])->frame, zi);
      horz_widgets.insert(hbar, z_scrollbars.size());
      vert_widgets.insert(vbar, 0);
      return true;
    }
  else if (!show && shown)
    {
      for (int zi = 0 ; zi < z_scrollbars.size() ; ++zi)
	horz_widgets.erase(((ZScroll *)z_scrollbars[zi])->frame);
      horz_widgets.erase(hbar);
      vert_widgets.erase(vbar);
      return true;
    }
  return false;
}

// ----------------------------------------------------------------------------
//
bool View_Window::edge_panel_shown(Edge_Panel ep)
{
  return edge_panels.contains(ep);
}

// ----------------------------------------------------------------------------
//
void View_Window::set_title(const Stringy &title, const Stringy &icon_title)
{
  if (is_top_level_window())
    {
      ws.set_dialog_title(frame(), title);
      ws.set_icon_name(frame(), icon_title);
    }
}

// ----------------------------------------------------------------------------
//
int View_Window::axis(Axis name) const
{
  int axis;

  if (name == X)
    axis = axisorder[0];
  else if (name == Y)
    axis = axisorder[1];
  else
    {
      axis = -1;	// Suppress compiler warning about uninitialized var.
      fatal_error("View_Window::axis(): Bad argument.\n");
    }

  return axis;
}

// ----------------------------------------------------------------------------
//
SRegion View_Window::view_region()
{
  return thicken(view_rectangle());
}

// ----------------------------------------------------------------------------
//
Rectangle View_Window::flatten(const SRegion &region)
{
  return ::flatten(region, axis(X), axis(Y));
}

// ----------------------------------------------------------------------------
//
static Rectangle flatten(const SRegion &region, int xaxis, int yaxis)
{
  return Rectangle(region.min[xaxis], region.min[yaxis],
		   region.max[xaxis], region.max[yaxis]);
}

// ----------------------------------------------------------------------------
//
SRegion View_Window::thicken(const Rectangle &r)
{
  int dim = dimension();
  SRegion region(dim);

  region.min[axis(X)] = r.min(X);
  region.max[axis(X)] = r.max(X);
  region.min[axis(Y)] = r.min(Y);
  region.max[axis(Y)] = r.max(Y);

  for (int a = 0 ; a < dim ; ++a)
    if (a != axis(X) && a != axis(Y))
      {
	region.min[a] = centr[a] - dpth[a]/2;
	region.max[a] = centr[a] + dpth[a]/2;
      }

  return region;
}

// ----------------------------------------------------------------------------
//
SRegion View_Window::flat_region(Rectangle r)
{
  SPoint c = view_region().center();
  SPoint min = c;
  SPoint max = c;

  min[axis(X)] = r.min(X);
  min[axis(Y)] = r.min(Y);
  max[axis(X)] = r.max(X);
  max[axis(Y)] = r.max(Y);

  return SRegion(min, max);
}

// ----------------------------------------------------------------------------
//
void View_Window::scroll_cb(Widget w, CB_Data vwindow, CB_Data)
{
  View_Window *vw = (View_Window *) vwindow;
  if (!vw->ws.more_motion_events())
    vw->scroll(w);
}

// ----------------------------------------------------------------------------
//
void View_Window::scroll(Widget sbar)
{
  int axis;

  if (sbar == hbar)
    axis = this->axis(X);
  else if (sbar == vbar)
    axis = this->axis(Y);
  else
    axis = find_z_scrollbar_axis(sbar);

  double offset = (ws.scrollbar_position(sbar) * scroll_reg.size(axis)
		   + view_region().size(axis) / 2);
  double pos = (sbar == vbar ?
		scroll_reg.min[axis] + offset :
		scroll_reg.max[axis] - offset);

  SPoint c = this->centr;
  c[axis] = pos;

  update_view(this->axisorder, c, this->pixel_sz, this->dpth, false, true);
}

// ----------------------------------------------------------------------------
//
int View_Window::find_z_scrollbar_axis(Widget sbar)
{
  for (int k = 0 ; k < z_scrollbars.size() ; ++k)
    if (sbar == ((ZScroll *) z_scrollbars[k])->sbar)
      return axis_order()[k+2];

  fatal_error("View_Window::find_z_scrollbar_axis(): Unknown widget.\n");
  return -1;
}

// ----------------------------------------------------------------------------
//
void View_Window::set_axis_order(const IPoint &axis_order)
{
  update_view(axis_order, this->centr, this->pixel_sz,
	      this->dpth, true, true);
}

// ----------------------------------------------------------------------------
//
void View_Window::set_font_height(const SPoint &font_height)
{
  this->font_height = font_height;
  Drawing::set_font_height(font_height[axis(Y)]);
}

// ----------------------------------------------------------------------------
//
#define MIN_ASPECT_CHANGE_FACTOR 1e-5

void View_Window::set_aspect(double aspect)
{
  if (aspect > 0 &&
      fabs(aspect - this->aspect()) >
      MIN_ASPECT_CHANGE_FACTOR * this->aspect())
    {
      SPoint ps = pixel_size();
      ps[axis(X)] *= this->aspect() / aspect;
      set_view(center(), ps);
    }
}

// ----------------------------------------------------------------------------
//
void View_Window::set_view(const SPoint &center, const SPoint &pixel_size,
			   bool sync_other_views)
{
  update_view(this->axisorder, center, pixel_size,
	      this->dpth, true, sync_other_views);
}

// ----------------------------------------------------------------------------
//
void View_Window::set_visible_depth(const SPoint &depth)
{
  update_view(this->axisorder, this->centr, this->pixel_sz, depth, true, true);
}

// ----------------------------------------------------------------------------
//
void View_Window::update_view(const IPoint &axis_order,
			      const SPoint &center,
			      const SPoint &pixel_size,
			      const SPoint &depth,
			      bool update_scrollbars,
			      bool sync_other_views)
{
  bool new_view_plane = false;
  for (int a = 0 ; a < dimension() ; ++a)
    if (a != axis(X) && a != axis(Y))
      if (center[a] != this->centr[a] ||
	  depth[a] != this->dpth[a])
	new_view_plane = true;

  bool axis_order_changed = (axis_order != this->axisorder);
  bool pixel_size_changed = (pixel_size != this->pixel_sz);

  this->axisorder = axis_order;
  this->centr = center;
  this->pixel_sz = pixel_size;
  this->dpth = depth;

  if (axis_order_changed)
    {
      Drawing::set_drawing(flatten(drawing_reg));
      Drawing::set_redraw_tile(flatten(repaint_tile));
      Drawing::set_font_height(font_height[axis(Y)]);
    }

  if (axis_order_changed || pixel_size_changed)
    Drawing::set_pixel_size(this->pixel_size(X), this->pixel_size(Y));

  Drawing::set_center(this->center(X), this->center(Y));

  if (new_view_plane)
    {
      update_z_labels();
      erase();
    }

  update_panels();

  if (update_scrollbars)
    this->update_scrollbars();

  if (sync_other_views)
    synchronize_other_views();
}

// ----------------------------------------------------------------------------
//
void View_Window::synchronize_other_views() {}

// ----------------------------------------------------------------------------
//
void View_Window::drawing_resized()
{
  update_scrollbars();
  update_panels();
  synchronize_other_views();
}

// ----------------------------------------------------------------------------
//
void View_Window::update_panels()
{
  Rectangle vr = view_rectangle();
  for (int ei = 0 ; ei < edge_panels.size() ; ++ei)
    {
      Edge_Panel ep = (Edge_Panel) edge_panels[ei];
      Axis a = (ep->is_horizontal() ? X : Y);
      ep->set_range(vr.min(a), vr.max(a));
      ep->view_region_changed();
    }
}

// ----------------------------------------------------------------------------
//
void View_Window::update_scrollbars()
{
  if (vbar == 0)
    return;

  scroll_reg = drawing_reg;
  SRegion viewreg = view_region();

  for (int a = 0 ; a < dimension() ; ++a)
    {
      double edge_pad = viewreg.size(a) / 2;
      scroll_reg.min[a] -= edge_pad;
      scroll_reg.max[a] += edge_pad;
    }

  scroll_reg.encompass(viewreg);

  Rectangle vr = view_rectangle();
  ws.set_scrollbar(hbar,
		   vr.max(X), vr.min(X),
		   scroll_reg.max[axis(X)], scroll_reg.min[axis(X)],
		   scroll_step[axis(X)]);
  ws.set_scrollbar(vbar,
		   vr.min(Y), vr.max(Y),
		   scroll_reg.min[axis(Y)], scroll_reg.max[axis(Y)],
		   scroll_step[axis(Y)]);
  for (int k = 0 ; k < z_scrollbars.size() ; ++k)
    {
      ZScroll *zs = (ZScroll *) z_scrollbars[k];
      int zaxis = axis_order()[k+2];
      ws.set_scrollbar(zs->sbar,
		       viewreg.max[zaxis], viewreg.min[zaxis],
		       scroll_reg.max[zaxis], scroll_reg.min[zaxis],
		       scroll_step[zaxis]);
    }
  update_z_labels();
}

// ----------------------------------------------------------------------------
//
void View_Window::set_axes(Stringy *labels,
			   const SPoint &lo_scale, const SPoint &hi_scale)
{
  this->lo_scale = lo_scale;
  this->hi_scale = hi_scale;
  for (int a = 0 ; a < dimension() ; ++a)
    axis_labels[a] = labels[a];

  update_z_labels();
}

// ----------------------------------------------------------------------------
//
void View_Window::set_repaint_region(const SRegion &fullview)
{
  this->drawing_reg = fullview;
  Drawing::set_drawing(flatten(drawing_reg));
  update_scrollbars();
}

// ----------------------------------------------------------------------------
//
void View_Window::set_repaint_tile(const SRegion &tile)
{
  this->repaint_tile = tile;
  Drawing::set_redraw_tile(flatten(repaint_tile));
}

// ----------------------------------------------------------------------------
//
void View_Window::update_z_labels()
{
  SPoint c = view_region().center();
  for (int k = 0 ; k < z_scrollbars.size() ; ++k)
    {
      ZScroll *zs = (ZScroll *) z_scrollbars[k];
      int zaxis = axis_order()[k+2];
      ws.set_edit_title(zs->value, axis_labels[zaxis]);
      double z = to_scale_value(c[zaxis], zaxis);
      ws.set_numeric_edit_value(zs->value, "%.4g", z);
    }
}

// ----------------------------------------------------------------------------
//
double View_Window::to_scale_value(double d, int axis)
{
  double size = drawing_reg.size(axis);
  double f = (size == 0 ? 0 : (d - drawing_reg.min[axis]) / size);

  return lo_scale[axis] + f * (hi_scale[axis] - lo_scale[axis]);
}

// ----------------------------------------------------------------------------
//
double View_Window::from_scale_value(double d, int axis)
{
  double size = hi_scale[axis] - lo_scale[axis];
  double f = (size == 0 ? 0 : (d - lo_scale[axis]) / size);

  return drawing_reg.min[axis] + f * drawing_reg.size(axis);
}

// ----------------------------------------------------------------------------
//
void View_Window::move(double x, double y)
{
  for (int epi = 0 ; epi < edge_panels.size() ; ++epi)
    ((Edge_Panel) edge_panels[epi])->pointer_move(x, y);
}

// ----------------------------------------------------------------------------
//
void View_Window::enter_window(double x, double y)
{
  for (int epi = 0 ; epi < edge_panels.size() ; ++epi)
    ((Edge_Panel) edge_panels[epi])->pointer_in(x, y);
}

// ----------------------------------------------------------------------------
//
void View_Window::exit_window(double x, double y)
{
  for (int epi = 0 ; epi < edge_panels.size() ; ++epi)
    ((Edge_Panel) edge_panels[epi])->pointer_out(x, y);
}

// ----------------------------------------------------------------------------
//
SPoint View_Window::visible_depth() const
  { return dpth; }
double View_Window::visible_depth(int axis) const
  { return dpth[axis]; }
