// ----------------------------------------------------------------------------
// Views display contoured spectra with peak markers and assignment labels,
// ppm scales, resonances panels, 1-D slices, scrollbars, ...,
// and allow zooming and peak selection with the mouse.
//
// The View class is derived from the View_Window class which manages
// the generic user interface (window layout and events) and knows nothing
// about spectra or peaks.
//

#ifndef VIEW_HEADER_INCLUDED
#define VIEW_HEADER_INCLUDED

#include "color.h"		// Use Color
#include "contour.h"		// Use Contour_Levels
#include "list.h"		// Use List
#include "ornament.h"		// Use Ornament_Type, ODraw
#include "rectangle.h"		// Use Rectangle
#include "spectrumdata.h"	// Use Units
#include "stringc.h"		// Use Stringy
#include "uiviewwin.h"		// Use View_Window
#include "winsystem.h"		// Use Widget, CB_Data, XEvent

class ContourScale;
class CrossPeak;
class Peak_Panel;
class RPanel;
class Scale;
class Session;
class Slice;
class Spectrum;

// ----------------------------------------------------------------------------
//
class Contour_Parameters
{
public:
  Contour_Parameters();
  Contour_Parameters(bool positive);
  Contour_Parameters(const Contour_Parameters &);
  ~Contour_Parameters();
  Contour_Parameters &operator=(const Contour_Parameters &);

  Contour_Levels levels;

  const Color &level_color(double level) const;
  Stringy color_scheme() const;
  void set_color_scheme(const Stringy &);

private:
  Stringy clr_scheme;
  List colors;
};

// ----------------------------------------------------------------------------
//
class View_Settings
{
public:
  View_Settings(Spectrum *sp);

  Stringy view_name;
  IPoint axis_order;
  int x, y, width, height;		// Screen window geometry
  SPoint center, pixel_size, visible_depth;
  Contour_Parameters pos_contours, neg_contours;
  bool subtract_fit_peaks;
  Units scale_units;			// Units to show on scales
  bool show_view;			// Is window on screen?
  bool show_ornaments;
  bool show_peaks;
  bool show_peakgroups;
  bool show_labels;
  bool show_lines;
  bool show_grids;
  bool show_ornament(Ornament_Type t);
  void show_ornament(Ornament_Type t, bool show);
  bool show_scales;
  bool show_scrollbars;
  bool show_resonance_panels;
  bool filter_resonances;  // Use assignment guesses to filter resonance panels
  bool show_slices;
  bool slice_auto_scale;
  bool slice_subtract_peaks;
  bool show_peak_info;
  bool show_contour_scale;
  bool show_crosshair;
  bool show_crosshair_from_other_views;
  bool show_transpose_crosshair;
  bool show_nucleus_names;
};

// ----------------------------------------------------------------------------
//
class View : public View_Window
{
public:
  View(Session &s, Widget parent, Spectrum *sp, const View_Settings &settings);
  virtual ~View();

  View *	duplicate(Rectangle r);
  void		zoom(Rectangle r);
  void		zoom(const SRegion &r, bool remember = true);
  void		zoom_previous();
  void		zoom_next();
  void		set_center(const SPoint &p);
  void		swap_axes();
  void		roll_axes();
  void		set_axis_order(const IPoint &axis_order);
  void		redraw();
  virtual void	redraw_rectangle(Rectangle r);
  void		set_name(const Stringy &name);
  void		set_axis_units(Units u);	// Set plot units
  void		set_contour_parameters(const Contour_Parameters &cp);
  void		set_footer(const char *text);
  void		scale_slices(double factor);
  void		shift_slices(double shift);
  virtual void	synchronize_other_views();
  void		update_label_size();
  void		unoverlap_labels();
  void		right_offset_labels();
  void		set_crosshair_position(const SPoint &);

  void		erase_ornaments(Ornament_Type type);
  void		erase_resonances();

  const View_Settings &	settings();
  void		configure(const View_Settings &settings);

  void		show_view(bool show);
  bool		subtract_fit_peaks() const;

  Session &	session() const;
  Stringy	name() const;
  Stringy 	fullname() const;	// Includes molecule / cond
  Spectrum *	spectrum() const;

  Units		axis_units();		// Plot units
  Axis		axis_name(int a) const;
  Stringy	axis_label(int axis);
  double	ornament_size(Ornament_Type type, int axis) const;
  CrossPeak *	over_visible_crosspeak(double x, double y);
  List		region_ornament_list(bool selected, const SRegion &region);
  Slice *	slice(Axis a);
  RPanel *	resonance_panel(Axis a);
  const Contour_Parameters & positive_contours();
  const Contour_Parameters & negative_contours();
  const Contour_Parameters & contour_parameters(double level);
  void		lowest_contour_levels(double *pos, double *neg);
  virtual bool ok_to_destroy();

  double	scale(double x, Axis a, Units from, Units to) const;
  double	map(double x, Axis a, Units from, Units to) const;
  double	map(double x, int axis, View *to, int to_axis);

  IRegion	two_dimensional_region(int x, int y, int w, int h);
  Rectangle	full_view();

  void		data_point(const SPoint &p, double *x, double *y);
  SPoint	point(double x, double y);
  bool		pointer_position(SPoint *p);

  ODraw &ornament_drawing_routines();

// ----------------------------------------------------------------------------
//
private:

  Session &ses;
  Spectrum *spect;
  View_Settings config;

  bool crosshair_drawn[2];
  bool transpose_crosshair_drawn[2];
  double crosshair_xy[2];	// Crosshair position.

  List local_ornaments;
  SRegion local_region;
  ODraw *odroutines;			// ornament drawing routines
  Rectangle ornament_update_rectangle;

  Scale *xscale, *yscale;
  RPanel *xrpanel, *yrpanel;
  Slice *xslice, *yslice;
  Peak_Panel *pkpanel;
  ContourScale *cscale;

  List zoom_prev, zoom_nxt;	// Regions for zoom previous, next

  int		mPts;		// points saved (for line mode)
  SPoint	line_start;	// Line mode start point

  struct {
    bool	down, extend, drag;
    Ornament	*ornament;
    double	downX, downY;		// data units
    double	lastX, lastY;		// data units
  } pick;  // Record of mouse actions.

  void set_redraw_priority();

  virtual void mapped_or_unmapped();
  virtual void buttondown(double x, double y, bool shift);
  virtual void buttonup(double x, double y);
  virtual void drag(double x, double y);
  virtual void enter_window(double x, double y);
  virtual void exit_window(double x, double y);
  virtual void move(double x, double y);
  virtual void key_pressed(char c);
  virtual void function_key_pressed(int f, bool shifted);
  virtual void got_focus();

  void erase(Ornament *op);

  void update_pointer_position(double x, double y);
  void update_scales();
  Ornament *pick_ornament(double x, double y);
  Ornament *over_visible_ornament(double x, double y);
  List visible_ornaments(Rectangle rect);
  List ornaments(const SRegion &region);
  bool ornaments_shown(Ornament_Type type);
  void show_ornaments(bool show);
  void show_ornaments(Ornament_Type type, bool show);

  Scale *axis_scale(Axis a);
  Peak_Panel *peak_panel();
  ContourScale *contour_scale();
  void configure_edge_panels(const View_Settings &s, bool resize_window);

  static void ppm_scale_shifted_cb(void *v, void *s);
  static void ornament_erase_cb(void *v, void *o);
  static void will_delete_ornament_cb(void *v, void *o);
  static void labels_resized_cb(void *v, void *s);
  static void resonance_update_cb(void *v, void *r);
  void buttonup_action();
  void draw_all_crosshairs(double x, double y);
  void erase_all_crosshairs();
  void draw_crosshair(Axis a, double p);
  void erase_crosshair(Axis a);
  bool toggle_line(double p, Axis a);
  bool is_homonuclear();
  void blank_crosshairs(bool *blanked_x, bool *blanked_y);
  void unblank_crosshairs(bool blanked_x, bool blanked_y);
  bool contains_crosshair(const Rectangle &r);

  void ornament_update(const Rectangle &r);
  void display_ornaments(Rectangle rect);

  void save_all_params(FILE *fp);
  void save_current_params(FILE *fp);

  void draw_move_box();
  void draw_drag_box();
  void move_ornament();

  void zoom_stack_push();
};

View *selected_view(Session &);
Spectrum *selected_spectrum(Session &);
void update_resonance_panels(const List &views);
void shift_views(const List &vlist, const SPoint &delta);

void xy_location(CrossPeak *xp, View *view, double *x, double *y);
void apply_mode_to_region(View *view, bool extend, const SRegion &region);

bool orthogonal_views(View *v1);

#endif
