// ----------------------------------------------------------------------------
//

#include "atom.h"		// Use Atom
#include "condition.h"		// Use Condition
#include "crosspeak.h"		// Use CrossPeak
#include "group.h"		// Use Group
#include "list.h"		// Use List
#include "memalloc.h"		// use new()
#include "molecule.h"		// Use Molecule
#include "project.h"		// use Project
#include "reporter.h"		// use Reporter
#include "resonance.h"		// Use Resonance
#include "session.h"		// use Session
#include "spectrum.h"		// Use Spectrum
#include "uicomponents.h"	// Use Molecule_Field, Condition_Field
#include "uidialogs.h"		// use help_cb()
#include "uidialog.h"		// use Dialog, Dialog_Table
#include "uimain.h"		// Use query()
#include "winsystem.h"		// use WinSys

// ----------------------------------------------------------------------------
//
enum Rename_Mode {RENAME_RESONANCES, MERGE_RESONANCES, SWAP_RESONANCES};

// ----------------------------------------------------------------------------
//
class rename_dialog : public Dialog
{
public:
  rename_dialog(Session &);
  ~rename_dialog();

  static rename_dialog *the(Session &);

  void show();
  void update();

private:
  Widget table;
  Molecule_Field *molecule;
  Condition_Field *condition;

  static void rename_cb(Widget, CB_Data, CB_Data);
  static void merge_cb(Widget, CB_Data, CB_Data);
  static void swap_cb(Widget, CB_Data, CB_Data);

  bool rename(Rename_Mode);
  bool is_legal_rename(Rename_Mode mode,
		       const Stringy &from_group, const Stringy &from_atom,
		       const Stringy &to_group, const Stringy &to_atom);
  bool rename(Rename_Mode mode, const List &rlist,
	      const Stringy &to_group, const Stringy &to_atom);
};

// ----------------------------------------------------------------------------
//
static List matching_resonances(Project &proj, const Stringy &molecule_name,
				const Stringy &condition_name,
				const Stringy &group_name,
				const Stringy &atom_name);
static bool confirm_merge(Session &, const List &rlist,
			  const Stringy &group, const Stringy &atom);
static void rename_resonance(Resonance *r, const Stringy &group,
			     const Stringy &atom, Reporter &);
static bool delete_resonance(Resonance *r, Reporter &);
static void rename_unassigned_resonance(Resonance *r, const Stringy &group,
					const Stringy &atom);
static void rename_assigned_resonance(Resonance *r, const Stringy &group,
				      const Stringy &atom);

// ----------------------------------------------------------------------------
//
void show_rename_dialog(Session &s)
  { rename_dialog::the(s)->show(); }

// ----------------------------------------------------------------------------
//
rename_dialog::rename_dialog(Session &s) : Dialog(s, "renameDialog")
{

  molecule = new Molecule_Field(s, dialog, "molecules");
  condition = new Condition_Field(s, dialog, "conditions");

  Table_Entry label = TABLE_LABEL;
  Table_Entry value = TABLE_TEXT_FIELD;
  table = ws.widget_table(dialog, "gaTable", 3, 3,
		       label, label, label,
		       label, value, value,
		       label, value, value);

  Widget separator = ws.create_separator(dialog, "separator");

  Widget controls = ws.button_row(dialog, "controls",
			       "rename", rename_cb, this,
			       "merge", merge_cb, this,
			       "swap", swap_cb, this,
			       "close", close_cb, this,
			       "help", help_cb, &s,
			       NULL);

  ws.column_attachments(separator,
		     molecule->edit_field(), condition->edit_field(),
		     table, separator, controls,
		     END_OF_WIDGETS);

}

// ----------------------------------------------------------------------------
//
rename_dialog::~rename_dialog()
{
  session.dialog_table().delete_dialog("rename_dialog", this);

  delete molecule;
  delete condition;

}

// ----------------------------------------------------------------------------
// The default rename_dialog instance.
//
rename_dialog *rename_dialog::the(Session &s)
{
  Stringy name = "rename_dialog";
  Dialog_Table &dt = s.dialog_table();
  if (dt.get_dialog(name) == NULL)
    dt.set_dialog(name, new rename_dialog(s));
  return (rename_dialog *) dt.get_dialog(name);
}

// ----------------------------------------------------------------------------
//
void rename_dialog::rename_cb(Widget, CB_Data client_data, CB_Data)
{
  rename_dialog *dd = (rename_dialog *) client_data;
  dd->rename(RENAME_RESONANCES);
}

// ----------------------------------------------------------------------------
//
void rename_dialog::merge_cb(Widget, CB_Data client_data, CB_Data)
{
  rename_dialog *dd = (rename_dialog *) client_data;
  dd->rename(MERGE_RESONANCES);
}

// ----------------------------------------------------------------------------
//
void rename_dialog::swap_cb(Widget, CB_Data client_data, CB_Data)
{
  rename_dialog *dd = (rename_dialog *) client_data;
  dd->rename(SWAP_RESONANCES);
}

// ----------------------------------------------------------------------------
//
void rename_dialog::show()
{
  update();
  ws.show_dialog(dialog);
  ws.raise_widget(dialog);
}

// ----------------------------------------------------------------------------
//
void rename_dialog::update()
{
}

// ----------------------------------------------------------------------------
//
bool rename_dialog::rename(Rename_Mode mode)
{
  Stringy molecule_name = ws.edit_value(molecule->edit_field());
  Stringy condition_name = ws.edit_value(condition->edit_field());
  Stringy from_group = ws.text_field_string(ws.table_element(table, 1, 1));
  Stringy from_atom = ws.text_field_string(ws.table_element(table, 1, 2));
  Stringy to_group = ws.text_field_string(ws.table_element(table, 2, 1));
  Stringy to_atom = ws.text_field_string(ws.table_element(table, 2, 2));

  if (! is_legal_rename(mode, from_group, from_atom, to_group, to_atom))
    return false;

  Project &proj = session.project();
  List rlist = matching_resonances(proj, molecule_name, condition_name,
				   from_group, from_atom);
  switch (mode)
    {
    case RENAME_RESONANCES:
    case MERGE_RESONANCES:
      return rename(mode, rlist, to_group, to_atom);
    case SWAP_RESONANCES:
      {
	Stringy temp_group = (from_group.is_empty() ?
			      from_group : from_group + "**swap**");
	Stringy temp_atom = (from_atom.is_empty() ?
			     from_atom : from_atom + "**swap**");
	rename(mode, rlist, temp_group, temp_atom);
	rlist = matching_resonances(proj, molecule_name, condition_name,
				    to_group, to_atom);
	rename(mode, rlist, from_group, from_atom);
	rlist = matching_resonances(proj, molecule_name, condition_name,
				    temp_group, temp_atom);
	rename(mode, rlist, to_group, to_atom);
      }
      break;
    }

  return true;
}

// ----------------------------------------------------------------------------
//
bool rename_dialog::is_legal_rename(Rename_Mode mode,
				    const Stringy &from_group,
				    const Stringy &from_atom,
				    const Stringy &to_group,
				    const Stringy &to_atom)
{
  if (from_group.is_empty() && from_atom.is_empty())
    return false;

  if (to_group.is_empty() && to_atom.is_empty())
    return (mode != SWAP_RESONANCES);

  Reporter &rr = session.reporter();
  if (from_group.is_empty() && !to_group.is_empty())
    { rr.warning("Missing group to copy from."); return false; }
  if (!from_group.is_empty() && to_group.is_empty())
    { rr.warning("Missing group to copy to."); return false; }
  if (from_atom.is_empty() && !to_atom.is_empty())
    { rr.warning("Missing atom to copy from."); return false; }
  if (!from_atom.is_empty() && to_atom.is_empty())
    { rr.warning("Missing atom to copy to."); return false; }

  return true;
}

// ----------------------------------------------------------------------------
//
bool rename_dialog::rename(Rename_Mode mode, const List &rlist,
			   const Stringy &to_group, const Stringy &to_atom)
{
  if (rlist.empty() && (mode == RENAME_RESONANCES || mode == MERGE_RESONANCES))
    {
      Reporter &rr = session.reporter();
      rr.warning("There are no resonances to rename.");
      return false;
    }

  if (mode == MERGE_RESONANCES &&
      !confirm_merge(session, rlist, to_group, to_atom))
    return false;

  Reporter &rr = session.reporter();
  for (int ri = 0 ; ri < rlist.size() ; ++ri)
    rename_resonance((Resonance *) rlist[ri], to_group, to_atom, rr);

  return true;
}

// ----------------------------------------------------------------------------
//
static List matching_resonances(Project &proj,
				const Stringy &molecule_name,
				const Stringy &condition_name,
				const Stringy &group_name,
				const Stringy &atom_name)
{
  List matches;
  List clist = proj.condition_list();
  for (int ci = 0 ; ci < clist.size() ; ++ci)
    {
      Condition *c = (Condition *) clist[ci];
      Molecule *m = c->molecule();
      if ((molecule_name.is_empty() || m->name() == molecule_name) &&
	  (condition_name.is_empty() || c->name() == condition_name))
	{
	  const List &rlist = c->resonance_list();
	  for (int ri = 0 ; ri < rlist.size() ; ++ri)
	    {
	      Resonance *r = (Resonance *) rlist[ri];
	      if ((group_name.is_empty() ||
		   r->group()->name() == group_name) &&
		  (atom_name.is_empty() ||
		   r->atom()->name() == atom_name))
		matches.append(r);
	    }
	}
    }

  return matches;
}

// ----------------------------------------------------------------------------
//
static bool confirm_merge(Session &s, const List &rlist,
			  const Stringy &group, const Stringy &atom)
{
  if (group.is_empty() && atom.is_empty())
    return true;

  bool merge = false;
  for (int ri = 0 ; ri < rlist.size() ; ++ri)
    {
      Resonance *r = (Resonance *) rlist[ri];
      Stringy gname = (group.is_empty() ? r->group()->name() : group);
      Stringy aname = (atom.is_empty() ? r->atom()->name() : atom);
      if (r->condition()->find_resonance(gname, aname))
	merge = true;
    }

  if (!merge)
    return true;

  return query(s, "Renaming will cause merging with existing resonances.",
	       "Merge", "Cancel") == 1;
}

// ----------------------------------------------------------------------------
//
static void rename_resonance(Resonance *r, const Stringy &group,
			     const Stringy &atom, Reporter &rr)
{
  if (group.is_empty() && atom.is_empty())
    delete_resonance(r, rr);
  else
    {
      Stringy gname = (group.is_empty() ? r->group()->name() : group);
      Stringy aname = (atom.is_empty() ? r->atom()->name() : atom);

      if (r->assignment_count() == 0)
	rename_unassigned_resonance(r, gname, aname);
      else
	rename_assigned_resonance(r, gname, aname);
    }
}

// ----------------------------------------------------------------------------
//
static bool delete_resonance(Resonance *r, Reporter &rr)
{
  if (r->assignment_count() > 0)
    {
      rr.message("Resonance rename: %s not deleted, %d assignments exist.\n",
		 r->name().cstring(), r->assignment_count());
      return false;
    }
  delete r;
  return true;
}

// ----------------------------------------------------------------------------
//
static void rename_unassigned_resonance(Resonance *r, const Stringy &group,
					const Stringy &atom)
{
  Condition *c = r->condition();
  Stringy nucleus =  r->atom()->nucleus();
  Resonance *to_r = c->define_resonance(group, atom, nucleus);

  if (to_r->assignment_count() == 0)
    to_r->set_frequency(r->frequency());

  if (r != to_r)
    delete r;
}

// ----------------------------------------------------------------------------
//
static void rename_assigned_resonance(Resonance *r, const Stringy &group,
				      const Stringy &atom)
{
  List plist = r->crosspeaks();
  for (int pi = 0 ; pi < plist.size() ; ++pi)
    {
      CrossPeak *xp = (CrossPeak *) plist[pi];
      for (int a = 0 ; a < xp->dimension() ; ++a)
	if (xp->resonance(a) == r)
	  xp->ChangeAssignment(a, atom, group);
    }

  //
  // Assignment count might not be zero.
  // Resonance could have been renamed to itself.
  //
  if (r->assignment_count() == 0)
    delete r;
}

