// ----------------------------------------------------------------------------
//

#include "memalloc.h"		// use new()
#include "notifier.h"		// use Notifier
#include "print.h"		// Use print_view()
#include "project.h"		// use Project
#include "reporter.h"		// use Reporter
#include "session.h"		// use Session
#include "spectrum.h"		// Use Spectrum
#include "uicomponents.h"	// Use View_Menu
#include "uidialogs.h"		// use help_cb()
#include "uidialog.h"		// use Dialog, Dialog_Table
#include "uiview.h"		// Use View
#include "uiwait.h"		// Use wait_callbacks()
#include "winsystem.h"		// use WinSys

// ----------------------------------------------------------------------------
//
enum Print_Mode { PRINT_TO_PRINTER, PRINT_PREVIEW, PRINT_TO_FILE};

// ----------------------------------------------------------------------------
//
class print_dialog : public Dialog
{
public:
  print_dialog(Session &);
  ~print_dialog();

  static print_dialog *the(Session &);

  void show(View *view);
  void update(View *view);

private:
  Widget table, settings, printcmd, title;
  View_Menu *view_menu;
  View *view;

  static void will_delete_view_cb(void *pdialog, void *view);
  static void format_cb(Widget, CB_Data, CB_Data);
  static void view_menu_cb(Widget, CB_Data, CB_Data);
  static void print_cb(Widget, CB_Data, CB_Data);
  static void preview_cb(Widget, CB_Data, CB_Data);
  static void save_cb(Widget, CB_Data, CB_Data);
  static void close_cb(Widget, CB_Data, CB_Data);

  bool set_options(Print_Mode mode);
  bool print(Print_Mode mode);
};

static void set_table_row(WinSys &, Widget table, View *view, Axis a,
			  const Print_Options &options);
static Rectangle requested_rectangle(View *view, Widget table);
static bool set_scales(WinSys &, Widget table, Print_Options *, Reporter &);

// ----------------------------------------------------------------------------
//
void show_print_dialog(Session &s, View *view)
  { print_dialog::the(s)->show(view); }

// ----------------------------------------------------------------------------
//
print_dialog::print_dialog(Session &s) : Dialog(s, "printDialog")
{
  this->view = NULL;

  view_menu = new View_Menu(s, dialog, "viewMenu");
  ws.option_callback(view_menu->option_menu(), view_menu_cb, this);

  Table_Entry label = TABLE_LABEL;
  Table_Entry value = TABLE_TEXT_FIELD;
  table = ws.widget_table(dialog, "table", 3, 7,
		       label, label, label, label, label, label, label,
		       label, label, value, value, value, value, value,
		       label, label, value, value, value, value, value);
  ws.add_text_field_changed_callback(ws.table_element(table,1,4), format_cb, this);
  ws.add_text_field_changed_callback(ws.table_element(table,1,5), format_cb, this);
  ws.add_text_field_changed_callback(ws.table_element(table,2,4), format_cb, this);
  ws.add_text_field_changed_callback(ws.table_element(table,2,5), format_cb, this);
  const char *swnames[] = {"landscape",
			   "banner",
			   "grids",
			   "peaks",
			   "peakGroups",
			   "blackLabels",
			   NULL};
  settings = ws.switches(dialog, "settings", swnames);

  printcmd = ws.edit_field(dialog, "printCommand");
  title = ws.edit_field(dialog, "bannerTitle");

  Widget separator = ws.create_separator(dialog, "separator");

  Widget controls = ws.button_row(dialog, "controls",
			       "print", print_cb, this,
			       "preview", preview_cb, this,
			       "save", save_cb, this,
			       "close", close_cb, this,
			       "help", help_cb, &s,
			       NULL);

  Widget preview_button = ws.child_widget(controls, "preview");
  ws.set_sensitive(preview_button, print_previewer_exists());

  ws.column_attachments(separator, view_menu->option_menu(), table,
		     settings, printcmd, title,
		     separator, controls, END_OF_WIDGETS);

  Notifier &n = session.notifier();
  n.notify_me(nt_will_delete_view, will_delete_view_cb, this);
}

// ----------------------------------------------------------------------------
//
print_dialog::~print_dialog()
{
  session.dialog_table().delete_dialog("print_dialog", this);

  Notifier &n = session.notifier();
  n.dont_notify_me(nt_will_delete_view, will_delete_view_cb, this);

  delete view_menu;
}

// ----------------------------------------------------------------------------
// The default print_dialog instance.
//
print_dialog *print_dialog::the(Session &s)
{
  Stringy name = "print_dialog";
  Dialog_Table &dt = s.dialog_table();
  if (dt.get_dialog(name) == NULL)
    dt.set_dialog(name, new print_dialog(s));
  return (print_dialog *) dt.get_dialog(name);
}

// ----------------------------------------------------------------------------
//
void print_dialog::will_delete_view_cb(void *pdialog, void *view)
{
  print_dialog *pd = (print_dialog *) pdialog;
  View *v = (View *) view;

  if (pd->view == v)
    pd->update(NULL);
}

// ----------------------------------------------------------------------------
//
void print_dialog::format_cb(Widget w, CB_Data pdialog, CB_Data)
{
  print_dialog *pd = (print_dialog *) pdialog;
  Widget table = pd->table;
  WinSys &ws = pd->ws;
  if (!ws.text_field_is_empty(w))
    if (w == ws.table_element(table, 1, 4))
      ws.set_text_field(ws.table_element(table, 1, 5), "");
    else if (w == ws.table_element(table, 2, 4))
      ws.set_text_field(ws.table_element(table, 2, 5), "");
    else if (w == ws.table_element(table, 1, 5))
      ws.set_text_field(ws.table_element(table, 1, 4), "");
    else if (w == ws.table_element(table, 2, 5))
      ws.set_text_field(ws.table_element(table, 2, 4), "");
}

// ----------------------------------------------------------------------------
//
void print_dialog::view_menu_cb(Widget, CB_Data client_data, CB_Data)
{
  print_dialog *pd = (print_dialog *) client_data;
  View *view = pd->view_menu->selected_view();
  if (view)
    pd->update(view);
}

// ----------------------------------------------------------------------------
//
void print_dialog::print_cb(Widget, CB_Data client_data, CB_Data)
{
  print_dialog *pd = (print_dialog *) client_data;
  if (pd->print(PRINT_TO_PRINTER))
    pd->ws.unshow_dialog(pd->dialog);
}

// ----------------------------------------------------------------------------
//
void print_dialog::preview_cb(Widget, CB_Data client_data, CB_Data)
{
  print_dialog *pd = (print_dialog *) client_data;
  pd->print(PRINT_PREVIEW);
}

// ----------------------------------------------------------------------------
//
void print_dialog::save_cb(Widget, CB_Data client_data, CB_Data)
{
  print_dialog *pd = (print_dialog *) client_data;

  Project &proj = pd->session.project();
  Print_Options &options = proj.print_options;
  Stringy path = options.print_file;
  if (path.is_empty())
    {
      Stringy file = (pd->view ?
		      pd->view->spectrum()->name() + ".ps" :
		      Stringy("plot.ps"));
      path = proj.list_path(file);
    }
  path = pd->ws.saveas_file_dialog(pd->dialog, "Save Postscript",
				   path, "postscript", true);
  if (! path.is_empty())
    {
      options.print_file = path;
      pd->print(PRINT_TO_FILE);
    }
}

// ----------------------------------------------------------------------------
//
void print_dialog::close_cb(Widget, CB_Data client_data, CB_Data)
{
  print_dialog *pd = (print_dialog *) client_data;
  pd->set_options(PRINT_TO_PRINTER);
  pd->ws.unshow_dialog(pd->dialog);
}

// ----------------------------------------------------------------------------
//
void print_dialog::show(View *view)
{
  update(view);
  ws.show_dialog(dialog);
  ws.raise_widget(dialog);
}

// ----------------------------------------------------------------------------
//
void print_dialog::update(View *view)
{
  this->view = view;

  Stringy dialog_title = (view == NULL ? Stringy("Print") :
			  Stringy("Print ") + view->name());
  ws.set_dialog_title(dialog, dialog_title);

  view_menu->set_view_choice(view);

  Print_Options &options = session.project().print_options;
  set_table_row(ws, table, view, X, options);
  set_table_row(ws, table, view, Y, options);

  ws.set_edit_value(printcmd, options.print_command);
  ws.set_edit_value(title, options.title);

  ws.set_switch(settings, "landscape", options.landscape);
  ws.set_switch(settings, "banner", options.show_banner);
  ws.set_switch(settings, "grids", options.show_grids);
  ws.set_switch(settings, "peaks", options.show_peaks);
  ws.set_switch(settings, "peakGroups", options.show_peakgroups);
  ws.set_switch(settings, "blackLabels", options.black_labels);
}

// ----------------------------------------------------------------------------
//
static void set_table_row(WinSys &ws, Widget table, View *view, Axis a,
			  const Print_Options &options)
{
  int row = (a == X ? 1 : 2);
  Stringy axislabel = formatted_string("w%d", (view ? view->axis(a)+1 : row));
  ws.set_label_text(ws.table_element(table, row, 0), axislabel);
  Stringy nucleus = (view == NULL ? Stringy("1H") :
		     view->spectrum()->nucleus_type(view->axis(a)));
  ws.set_label_text(ws.table_element(table, row, 1), nucleus);
  if (view)
    {
      Rectangle vr = view->view_rectangle();
      double downfield = view->map(vr.max(a), a, PPM, view->axis_units());
      double upfield = view->map(vr.min(a), a, PPM, view->axis_units());
      ws.set_numeric_text_field(ws.table_element(table, row, 2),
				"%.4g", downfield);
      ws.set_numeric_text_field(ws.table_element(table, row, 3),
				"%.4g", upfield);
    }
  else
    {
      ws.set_text_field(ws.table_element(table, row, 2), "");
      ws.set_text_field(ws.table_element(table, row, 3), "");
    }

  if (options.fixed_scale)
    ws.set_numeric_text_field(ws.table_element(table, row, 5), "%.4g",
			      options.fixed_scales[a]);
  else
    ws.set_numeric_text_field(ws.table_element(table, row, 4), "%.0f",
			      options.pages[a]);

  if (options.label_spacing[a] > 0)
    ws.set_numeric_text_field(ws.table_element(table, row, 6), "%.4g",
			      options.label_spacing[a]);
  else
    ws.set_text_field(ws.table_element(table, row, 6), "");
}

// ----------------------------------------------------------------------------
//
bool print_dialog::set_options(Print_Mode mode)
{
  Print_Options &options = session.project().print_options;

  options.preview = (mode == PRINT_PREVIEW);
  options.print_to_file = (mode == PRINT_TO_FILE);

  if (!set_scales(ws, table, &options, session.reporter()))
    return false;

  options.landscape = ws.switch_state(settings, "landscape");
  options.show_banner = ws.switch_state(settings, "banner");
  options.show_grids = ws.switch_state(settings, "grids");
  options.show_peaks = ws.switch_state(settings, "peaks");
  options.show_peakgroups = ws.switch_state(settings, "peakGroups");
  options.black_labels = ws.switch_state(settings, "blackLabels");

  options.print_command = ws.edit_value(printcmd);
  options.title = ws.edit_value(title);

  return true;
}

// ----------------------------------------------------------------------------
//
static Rectangle requested_rectangle(View *view, Widget table)
{
  WinSys &ws = view->session().window_system();
  double downfield_x = ws.numeric_text_field(ws.table_element(table, 1, 2));
  double upfield_x = ws.numeric_text_field(ws.table_element(table, 1, 3));
  double downfield_y = ws.numeric_text_field(ws.table_element(table, 2, 2));
  double upfield_y = ws.numeric_text_field(ws.table_element(table, 2, 3));
  Rectangle rect(view->map(upfield_x, X, view->axis_units(), PPM),
		 view->map(upfield_y, Y, view->axis_units(), PPM),
		 view->map(downfield_x, X, view->axis_units(), PPM),
		 view->map(downfield_y, Y, view->axis_units(), PPM));
  return rect;
}

// ----------------------------------------------------------------------------
//
static bool set_scales(WinSys &ws, Widget table,
		       Print_Options *options, Reporter &rr)
{
  int xpages = (int) ws.numeric_text_field(ws.table_element(table, 1, 4));
  int ypages = (int) ws.numeric_text_field(ws.table_element(table, 2, 4));
  double xppmcm = ws.numeric_text_field(ws.table_element(table, 1, 5));
  double yppmcm = ws.numeric_text_field(ws.table_element(table, 2, 5));
  if (xppmcm == 0 && yppmcm == 0 && xpages > 0 && ypages > 0)
    {
      options->fixed_scale = false;
      options->pages[0] = xpages;
      options->pages[1] = ypages;
    }
  else if (xpages == 0 && ypages == 0 && xppmcm > 0 && yppmcm > 0)
    {
      options->fixed_scale = true;
      options->fixed_scales[0] = xppmcm;
      options->fixed_scales[1] = yppmcm;
    }
  else
    {
      rr.warning("You must specify either the number of pages for both axes,\n"
		 "or the ppm/cm scale for both axes.\n");

      return false;
    }

  options->label_spacing[0] = ws.numeric_text_field(ws.table_element(table, 1, 6));
  options->label_spacing[1] = ws.numeric_text_field(ws.table_element(table, 2, 6));

  return true;
}

// ----------------------------------------------------------------------------
//
bool print_dialog::print(Print_Mode mode)
{
  if (view == NULL || !set_options(mode))
    return false;

  Rectangle r = requested_rectangle(view, table);
  return print_view(view, r,
		    session.project().print_options,
		    wait_callbacks(session));
}
