// ----------------------------------------------------------------------------
// Peak list options dialog
//
#include <float.h>		// use FLT_MAX
#include <string.h>		// use strcmp()

#include "atom.h"		// use Atom
#include "crosspeak.h"		// Use CrossPeak
#include "group.h"		// Use Group
#include "list.h"		// Use List
#include "memalloc.h"		// use new()
#include "model.h"		// use Model
#include "molecule.h"		// use Molecule
#include "peak.h"		// Use Peak
#include "resonance.h"		// Use Resonance
#include "session.h"		// use Session
#include "spectrum.h"		// Use Spectrum
#include "stringc.h"		// Use Stringy
#include "uidialogs.h"		// use help_cb()
#include "uidialog.h"		// use Dialog, Dialog_Table
#include "uipeak.h"		// use format_changed()
#include "uiplop.h"		// Use Peak_List_Options
#include "utility.h"		// Use fatal_error()
#include "winsystem.h"		// Use Widget

// ----------------------------------------------------------------------------
//
class peak_list_options_dialog : public Dialog
{
public:
  peak_list_options_dialog(Session &);
  ~peak_list_options_dialog();

  static peak_list_options_dialog *the(Session &);
  static bool the_exists(Session &);

  void show(const Peak_List_Options &plo, peak_list_dialog *pld);
  void unshow(peak_list_dialog *pld);

private:
  Widget toggles, sortby_menu, sortby_axis, sortby_pairx, assign_format;
  Peak_List_Options plo;
  peak_list_dialog *pld;


  void apply();
};

static bool peak_resonance_axis(const CrossPeak *xp, const Stringy &resname,
				int *axis);
static Stringy fit_residual_value(const CrossPeak *xp);
static Stringy fit_height_value(const CrossPeak *xp);
static Stringy data_height_value(const CrossPeak *xp);
static Stringy volume_value(const CrossPeak *xp);
static Stringy volume_error_value(const CrossPeak *xp);
static Stringy linewidth_value(const CrossPeak *xp, int axis);
static bool assignment_distance(const CrossPeak *xp, double *d);
static bool proton_axes(Spectrum *sp, int *axis1, int *axis2);
static Stringy mardigras_format(const CrossPeak *xp);
static Stringy diana_format(const CrossPeak *xp);
static CrossPeak *transposed_peak(const CrossPeak *xp);
static const CrossPeak *above_diagonal(const CrossPeak *xp1,
				       const CrossPeak *xp2);
static CrossPeak *assigned_peak(const List &peaks, Spectrum *sp,
				Resonance *r1, Resonance *r2);
static int user_compare(const void *a1, const void *a2);
static int volume_compare(const void *a1, const void *a2);
static int assignment_distance_compare(const void *a1, const void *a2);
static int frequency_compare(const CrossPeak *xp1, const CrossPeak *xp2,
			     int axis);

// ----------------------------------------------------------------------------
//
const char *Peak_List_Field_Names[] =
{
  "assignmentName",
  "userName",
  "frequency",
  "frequencyHz",
  "resonanceDev",
  "volume",
  "volumeError",
  "transVolume",
  "fitResidual",
  "fitHeight",
  "dataHeight",
  "noise",
  "linewidth",
  "resFreq",
  "resDev",
  "spectrumName",
  "assignmentDistance",
  "mardigras",
  "diana",
  "note",
  NULL
};

// ----------------------------------------------------------------------------
//
const char *Peak_List_Sort_Names[] =
{
  "noSort",
  "resonanceNameSort",
  "userNameSort",
  "frequencySort",
  "volumeSort",
  "assignmentDistanceSort",
  NULL
};

// ----------------------------------------------------------------------------
//
Peak_List_Options::Peak_List_Options()
{
  title = "Peak List";
  for (int f = 0 ; f < PEAK_LIST_FIELD_COUNT ; ++f)
    fields[f] = false;
  fields[ASSIGNMENT_NAME_FIELD] = true;
  fields[FREQUENCY_FIELD] = true;
  sort_type = RESONANCE_NAME_SORT;
  sort_axis = 0;
  sort_pairx = false;
  assignment_format = "";
}

// ----------------------------------------------------------------------------
//
void show_peak_list_options_dialog(Session &s, const Peak_List_Options &plo,
				   peak_list_dialog *pld)
  { peak_list_options_dialog::the(s)->show(plo, pld); }
void unshow_peak_list_options_dialog(Session &s, peak_list_dialog *pld)
{
  if (peak_list_options_dialog::the_exists(s))
    peak_list_options_dialog::the(s)->unshow(pld);
}

// ----------------------------------------------------------------------------
//
Stringy Peak_List_Options::heading(int max_dimension)
{
  Stringy h;

  for (int f = 0 ; f < PEAK_LIST_FIELD_COUNT ; ++f)
    if (fields[f])
      h << field_heading(f, max_dimension);

  return h;
}

// ----------------------------------------------------------------------------
//
Stringy Peak_List_Options::text(const CrossPeak *xp, int max_dimension)
{
  Stringy h;

  for (int f = 0 ; f < PEAK_LIST_FIELD_COUNT ; ++f)
    if (fields[f])
      h << field_value(f, xp, max_dimension);

  return h;
}

// ----------------------------------------------------------------------------
//
peak_list_options_dialog::peak_list_options_dialog(Session &s)
  : Dialog(s, "peakListOptionsDialog")
{
  this->pld = NULL;

  toggles = ws.switches(dialog, "switches", Peak_List_Field_Names, 2);

  sortby_menu = ws.option_menu(dialog, "sortBy", Peak_List_Sort_Names);

  sortby_axis = ws.edit_field(dialog, "sortAxis");
  sortby_pairx = ws.create_toggle_button(dialog, "sortPairX");
  assign_format = ws.edit_field(dialog, "assignFormat");

  Widget separator = ws.create_separator(dialog, "separator");

  Widget controls = ws.button_row(dialog, "controls",
			       "ok", ok_cb, this,
			       "apply", apply_cb, this,
			       "close", close_cb, this,
			       "help", help_cb, &s,
			       NULL);

  ws.column_attachments(separator, toggles,
		     sortby_menu, sortby_axis, sortby_pairx, assign_format,
		     separator, controls, END_OF_WIDGETS);
}

// ----------------------------------------------------------------------------
//
peak_list_options_dialog::~peak_list_options_dialog()
{
  session.dialog_table().delete_dialog("peak_list_options_dialog", this);
}

// ----------------------------------------------------------------------------
// The default peak_list_options_dialog instance.
//
peak_list_options_dialog *peak_list_options_dialog::the(Session &s)
{
  Stringy name = "peak_list_options_dialog";
  Dialog_Table &dt = s.dialog_table();
  if (dt.get_dialog(name) == NULL)
    dt.set_dialog(name, new peak_list_options_dialog(s));
  return (peak_list_options_dialog *) dt.get_dialog(name);
}

// ----------------------------------------------------------------------------
//
bool peak_list_options_dialog::the_exists(Session &s)
{
  return (s.dialog_table().get_dialog("peak_list_options_dialog") != NULL);
}

// ----------------------------------------------------------------------------
//
void peak_list_options_dialog::show(const Peak_List_Options &plo,
				    peak_list_dialog *pld)
{
  this->plo = plo;
  this->pld = pld;

  Stringy dialog_title = Stringy("Options ") + plo.title;
  ws.set_dialog_title(dialog, dialog_title);

  for (int f = 0 ; f < PEAK_LIST_FIELD_COUNT ; ++f)
    ws.set_switch(toggles, Peak_List_Field_Names[f], plo.fields[f]);

  ws.set_option(sortby_menu, index_name(Peak_List_Sort_Names, plo.sort_type));
  ws.set_numeric_edit_value(sortby_axis, "%.0f", plo.sort_axis + 1);
  ws.set_toggle_button(sortby_pairx, plo.sort_pairx, true);
  ws.set_edit_value(assign_format, plo.assignment_format);

  ws.show_dialog(dialog);
  ws.raise_widget(dialog);
}

// ----------------------------------------------------------------------------
//
void peak_list_options_dialog::unshow(peak_list_dialog *pld)
{
  if (this->pld == pld)
    {
      this->pld = NULL;
      ws.unshow_dialog(dialog);
    }
}

// ----------------------------------------------------------------------------
//
void peak_list_options_dialog::apply()
{
  for (int f = 0 ; f < PEAK_LIST_FIELD_COUNT ; ++f)
    plo.fields[f] =
      ws.switch_state(toggles, index_name(Peak_List_Field_Names, f));

  int t = name_index(Peak_List_Sort_Names, ws.option_selected(sortby_menu));
  plo.sort_type = (t >= 0 ? (Peak_List_Sort) t : NO_SORT);
  plo.sort_axis = (int) ws.numeric_edit_value(sortby_axis) - 1;
  plo.sort_pairx = ws.toggle_button_state(sortby_pairx);
  plo.assignment_format = ws.edit_value(assign_format);
  
  if (pld)
    change_peak_list_format(pld, plo);		// Notify peak list
}

// ----------------------------------------------------------------------------
//
Stringy Peak_List_Options::field_heading(int field, int max_dimension)
{
  Stringy heading;

  switch (field)
    {
    case ASSIGNMENT_NAME_FIELD: heading = "      Assignment  "; break;
    case USER_NAME_FIELD:	heading = "      User  "; break;
    case RES_FREQ_FIELD:	heading = "    Freq   "; break;
    case RES_DEV_FIELD:		heading = "    Dev    "; break;
    case SPECTRUM_NAME_FIELD:	heading = "   Spectrum   "; break;
    case FREQUENCY_FIELD:
      for (int a = 0 ; a < max_dimension ; ++a)
	heading << formatted_string("       w%d  ", a+1);
      break;
    case HZ_FREQUENCY_FIELD:
      for (int a = 0 ; a < max_dimension ; ++a)
	heading << formatted_string("   w%d (Hz) ", a+1);
      break;
    case RESONANCE_DEV_FIELD:
      for (int a = 0 ; a < max_dimension ; ++a)
	heading << formatted_string("  Dev w%d ", a+1);
      break;
    case VOLUME_FIELD:		heading = "      Volume  "; break;
    case VOLUME_ERROR_FIELD:	heading = " Volume Error "; break;
    case TRANSPOSED_VOLUME_FIELD: heading = " Trans-Volume "; break;
    case FIT_RESIDUAL_FIELD:	heading = "  Fit RMS %  "; break;
    case FIT_HEIGHT_FIELD:	heading = "  Fit Height "; break;
    case DATA_HEIGHT_FIELD:	heading = " Data Height "; break;
    case NOISE_FIELD:		heading = "      S/N  "; break;
    case LINEWIDTH_FIELD:
      for (int a = 0 ; a < max_dimension ; ++a)
	heading << formatted_string("  lw%d (hz) ", a+1);
      break;
    case ASSIGNMENT_DISTANCE_FIELD: heading = "  Distance  "; break;
    case MARDIGRAS_FIELD:
      heading = "ATOM1   ATOM2   INTENSITY   ERROR";
      break;
    case DIANA_FIELD:
      heading = "    ATOM1           ATOM2        NOTE   ";
      break;
    case NOTE_FIELD:		heading = "    Note    "; break;
    default:
      fatal_error("Peak_List_Options::field_heading(): Bad field.\n");
    }

  return heading;
}
// ----------------------------------------------------------------------------
//
Stringy Peak_List_Options::field_value(int field, const CrossPeak *xp,
				       int max_dimension)
{
  Stringy value;

  switch (field)
    {
    case ASSIGNMENT_NAME_FIELD:
      {
	Stringy assignment = (assignment_format.is_empty() ?
			      xp->assignment_name() :
			      xp->assignment_name(assignment_format));
	value = formatted_string("%17s ", assignment.cstring());
	break;
      }
    case USER_NAME_FIELD:
      value = formatted_string("%11s ", xp->user_name().cstring()); break;
    case RES_FREQ_FIELD:
      {
	int axis;
	if (peak_resonance_axis(xp, selected_resonance, &axis))
	  value = formatted_string("%10.3f ", xp->frequency(axis));
	else
	  value = formatted_string("%10s ", "");
	break;
      }
    case RES_DEV_FIELD:
      {
	int axis;
	if (peak_resonance_axis(xp, selected_resonance, &axis))
	  {
	    double rfreq = xp->resonance(axis)->frequency();
	    value = formatted_string("%+10.3f ", xp->frequency(axis) - rfreq);
	  }
	else
	  value = formatted_string("%10s ", "");
	break;
      }
    case SPECTRUM_NAME_FIELD:
      value = formatted_string("%17s ", xp->spectrum()->name().cstring());
      break;
    case FREQUENCY_FIELD:
      {
	int dim = xp->dimension();
	for (int a = 0 ; a < dim ; ++a)
	  value << formatted_string("%10.3f ", xp->frequency(a));
	for (int a = dim ; a < max_dimension ; ++a)
	  value << formatted_string("%10s ", "");
	break;
      }
    case HZ_FREQUENCY_FIELD:
      {
	int dim = xp->dimension();
	for (int a = 0 ; a < dim ; ++a)
	  {
	    double freq_hz = xp->spectrum()->map(xp->frequency(a), a, PPM, HZ);
	    value << formatted_string("%10.2f ", freq_hz);
	  }
	for (int a = dim ; a < max_dimension ; ++a)
	  value << formatted_string("%10s ", "");
	break;
      }
    case RESONANCE_DEV_FIELD:
      {
	int dim = xp->dimension();
	for (int a = 0 ; a < dim ; ++a)
	  {
	    Resonance *res = xp->resonance(a);
	    if (res)
	      value << formatted_string("%+8.3f ", (xp->frequency(a)
						    - res->frequency()));
	    else
	      value << formatted_string("%8s ", "");
	  }
	for (int a = dim ; a < max_dimension ; ++a)
	  value << formatted_string("%8s ", "");
	break;
      }
    case VOLUME_FIELD:
      value = volume_value(xp); break;
    case VOLUME_ERROR_FIELD:
      value = volume_error_value(xp); break;
    case TRANSPOSED_VOLUME_FIELD:
      value = volume_value(transposed_peak(xp)); break;
    case FIT_RESIDUAL_FIELD:
      value = fit_residual_value(xp); break;
    case FIT_HEIGHT_FIELD:
      value = fit_height_value(xp); break;
    case DATA_HEIGHT_FIELD:
      value = data_height_value(xp); break;
    case NOISE_FIELD:
      value = formatted_string("%10.0f ", xp->SignalToNoise()); break;
    case LINEWIDTH_FIELD:
      for (int a = 0 ; a < max_dimension ; ++a)
	value << linewidth_value(xp, a);
      break;
    case ASSIGNMENT_DISTANCE_FIELD:
      {
	double distance;
	value = (assignment_distance(xp, &distance) ?
		 formatted_string("%10.3g ", distance) :
		 formatted_string("%10s ", ""));
	break;
      }
    case MARDIGRAS_FIELD:
      value = mardigras_format(xp); break;
    case DIANA_FIELD:
      value = diana_format(xp); break;
    case NOTE_FIELD:
      value = xp->GetNote(); break;
    default:
      fatal_error("Peak_List_Options::field_heading(): Bad field.\n");
    }

  return value;
}

// ----------------------------------------------------------------------------
//
static bool peak_resonance_axis(const CrossPeak *xp, const Stringy &resname,
				int *axis)
{
  for (int a = xp->dimension() - 1 ; a >= 0 ; --a)
    {
      Resonance *r = xp->resonance(a);
      if (r && r->name() == resname)
	{
	  *axis = a;
	  return true;
	}
    }

  return false;
}

// ----------------------------------------------------------------------------
//
static Stringy fit_residual_value(const CrossPeak *xp)
{
  if (xp->type() == peak)
    {
      Peak *pk = (Peak *) xp;
      if (pk->IntegrateByFit())
	return formatted_string("%12.1f ", 100 * pk->fit_residual());
    }
  return formatted_string("%12s ", "");
}

// ----------------------------------------------------------------------------
//
static Stringy fit_height_value(const CrossPeak *xp)
{
  if (xp->type() == peak)
    {
      Peak *pk = (Peak *) xp;
      if (pk->IntegrateByFit())
	return formatted_string("%12.0f ", pk->FitHeight());
    }
  return formatted_string("%12s ", "");
}

// ----------------------------------------------------------------------------
//
static Stringy data_height_value(const CrossPeak *xp)
{
  if (xp->type() == peak)
    {
      Peak *pk = (Peak *) xp;
      return formatted_string("%12.0f ", pk->DataHeight());
    }
  return formatted_string("%12s ", "");
}

// ----------------------------------------------------------------------------
//
static Stringy volume_value(const CrossPeak *xp)
{
  double volume;
  if (xp && xp->volume(&volume))
    {
      Stringy method = (xp->type() == peak ?
			index_name(Integration_Method_Short_Names,
				   ((Peak *)xp)->GetIntegrateMethod()) :
			Stringy(""));
      return formatted_string("%10.2e %2s ", volume, method.cstring());
    }

  return formatted_string("%10s %2s ", "", "");
}

// ----------------------------------------------------------------------------
//
static Stringy volume_error_value(const CrossPeak *xp)
{
  double verror;
  Stringy method;
  if (xp && xp->volume_error(&verror, &method))
    return formatted_string("%3.0f %-9s ", 100 * verror, method.cstring());

  return formatted_string("%3s %9s ", "", "");
}

// ----------------------------------------------------------------------------
//
static Stringy linewidth_value(const CrossPeak *xp, int axis)
{
  if (xp->type() == peak &&
      ((Peak *)xp)->HasLinewidth() &&
      axis < xp->dimension())
    {
      Peak *pk = (Peak *) xp;
      double lw = pk->spectrum()->scale(pk->linewidth(axis), axis, PPM, HZ);
      return formatted_string("%10.1f ", lw);
    }

  return formatted_string("%10s ", "");
}

// ----------------------------------------------------------------------------
// Distance between atoms assigned to first and last axes.
//
static bool assignment_distance(const CrossPeak *xp, double *d)
{
  Spectrum *sp = xp->spectrum();
  Molecule *m = sp->molecule();
  const List &models = m->model_list();
  if (models.size() == 0)
    return false;
  Model *model = (Model *) models[0];

  int axis1, axis2;
  if (sp->dimension() == 2 || !proton_axes(sp, &axis1, &axis2))
    {
      axis1 = 0;
      axis2 = xp->dimension() - 1;
    }

  SPoint xyz1, xyz2;
  if (xp->resonance(axis1) == NULL ||
      xp->resonance(axis2) == NULL ||
      !model->coordinates(*xp->resonance(axis1)->atom(), &xyz1) ||
      !model->coordinates(*xp->resonance(axis2)->atom(), &xyz2))
    return false;

  *d = distance(xyz1, xyz2);
  return true;
}

// ----------------------------------------------------------------------------
//
static bool proton_axes(Spectrum *sp, int *axis1, int *axis2)
{
  int a1, a2, dim = sp->dimension();
  for (a1 = 0 ; a1 < dim ; ++a1)
    if (sp->nucleus_type(a1) == "1H")
      break;
  for (a2 = a1 + 1  ; a2 < dim ; ++a2)
    if (sp->nucleus_type(a2) == "1H")
      break;
  if (a2 < dim)
    {
      *axis1 = a1;
      *axis2 = a2;
      return true;
    }
  return false;
}

// ----------------------------------------------------------------------------
//
static Stringy mardigras_format(const CrossPeak *xp)
{
  Spectrum *sp = xp->spectrum();
  double volume;
  if (sp->dimension() == 2 && xp->volume(&volume))
    {
      Resonance *r1 = xp->resonance(0);
      Resonance *r2 = xp->resonance(1);
      if (r1 && r1->group()->number() > 0 &&
	  r2 && r2->group()->number() > 0)
	{
	  double ve;
	  Stringy verror = (xp->volume_error(&ve, NULL) ?
			    formatted_string("%.0f", 100 * ve) : Stringy(""));
	  return formatted_string("%-4.4s%3d %-4.4s%3d %14.6g %7s ",
				  r1->atom()->name().cstring(),
				  r1->group()->number(),
				  r2->atom()->name().cstring(),
				  r2->group()->number(),
				  volume, verror.cstring());
	}
    }

  return formatted_string("%4s%3s %4s%3s %14s %7s ", "", "", "", "", "", "");
}

// ----------------------------------------------------------------------------
//
static Stringy diana_format(const CrossPeak *xp)
{
  Spectrum *sp = xp->spectrum();
  if (sp->dimension() == 2)
    {
      Resonance *r1 = xp->resonance(0);
      Resonance *r2 = xp->resonance(1);
      if (r1 && r1->group()->number() > 0 &&
	  r2 && r2->group()->number() > 0)
	return formatted_string("%3d %-5.5s%-5.5s%4d %-5.5s%-5.5s  %s ",
				r1->group()->number(),
				r1->group()->symbol().cstring(),
				r1->atom()->name().cstring(),
				r2->group()->number(),
				r2->group()->symbol().cstring(),
				r2->atom()->name().cstring(),
				xp->GetNote().cstring());
    }

  return formatted_string("%3s %5s%5s%4s %5s%5s  %s ",
			  "", "", "", "", "", "", "");
}

// ----------------------------------------------------------------------------
//
static CrossPeak *transposed_peak(const CrossPeak *xp)
{
  Spectrum *sp = xp->spectrum();

  if (sp->dimension() == 2 && sp->is_homonuclear(0, 1))
    {
      Resonance *r1 = xp->resonance(0);
      Resonance *r2 = xp->resonance(1);
      if (r1 && r2)
	return assigned_peak(r1->crosspeaks(), sp, r2, r1);
    }

  return NULL;
}

// ----------------------------------------------------------------------------
//
static const CrossPeak *above_diagonal(const CrossPeak *xp1,
				       const CrossPeak *xp2)
  { return (xp1->frequency(1) - xp1->frequency(0)
	    > xp2->frequency(1) - xp2->frequency(0) ? xp1 : xp2); }

// ----------------------------------------------------------------------------
//
static CrossPeak *assigned_peak(const List &peaks, Spectrum *sp,
				Resonance *r1, Resonance *r2)
{
  for (int pi = 0 ; pi < peaks.size() ; ++pi)
    {
      CrossPeak *pk = (CrossPeak *) peaks[pi];
      if (pk->spectrum() == sp &&
	  pk->resonance(0) == r1 &&
	  pk->resonance(1) == r2)
	return pk;
    }
  return NULL;
}

// ----------------------------------------------------------------------------
//
class axis_comparitor : public ListOrder
{
public:
  axis_comparitor(int axis,
		  int (*compare)(const CrossPeak *, const CrossPeak *, int));
  virtual int operator()(const void *a, const void *b) const;
private:
  int axis;
  int (*compare)(const CrossPeak *, const CrossPeak *, int);
};

// ----------------------------------------------------------------------------
//
axis_comparitor::axis_comparitor(int axis, int (*compare)(const CrossPeak *,
							  const CrossPeak *,
							  int))
				 
{
  this->axis = (axis < 0 ? 0 : axis);
  this->compare = compare;
}

// ----------------------------------------------------------------------------
//
int axis_comparitor::operator()(const void *a, const void *b) const
{
  return (*compare)((const CrossPeak *) a, (const CrossPeak *) b, axis);
}

// ----------------------------------------------------------------------------
//
class pairx_comparitor : public ListOrder
{
public:
  pairx_comparitor(ListOrder *c) { this->c = c; }
  virtual int operator()(const void *a, const void *b) const;
private:
  ListOrder *c;
};

// ----------------------------------------------------------------------------
//
int pairx_comparitor::operator()(const void *a, const void *b) const
{
  const CrossPeak *xp1 = (const CrossPeak *) a;
  const CrossPeak *txp1 = transposed_peak(xp1);
  const CrossPeak *cxp1 = (txp1 ? above_diagonal(xp1, txp1) : xp1);

  const CrossPeak *xp2 = (const CrossPeak *) b;
  const CrossPeak *txp2 = transposed_peak(xp2);
  const CrossPeak *cxp2 = (txp2 ? above_diagonal(xp2, txp2) : xp2);

  if (cxp1 == cxp2 && xp1 != xp2)
    return (above_diagonal(xp1, xp2) == xp1 ? +1 : -1);
  
  return (*c)(cxp1, cxp2);
}

// ----------------------------------------------------------------------------
//
void Peak_List_Options::sort(List &peak_list)
{
  ListOrder *compare = NULL;

  switch (sort_type)
    {
    case NO_SORT: break;
    case USER_NAME_SORT: compare = new ListOrder(user_compare); break;
    case FREQUENCY_SORT:
      compare = new axis_comparitor(sort_axis, frequency_compare); break;
    case RESONANCE_NAME_SORT:
      compare = new axis_comparitor(sort_axis, compare_assignments); break;
    case VOLUME_SORT: compare = new ListOrder(volume_compare); break;
    case ASSIGNMENT_DISTANCE_SORT:
      compare = new ListOrder(assignment_distance_compare); break;
    }

  if (compare == NULL)
    return;

  if (sort_pairx)
    peak_list.sort(pairx_comparitor(compare));
  else
    peak_list.sort(*compare);

  delete compare;
}

// ----------------------------------------------------------------------------
//
static int user_compare(const void *a1, const void *a2)
{
  CrossPeak *xp1 = (CrossPeak *) a1;
  CrossPeak *xp2 = (CrossPeak *) a2;

  return strcmp(xp1->user_name().cstring(), xp2->user_name().cstring());
}

// ----------------------------------------------------------------------------
//
static int volume_compare(const void *a1, const void *a2)
{
  CrossPeak *xp1 = (CrossPeak *) a1;
  CrossPeak *xp2 = (CrossPeak *) a2;

  double v1, v2;
  if (!xp1->volume(&v1))
    return (xp2->volume() ? +1 : 0);
  else if (!xp2->volume(&v2))
    return -1;

  return (v2 > v1 ? +1 : (v2 < v1 ? -1 : 0));
}

// ----------------------------------------------------------------------------
//
static int assignment_distance_compare(const void *a1, const void *a2)
{
  CrossPeak *xp1 = (CrossPeak *) a1;
  CrossPeak *xp2 = (CrossPeak *) a2;

  double d1, d2;
  bool has_dist_1 = assignment_distance(xp1, &d1);
  bool has_dist_2 = assignment_distance(xp2, &d2);

  if (has_dist_1 && has_dist_2)
    return (d2 > d1 ? +1 : (d2 < d1 ? -1 : 0));
  else if (has_dist_1 && !has_dist_2)
    return -1;
  else if (!has_dist_1 && has_dist_2)
    return +1;

  return 0;
}

// ----------------------------------------------------------------------------
//
static int frequency_compare(const CrossPeak *xp1, const CrossPeak *xp2, int a)
{
  double freq1 = (a < xp1->dimension() ? xp1->frequency(a) : FLT_MAX);
  double freq2 = (a < xp2->dimension() ? xp2->frequency(a) : FLT_MAX);
  return compare_doubles(freq1, freq2);
}

// ----------------------------------------------------------------------------
//
Peak_List_Options default_peak_list_format()
{
  Peak_List_Options dpo;

  for (int f = 0 ; f < PEAK_LIST_FIELD_COUNT ; ++f)
    dpo.fields[f] = false;

  dpo.fields[ASSIGNMENT_NAME_FIELD] = true;
  dpo.fields[FREQUENCY_FIELD] = true;

  dpo.sort_type = NO_SORT;
  dpo.sort_axis = 0;
  dpo.sort_pairx = false;
  dpo.assignment_format = "";

  return dpo;
}

// ----------------------------------------------------------------------------
//
Peak_List_Options resonance_peaks_format()
{
  Peak_List_Options rpo;

  for (int f = 0 ; f < PEAK_LIST_FIELD_COUNT ; ++f)
    rpo.fields[f] = false;

  rpo.fields[ASSIGNMENT_NAME_FIELD] = true;
  rpo.fields[RES_FREQ_FIELD] = true;
  rpo.fields[RES_DEV_FIELD] = true;
  rpo.fields[SPECTRUM_NAME_FIELD] = true;

  rpo.sort_type = RESONANCE_NAME_SORT;
  rpo.sort_axis = 0;
  rpo.sort_pairx = false;
  rpo.assignment_format = "";

  return rpo;
}
