// ----------------------------------------------------------------------------
//
#include <stdio.h>		// Use FILE, sprintf(), fprintf(), fopen()
#include <string.h>		// Use strcmp()

#include "atom.h"		// use Atom
#include "command.h"		// use parse_key()
#include "condition.h"		// Use Condition
#include "crosspeak.h"		// Use CrossPeak
#include "group.h"		// Use Group
#include "label.h"		// Use Label
#include "list.h"		// Use List
#include "memalloc.h"		// use new()
#include "num.h"		// Use max()
#include "peak.h"
#include "project.h"		// use Project
#include "reporter.h"		// use Reporter
#include "resonance.h"		// Use Resonance
#include "session.h"		// use Session
#include "spectrum.h"		// Use Spectrum
#include "stringc.h"		// Use Stringy
#include "table.h"		// Use Table
#include "uicomponents.h"	// Use resonance_field(), ...
#include "uidialogs.h"		// use help_cb()
#include "uidialog.h"		// use Dialog, Dialog_Table
#include "uipeak.h"
#include "uiplop.h"		// Use resonance_peaks_format()
#include "notifier.h"		// use Notifier
#include "utility.h"		// Use fatal_error(), message()
#include "uiview.h"		// Use View
#include "winsystem.h"		// use WinSys

// ----------------------------------------------------------------------------
//
class peak_list_dialog : public Dialog
{
public:
  peak_list_dialog(Session &, const List &peaks, const Peak_List_Options &);
  virtual ~peak_list_dialog();

  void show();
  virtual void update();
  void unshow();
  Stringy title();
  void set_title(const Stringy &title);
  void set_peak_list(const List &peaks);
  bool has_peak(CrossPeak *xp);
  Widget customize();

  void peak_changed(const CrossPeak *xp);
  void show_peak(const CrossPeak *xp);
  void delete_peak(CrossPeak *xp);
  Peak_List_Options &format();
  virtual void change_format(const Peak_List_Options &);

private:
  Widget custom, scroller, heading, list, search_result, search_text;
  List peaks;
  int max_dimension;
  Table peak_table;
  Peak_List_Options plformat;
  bool center;			// Scroll to center selected peak

  static void select_cb(Widget, CB_Data, CB_Data);
  static void goto_cb(Widget, CB_Data, CB_Data);
  static void key_press_cb(Widget, CB_Data, CB_Data);
  static void focus_cb(Widget, CB_Data, CB_Data);
  static void search_cb(Widget, CB_Data, CB_Data);
  static void search_next_cb(Widget, CB_Data, CB_Data);
  static void update_cb(Widget, CB_Data, CB_Data);
  static void save_cb(Widget, CB_Data, CB_Data);
  static void options_cb(Widget, CB_Data, CB_Data);
  static void close_cb(Widget, CB_Data, CB_Data);
  static void ornaments_changed_cb(void *pldialog, void *olist);
  static void selection_changed_cb(void *pldialog, void *);
  static void will_delete_crosspeak_cb(void *pldialog, void *xpeak);

  void load_list(const List &plist);
  bool write_file(const Stringy &path);
  void search(bool next_occurence);
};

// ----------------------------------------------------------------------------
//
class spectrum_peaks_dialog : public peak_list_dialog
{
public:
  spectrum_peaks_dialog(Session &, Spectrum *, const Peak_List_Options &);
  virtual ~spectrum_peaks_dialog();
  Spectrum *spectrum();
  virtual void update();
  virtual void change_format(const Peak_List_Options &);
  static List dialog_list(Dialog_Table &);
private:
  Spectrum *sp;

  static void will_delete_spectrum_cb(void *pldialog, void *spectrum);
};

// ----------------------------------------------------------------------------
//
class resonance_peaks_dialog : public peak_list_dialog
{
public:
  resonance_peaks_dialog(Session &, Resonance *r);
  ~resonance_peaks_dialog();

  static resonance_peaks_dialog *the(Session &);

  Resonance *resonance();
  void show(Resonance *res);
  virtual void update();
  void unshow(const Resonance *res);
  bool shown(const Resonance *res);
  void set_resonance(const Resonance *res);
  void changed(const Resonance *res);
private:
  Resonance_Field *resfield;
  Resonance *res;

  static void resonance_chosen_cb(Widget, CB_Data, CB_Data);
  static void will_delete_resonance_cb(void *pldialog, void *resonance);
  static void resonance_changed_cb(void *pldialog, void *resonance);
  static void deferred_update_cb(CB_Data pldialog);
};

// ----------------------------------------------------------------------------
//
static bool show_spectrum_peaks_dialogs(Session &, Spectrum *);
static View *preferred_view(Session &, Spectrum *sp);
static Spectrum *peaks_spectrum(const List &peaks);

// ----------------------------------------------------------------------------
//
void show_peak_list_dialog(Session &s, const List &peaks,
			   const Peak_List_Options &plo)
{
  peak_list_dialog *pld = new peak_list_dialog(s, peaks, plo);

  pld->show();
}

// ----------------------------------------------------------------------------
//
peak_list_dialog::peak_list_dialog(Session &s, const List &peaks,
				   const Peak_List_Options &plo)
  : Dialog(s, "peakListDialog"), plformat(plo)
{
  this->center = true;

  ws.set_dialog_title(dialog, plo.title);

  custom = ws.create_frame(dialog, "customize");
  
  scroller = ws.create_scrolled_list(dialog, "list", &list);
  ws.extended_selection_list(list);
  ws.add_list_selection_callback(list, select_cb, this);
  ws.add_list_double_click_callback(list, goto_cb, this);
  heading = ws.scrolled_list_heading(scroller);
  ws.add_event_callback(key_press_event, scroller, key_press_cb, this);
  ws.add_event_callback(button_1_press_event, list, focus_cb, this);

  Widget search = ws.create_form(dialog, "search");
  Widget search_title = ws.create_label(search, "title");
  search_text = ws.create_text_field(search, "text");
  ws.add_text_field_changed_callback(search_text, search_cb, this);
  ws.add_event_callback(enter_pressed_event, search_text, search_next_cb, this);
  search_result = ws.create_label(search, "result");
  ws.row_attachments(search_result,
		     search_title, search_text, search_result, END_OF_WIDGETS);

  Widget separator = ws.create_separator(dialog, "separator");

  Widget controls = ws.button_row(dialog, "controls",
			       "update", update_cb, this,
			       "save", save_cb, this,
			       "options", options_cb, this,
			       "close", close_cb, this,
			       "help", help_cb, &s,
			       NULL);

  ws.column_attachments(scroller, custom, scroller, search,
		     separator, controls, END_OF_WIDGETS);

  Notifier &n = session.notifier();
  n.notify_me(nt_changed_ornaments, ornaments_changed_cb, this);
  n.notify_me(nt_selected_ornaments_changed, selection_changed_cb, this);
  n.notify_me(nt_will_delete_crosspeak, will_delete_crosspeak_cb, this);

  set_peak_list(peaks);
}

// ----------------------------------------------------------------------------
//
peak_list_dialog::~peak_list_dialog()
{
  unshow_peak_list_options_dialog(session, this);

  Notifier &n = session.notifier();
  n.dont_notify_me(nt_changed_ornaments, ornaments_changed_cb, this);
  n.dont_notify_me(nt_selected_ornaments_changed, selection_changed_cb, this);
  n.dont_notify_me(nt_will_delete_crosspeak, will_delete_crosspeak_cb, this);

  ws.remove_list_selection_callback(list, select_cb, this);
  ws.remove_list_double_click_callback(list, goto_cb, this);
  ws.remove_text_field_changed_callback(search_text, search_cb, this);
  ws.remove_event_callback(enter_pressed_event, search_text, search_next_cb,this);
  ws.remove_event_callback(key_press_event, scroller, key_press_cb, this);
  ws.remove_event_callback(button_1_press_event, list, focus_cb, this);
}

// ----------------------------------------------------------------------------
//
Widget peak_list_dialog::customize()
  { return custom; }

// ----------------------------------------------------------------------------
//
void peak_list_dialog::ornaments_changed_cb(void *pldialog, void *ornlist)
{
  peak_list_dialog *pld = (peak_list_dialog *) pldialog;

  if (pld->shown())
    {
      const List &olist = *(List *) ornlist;
      List plist;
      for (int oi = 0 ; oi < olist.size() ; ++oi)
	{
	  Ornament *op = (Ornament *) olist[oi];
	  if (is_crosspeak(op))
	    {
	      CrossPeak *xp = (CrossPeak *) op;
	      if (pld->has_peak(xp))
		plist.append(xp);
	    }
	}
      if (!plist.empty())
	if (plist.size() < pld->peaks.size() / 10)
	  {
	    //
	    // Optimization.  If less than a tenth of all peaks changed
	    // update just those peak list lines.
	    //
	    for (int pi = 0 ; pi < plist.size() ; ++pi)
	      pld->peak_changed((CrossPeak *) plist[pi]);
	  }
	else
	  pld->update();
    }
}

// ----------------------------------------------------------------------------
//
void peak_list_dialog::selection_changed_cb(void *pldialog, void *)
{
  peak_list_dialog *pld = (peak_list_dialog *) pldialog;

  if (pld->shown() && pld->center)
    {
      CrossPeak *xp = pld->session.project().single_peak_selected();
      if (xp)
	pld->show_peak(xp);
    }
  pld->center = true;
}

// ----------------------------------------------------------------------------
//
void peak_list_dialog::will_delete_crosspeak_cb(void *pldialog, void *xpeak)
{
  peak_list_dialog *pld = (peak_list_dialog *) pldialog;
  CrossPeak *xp = (CrossPeak *) xpeak;

  if (pld->shown() && pld->has_peak(xp))
    pld->delete_peak(xp);
}

// ----------------------------------------------------------------------------
//
void change_peak_list_format(peak_list_dialog *pld,
			     const Peak_List_Options &plo)
{
  pld->change_format(plo);
}

// ----------------------------------------------------------------------------
//
bool peak_list_dialog::write_file(const Stringy &path)
{
  FILE *fp = fopen(path.cstring(), "w");	// Peak list
  if (fp == NULL)
   session.reporter().message("Couldn't write %s\n", path.cstring());

  if (fp)
    {
      fprintf(fp, "%s\n\n", format().heading(max_dimension).cstring());
      for (int pi = 0 ; pi < peaks.size() ; ++pi)
	{
	  Stringy line = format().text((CrossPeak *) peaks[pi], max_dimension);
	  if (!line.is_white())
	    fprintf(fp, "%s\n", line.cstring());
	}
      fclose(fp);
    }

  return fp != NULL;
}

// ----------------------------------------------------------------------------
//
void peak_list_dialog::select_cb(Widget w, CB_Data client_data, CB_Data)
{
  peak_list_dialog *pd = (peak_list_dialog *) client_data;

  pd->session.project().unselect_all_ornaments();

  int num, *positions;
  pd->ws.selected_list_positions(w, &num, &positions);
  for (int k = 0 ; k < num ; ++k)
    {
      int pos = positions[k];
      CrossPeak *xp = (CrossPeak *) pd->peaks[pos];
      xp->select(true);
    }
  pd->ws.free_list_positions(positions);

  if (num == 1)
    pd->center = false;    // Don't scroll to center this peak.
}

// ----------------------------------------------------------------------------
// Select peak and center it in first shown view of spectrum, or if none
// shown, in the first hidden view.
//
void peak_list_dialog::goto_cb(Widget w, CB_Data client_data, CB_Data)
{
  peak_list_dialog *pd = (peak_list_dialog *) client_data;
  int pos;
  if (pd->ws.selected_list_position(w, &pos))
    {
      CrossPeak *xp = (CrossPeak *) pd->peaks[pos];
      View *view = preferred_view(pd->session, xp->spectrum());
      if (view)
	{
	  view->set_center(xp->position());
	  view->show_view(true);
	}

      pd->session.project().unselect_all_ornaments();
      xp->select(true);
      pd->center = false;
    }
}

// ----------------------------------------------------------------------------
//
void peak_list_dialog::key_press_cb(Widget, CB_Data pldialog, CB_Data event)
{
  peak_list_dialog *pld = (peak_list_dialog *) pldialog;
  pld->session.command_dispatcher().parse_key_event(event);
}

// ----------------------------------------------------------------------------
//
void peak_list_dialog::focus_cb(Widget, CB_Data pldialog, CB_Data)
{
  peak_list_dialog *pld = (peak_list_dialog *) pldialog;
  pld->ws.set_keyboard_focus_child((Widget) pld->scroller);
}

// ----------------------------------------------------------------------------
//
static View *preferred_view(Session &s, Spectrum *sp)
{
  View *view = selected_view(s);
  if (view && view->spectrum() == sp && view->is_top_level_window())
    return view;

  List vlist = s.project().view_list(sp);
  List hvlist;
  for (int vi = 0 ; vi < vlist.size() ; ++vi)
    {
      View *v = (View *) vlist[vi];
      if (v->is_top_level_window())
	if (v->shown())
	  return v;
	else
	  hvlist.append(v);
    }

  if (hvlist.size() > 0)
    return (View *) hvlist[0];

  return NULL;
}

// ----------------------------------------------------------------------------
//
void peak_list_dialog::search_cb(Widget, CB_Data client_data, CB_Data)
{
  peak_list_dialog *pd = (peak_list_dialog *) client_data;

  pd->search(false);
}

// ----------------------------------------------------------------------------
//
void peak_list_dialog::search_next_cb(Widget, CB_Data client_data, CB_Data)
{
  peak_list_dialog *pd = (peak_list_dialog *) client_data;

  pd->search(true);
}


// ----------------------------------------------------------------------------
//
void peak_list_dialog::search(bool next_occurence)
{
  ws.set_label_text(search_result, "");
  Stringy text = ws.text_field_string(search_text);
  if (text.is_empty())
    return;
  
  int line, start = ws.top_visible_list_position(list);
  if (next_occurence)
    start += 1;
  if (ws.find_list_substring(list, start, text, &line))
    ws.set_top_visible_list_position(list, line);
  else if (start > 0 && ws.find_list_substring(list, 0, text, &line))
    ws.set_top_visible_list_position(list, line);
  else
    ws.set_label_text(search_result, "not found");
}

// ----------------------------------------------------------------------------
//
void peak_list_dialog::update_cb(Widget, CB_Data client_data, CB_Data)
{
  peak_list_dialog *pd = (peak_list_dialog *) client_data;
  pd->update();
}

// ----------------------------------------------------------------------------
//
void peak_list_dialog::save_cb(Widget, CB_Data client_data, CB_Data)
{
  peak_list_dialog *pd = (peak_list_dialog *) client_data;

  Spectrum *sp = peaks_spectrum(pd->peaks);
  Stringy file = (sp ? sp->name() + ".list" : Stringy("peaks.list"));
  Stringy defpath = pd->session.project().list_path(file);
  Stringy path = pd->ws.saveas_file_dialog(pd->dialog, "Save Peak List",
					   defpath, "peaklist", true);
  if (! path.is_empty())
    pd->write_file(path);
}

// ----------------------------------------------------------------------------
//
static Spectrum *peaks_spectrum(const List &peaks)
{
  Spectrum *sp = NULL;
  for (int pi = 0 ; pi < peaks.size() ; ++pi)
    {
      Spectrum *peak_sp = ((CrossPeak *) peaks[pi])->spectrum();
      if (sp == NULL)
	sp = peak_sp;
      else if (sp != peak_sp)
	return NULL;
    }
  return sp;
}

// ----------------------------------------------------------------------------
//
void peak_list_dialog::options_cb(Widget, CB_Data pldialog, CB_Data)
{
  peak_list_dialog *pd = (peak_list_dialog *) pldialog;
  show_peak_list_options_dialog(pd->session, pd->format(), pd);
}

// ----------------------------------------------------------------------------
//
void peak_list_dialog::close_cb(Widget, CB_Data client_data, CB_Data)
{
  peak_list_dialog *pd = (peak_list_dialog *) client_data;

  if (pd == (peak_list_dialog *) resonance_peaks_dialog::the(pd->session))
    pd->ws.unshow_dialog(pd->dialog);
  else
    delete pd;
}

// ----------------------------------------------------------------------------
//
Stringy peak_list_dialog::title()
  { return format().title; }
void peak_list_dialog::set_title(const Stringy &title)
  { format().title = title; ws.set_dialog_title(dialog, title); }

// ----------------------------------------------------------------------------
//
void peak_list_dialog::show()
{
  ws.show_dialog(dialog);
  ws.raise_widget(dialog);
}

// ----------------------------------------------------------------------------
//
void peak_list_dialog::unshow()
  { ws.unshow_dialog(dialog); }

// ----------------------------------------------------------------------------
//
bool peak_list_dialog::has_peak(CrossPeak *xp)
  { return peak_table.find(xp, NULL); }

// ----------------------------------------------------------------------------
//
void peak_list_dialog::set_peak_list(const List &plist)
{   
  List sorted_plist = plist;

  format().sort(sorted_plist);

  load_list(sorted_plist);

  ws.set_label_text(heading, format().heading(max_dimension));
}

// ----------------------------------------------------------------------------
//
Peak_List_Options &peak_list_dialog::format()
{
  return plformat;
}

// ----------------------------------------------------------------------------
//
void peak_list_dialog::change_format(const Peak_List_Options &plo)
{
  plformat = plo;
  update();
}

// ----------------------------------------------------------------------------
//
void peak_list_dialog::update()
{   
  set_peak_list(peaks);
}

// ----------------------------------------------------------------------------
//
void peak_list_dialog::load_list(const List &plist)
{
  peak_table.erase();

  //  int selected = 0;
  max_dimension = 0;
  for (int pi = 0 ; pi < plist.size() ; ++pi)
    {
      CrossPeak *xp = (CrossPeak *) plist[pi];
      peak_table.insert(xp, NULL);
      max_dimension = max(xp->dimension(), max_dimension);
      //      if (xp->IsSelected())
      //	selected += 1;
    }

  List strings;
  for (int pi = 0 ; pi < plist.size() ; ++pi)
    {
      CrossPeak *xp = (CrossPeak *) plist[pi];
      Stringy *line = new Stringy(format().text(xp, max_dimension));
      strings.append(line);
    }

  peaks = plist;
  int top = ws.top_visible_list_position(list);
  ws.set_list_items(list, strings);
  free_string_list_entries(strings);

  //
  // If over 100 peaks are selected I don't show any as selected.
  // This hack is to avoid long delays because of the O(n^2) updating
  // of Motif list widgets.  30 second delay for 1000 element list.
  //
  // if (selected < 100)
  //
    for (int p = 0 ; p < plist.size() ; ++p)
      {
	CrossPeak *xp = (CrossPeak *) plist[p];
	if (xp->IsSelected())
	  ws.select_list_position(list, p, true);
      }

  CrossPeak *xp = session.project().single_peak_selected();
  if (xp)
    show_peak(xp);
  else if (top < plist.size())
    ws.set_top_visible_list_position(list, top);
}

// ----------------------------------------------------------------------------
//
void peak_list_dialog::show_peak(const CrossPeak *xp)
{
  int pos;
  if (peaks.find(xp, &pos))
    ws.center_list_position(list, pos, false);
}

// ----------------------------------------------------------------------------
//
void peak_list_dialog::peak_changed(const CrossPeak *xp)
{
  int pos;

  if (peaks.find(xp, &pos))
    {
      ws.replace_list_item(list, pos, format().text(xp, max_dimension));
      ws.select_list_position(list, pos, xp->IsSelected());
    }
}

// ----------------------------------------------------------------------------
//
void peak_list_dialog::delete_peak(CrossPeak *xp)
{
  int pi = peaks.find(xp);
  if (pi < peaks.size())
    {
      ws.delete_list_position(list, pi);
      peaks.erase(pi);
      peak_table.remove(xp);
    }
}

// ----------------------------------------------------------------------------
//
spectrum_peaks_dialog::spectrum_peaks_dialog(Session &s, Spectrum *sp,
					     const Peak_List_Options &plo) :
  peak_list_dialog(s, sp->crosspeaks(), plo)
{
  this->sp = sp;

  s.dialog_table().append_to_dialog_list("spectrum_peaks_dialogs", this);

  Notifier &n = session.notifier();
  n.notify_me(nt_will_delete_spectrum, will_delete_spectrum_cb, this);

  set_title(Stringy("Spectrum Peaks ") + sp->fullname());
}

// ----------------------------------------------------------------------------
//
spectrum_peaks_dialog::~spectrum_peaks_dialog()
{
  Notifier &n = session.notifier();
  n.dont_notify_me(nt_will_delete_spectrum, will_delete_spectrum_cb, this);

  Dialog_Table &dt = session.dialog_table();
  dt.delete_from_dialog_list("spectrum_peaks_dialogs", this);
}

// ----------------------------------------------------------------------------
//
Spectrum *spectrum_peaks_dialog::spectrum()
  { return sp; }

// ----------------------------------------------------------------------------
//
void show_peak_list_dialog(Session &s, Spectrum *sp,
			   const Peak_List_Options &plo)
{
  if (!show_spectrum_peaks_dialogs(s, sp))
    new_peak_list_dialog(s, sp, plo);
}

// ----------------------------------------------------------------------------
//
void new_peak_list_dialog(Session &s, Spectrum *spectrum,
			  const Peak_List_Options &plo)
{
  spectrum_peaks_dialog *spd = new spectrum_peaks_dialog(s, spectrum, plo);
  spd->show();
}

// ----------------------------------------------------------------------------
//
static bool show_spectrum_peaks_dialogs(Session &s, Spectrum *spectrum)
{
  bool found_dialog = false;
  List dlist = spectrum_peaks_dialog::dialog_list(s.dialog_table());
  for (int di = 0 ; di < dlist.size() ; ++di)
    {
      spectrum_peaks_dialog *spd = (spectrum_peaks_dialog *) dlist[di];
      if (spd->spectrum() == spectrum)
	{
	  found_dialog = true;
	  spd->show();
	}
    }
  return found_dialog;
}

// ----------------------------------------------------------------------------
//
List spectrum_peaks_dialog::dialog_list(Dialog_Table &dt)
{
  List *dlist = dt.dialog_list("spectrum_peaks_dialogs");
  return (dlist == NULL ? List() : *dlist);
}

// ----------------------------------------------------------------------------
//
void spectrum_peaks_dialog::will_delete_spectrum_cb(void *spdialog,
						    void *spectrum)
{
  spectrum_peaks_dialog *spd = (spectrum_peaks_dialog *) spdialog;
  Spectrum *sp = (Spectrum *) spectrum;

  if (spd->spectrum() == sp)
    delete spd;
}

// ----------------------------------------------------------------------------
//
void spectrum_peaks_dialog::change_format(const Peak_List_Options &plo)
{
  spectrum()->mPeakListOptions = plo;
  peak_list_dialog::change_format(plo);
}

// ----------------------------------------------------------------------------
//
void spectrum_peaks_dialog::update()
{
  set_peak_list(sp->crosspeaks());
}

// ----------------------------------------------------------------------------
//
resonance_peaks_dialog::resonance_peaks_dialog(Session &s, Resonance *res) :
  peak_list_dialog(s, List(), resonance_peaks_format())
{
  this->res = NULL;

  resfield = new Resonance_Field(s, customize(), "resField");
  
  ws.add_text_field_changed_callback(resfield->resonance_text_field(),
				     resonance_chosen_cb, this);
  ws.add_text_field_changed_callback(resfield->condition_text_field(),
				  resonance_chosen_cb, this);
  ws.column_attachments(resfield->frame(), resfield->frame(), END_OF_WIDGETS);

  Notifier &n = session.notifier();
  n.notify_me(nt_will_delete_resonance, will_delete_resonance_cb, this);
  n.notify_me(nt_changed_resonance, resonance_changed_cb, this);

  set_resonance(res);
}

// ----------------------------------------------------------------------------
//
resonance_peaks_dialog::~resonance_peaks_dialog()
{
  session.dialog_table().delete_dialog("resonance_peaks_dialog", this);

  ws.remove_text_field_changed_callback(resfield->resonance_text_field(),
					resonance_chosen_cb, this);
  ws.remove_text_field_changed_callback(resfield->condition_text_field(), 
					resonance_chosen_cb, this);
  Notifier &n = session.notifier();
  n.dont_notify_me(nt_will_delete_resonance, will_delete_resonance_cb, this);
  n.dont_notify_me(nt_changed_resonance, resonance_changed_cb, this);

  delete resfield;
  resfield = NULL;
}

// ----------------------------------------------------------------------------
// The default resonance_peaks_dialog instance.
//
resonance_peaks_dialog *resonance_peaks_dialog::the(Session &s)
{
  Stringy name = "resonance_peaks_dialog";
  Dialog_Table &dt = s.dialog_table();
  if (dt.get_dialog(name) == NULL)
    dt.set_dialog(name, new resonance_peaks_dialog(s, NULL));
  return (resonance_peaks_dialog *) dt.get_dialog(name);
}

// ----------------------------------------------------------------------------
//
Resonance *resonance_peaks_dialog::resonance()
  { return res; }

// ----------------------------------------------------------------------------
//
void show_peak_list_dialog(Session &s, Resonance *res)
  { resonance_peaks_dialog::the(s)->show(res); }
void show_resonance_peaks_dialog(Session &s)
  { resonance_peaks_dialog::the(s)->peak_list_dialog::show(); }

// ----------------------------------------------------------------------------
//
void update_peak_list_dialog(Session &s, const Resonance *res)
  { if (resonance_peaks_dialog::the(s)->peak_list_dialog::shown())
      resonance_peaks_dialog::the(s)->set_resonance(res); }

// ----------------------------------------------------------------------------
//
void resonance_peaks_dialog::will_delete_resonance_cb(void *rpdialog,
						      void *resonance)
{
 ((resonance_peaks_dialog *) rpdialog)->unshow((Resonance *) resonance);
}

// ----------------------------------------------------------------------------
//
void resonance_peaks_dialog::resonance_changed_cb(void *rpdialog,
						  void *resonance)
{
  resonance_peaks_dialog *rpd = (resonance_peaks_dialog *) rpdialog;
  Resonance *r = (Resonance *) resonance;

  if (rpd->shown(r))
    rpd->ws.add_work_procedure(deferred_update_cb, rpd);
}

// ----------------------------------------------------------------------------
//
void resonance_peaks_dialog::deferred_update_cb(CB_Data rpdialog)
{
  resonance_peaks_dialog *rpd = (resonance_peaks_dialog *) rpdialog;

  if (rpd->peak_list_dialog::shown())
    rpd->set_peak_list(rpd->resonance()->crosspeaks());
}

// ----------------------------------------------------------------------------
//
void resonance_peaks_dialog::resonance_chosen_cb(Widget, CB_Data rpdialog,
						 CB_Data)
{
  resonance_peaks_dialog *rpd = (resonance_peaks_dialog *) rpdialog;

  Resonance *r = rpd->resfield->resonance_chosen();
  if (rpd->res != r)
    rpd->set_resonance(r);
}

// ----------------------------------------------------------------------------
//
void resonance_peaks_dialog::show(Resonance *res)
{
  set_resonance(res);
  peak_list_dialog::show();
}

// ----------------------------------------------------------------------------
//
void resonance_peaks_dialog::set_resonance(const Resonance *res)
{
  this->res = (Resonance *) res;
  update();
}

// ----------------------------------------------------------------------------
//
void resonance_peaks_dialog::update()
{
  if (res == NULL)
    {
      set_title("Resonance Peak List");
      set_peak_list(List());
    }
  else
    {
      set_title(Stringy("Resonance Peaks ") +
		res->condition()->fullname() +
		Stringy(" / ") + Stringy(res->name()));
      resfield->set_resonance_field(res);
      if (format().selected_resonance != res->name())
	{
	  Peak_List_Options plo = format();
	  plo.selected_resonance = res->name();
	  change_format(plo);
	}
      set_peak_list(res->crosspeaks());
    }
}

// ----------------------------------------------------------------------------
//
void resonance_peaks_dialog::unshow(const Resonance *res)
{
  if (peak_list_dialog::shown() && this->res == res)
    {
      this->res = NULL;
      peak_list_dialog::unshow();
    }
}

// ----------------------------------------------------------------------------
//
void resonance_peaks_dialog::changed(const Resonance *res)
{
  if (peak_list_dialog::shown() && this->res == res)
    update_peak_list_dialog(session, res);
}

// ----------------------------------------------------------------------------
//
bool resonance_peaks_dialog::shown(const Resonance *res)
{
  return this->res == res && peak_list_dialog::shown();
}

// ----------------------------------------------------------------------------
//
Resonance *resonance_peak_list_resonance(Session &s)
{
  return resonance_peaks_dialog::the(s)->resonance();
}
