// ----------------------------------------------------------------------------
// Contour level edge panel.
//
#include <math.h>		// use pow()

#include "notifier.h"		// use Notifier
#include "num.h"		// use is_between()
#include "session.h"		// use Session
#include "spectrum.h"		// use Spectrum
#include "uicscale.h"		// use ContourScale
#include "uiview.h"		// use View
#include "winsystem.h"		// use WinSys

// ----------------------------------------------------------------------------
//
static bool adjust_levels(double low, double high, int n, Contour_Levels *lev);

// ----------------------------------------------------------------------------
//
ContourScale::ContourScale(View *view, double a, double b, int screen_length)
  : edge_panel(view->session().window_system(), a, b, screen_length, false,
	       view->frame(), "contourScale", false)
{
  this->view = view;
  this->scale = 1;
  this->showing_current_level = false;

  ws.add_event_callback(button_1_press_event, widget(), button_press_cb, this);
  ws.add_event_callback(pointer_drag_event, widget(), pointer_drag_cb, this);

  Notifier &n = view->session().notifier();
  n.notify_me(nt_changed_view_contours, contours_changed_cb, this);
}

// ----------------------------------------------------------------------------
//
ContourScale::~ContourScale()
{
  ws.remove_event_callback(button_1_press_event, widget(), button_press_cb, this);
  ws.remove_event_callback(pointer_drag_event, widget(), pointer_drag_cb, this);

  Notifier &n = view->session().notifier();
  n.dont_notify_me(nt_changed_view_contours, contours_changed_cb, this);
}

// ----------------------------------------------------------------------------
//
void ContourScale::contours_changed_cb(void *cscale, void *view)
{
  View *v = (View *) view;
  ContourScale *cs = (ContourScale *) cscale;
  if (cs->view == v)
    cs->erase();
}

// ----------------------------------------------------------------------------
//
void ContourScale::button_press_cb(Widget, CB_Data cscale, CB_Data event)
{
  ContourScale *cs = (ContourScale *) cscale;
  int x, y;
  if (cs->ws.event_xy(event, &x, &y))
    cs->process_button(y);
}

// ----------------------------------------------------------------------------
//
void ContourScale::pointer_drag_cb(Widget, CB_Data cscale, CB_Data event)
{
  ContourScale *cs = (ContourScale *) cscale;
  int x, y;
  if (cs->ws.event_xy(event, &x, &y))
    cs->process_drag(y);
}

// ----------------------------------------------------------------------------
//
void ContourScale::process_button(int sy)
{
  this->last_y = sy;

  double y = draw_position(sy);
  double ymin, ymax;
  range(&ymin, &ymax);
  double y_middle = .5 * (ymin + ymax);

  this->drag_sign = (y <= y_middle ? 1 : -1);

  Contour_Levels lev = this->view->contour_parameters(drag_sign).levels;
  if (lev.levels == 0)
    this->drag_mode = AdjustLevels;
  else
    {
      double y_high = height_y_position(lev.highest(), drag_sign >= 0);
      double y_low = height_y_position(lev.lowest, drag_sign >= 0);
      if (drag_sign >= 0 ? y < y_high : y > y_high)
	this->drag_mode = AdjustHigh;
      else if (drag_sign >= 0 ? y > y_low : y < y_low)
	this->drag_mode = AdjustLow;
      else
	this->drag_mode = AdjustLevels;
    }
}

// ----------------------------------------------------------------------------
//
#define PIXELS_PER_LEVEL_STEP 3

void ContourScale::process_drag(int sy)
{
  int steps = ((drag_sign >= 0 || drag_mode == AdjustLevels) ?
	       last_y - sy : sy - last_y) / PIXELS_PER_LEVEL_STEP;
  if (steps != 0)
    {
      Contour_Parameters cp = view->contour_parameters(drag_sign);
      Contour_Levels &lev = cp.levels;
      double low = lev.lowest;
      double high = lev.highest();
      int n = lev.levels;
      if (drag_mode == AdjustHigh)
	{
	  n += steps;
	  high *= pow(lev.factor, steps);
	  if (n < 1)
	    {
	      n = 1;
	      low = high;
	    }
	}
      else if (drag_mode == AdjustLow)
	{
	  n -= steps;
	  low *= pow(lev.factor, steps);
	  if (n < 1)
	    n = 1;
	}
      else if (drag_mode == AdjustLevels)
	{
	  if (n < 2 && n + steps >= 2)
	    high = low * lev.factor;
	  n += steps;
	}

      if (adjust_levels(low, high, n, &lev))
	{
	  view->set_contour_parameters(cp);
	  last_y = sy;
	}
    }
}

// ----------------------------------------------------------------------------
//
#define CENTER_SPACER_SIZE .05		// Fraction of total height

double ContourScale::height_y_position(double h, bool positive_half)
{
  double ymin, ymax;
  range(&ymin, &ymax);
  double ymiddle = (ymin + ymax) / 2;
  double yspacer = CENTER_SPACER_SIZE * (ymax - ymin);
  double ybase = ymiddle + (positive_half ? -yspacer/2 : yspacer/2);
  double y = ybase - h / scale;
  return y;
}

// ----------------------------------------------------------------------------
//
double ContourScale::y_range()
{
  double ymin, ymax;
  range(&ymin, &ymax);
  double yrange = .5 * (1 - CENTER_SPACER_SIZE) * (ymax - ymin);
  return yrange;
}

// ----------------------------------------------------------------------------
//
static bool adjust_levels(double low, double high, int n, Contour_Levels *lev)
{
  if (n < 0)
    n = 0;

  if (n >= 2)
    if ((low >= 0 && high <= low) || (low < 0 && high > low))
      return false;

  lev->lowest = low;
  lev->levels = n;
  if (n > 1)
    lev->factor = pow(high / low, 1.0 / (n - 1));

  return true;
}

// ----------------------------------------------------------------------------
//
void ContourScale::redraw(double, double)
{
  const Contour_Parameters &pos = view->positive_contours();
  const Contour_Parameters &neg = view->negative_contours();

  adjust_scale(pos, neg);

  draw_levels(pos, neg);

  if (showing_current_level)
    draw_current_level();
}

// ----------------------------------------------------------------------------
//
void ContourScale::draw_levels(const Contour_Parameters &pos,
			       const Contour_Parameters &neg)
{
  double xwidth = thickness();

  const Contour_Levels &positive = pos.levels;
  double h = positive.lowest;
  for (int level = 0 ; level < positive.levels ; ++level)
    {
      set_drawing_color(pos.level_color(h));
      double y = height_y_position(h, true);
      draw_line(0, y, xwidth, y);
      h *= positive.factor;
    }

  const Contour_Levels &negative = neg.levels;
  h = negative.lowest;
  for (int level = 0 ; level < negative.levels ; ++level)
    {
      set_drawing_color(neg.level_color(h));
      double y = height_y_position(h, false);
      draw_line(0, y, xwidth, y);
      h *= negative.factor;
    }

  set_drawing_color("gray");
  double ypos0 = height_y_position(0, true);
  double yneg0 = height_y_position(0, false);
  fill_rectangle(0, ypos0, xwidth, yneg0);		// Spacer
}

// ----------------------------------------------------------------------------
//
void ContourScale::draw_current_level()
{
  double xwidth = thickness();
  double y = height_y_position(current_level, current_level >= 0);

  set_drawing_color("white");
  draw_line(0, y, xwidth, y, true);
}

// ----------------------------------------------------------------------------
//
void ContourScale::adjust_scale(const Contour_Parameters &pos,
				const Contour_Parameters &neg)
{
  double rescale_high_threshold = .9;
  double rescale_low_threshold = .5;

  double yrange = y_range();
  double hrange = yrange * scale;

  double hmax = 0;
  if (pos.levels.levels > 0)
    hmax = max(hmax, pos.levels.highest());
  if (neg.levels.levels > 0)
    hmax = max(hmax, -neg.levels.highest());

  if (hmax == 0)
    scale = 1;
  else if (hmax > rescale_high_threshold * hrange)
    scale = hmax / (rescale_high_threshold * yrange);
  else if (hmax < rescale_low_threshold * hrange)
    scale = hmax / (rescale_low_threshold * yrange);
}

// ----------------------------------------------------------------------------
//
void ContourScale::show_pointer_level()
{
  if (showing_current_level)
    {
      showing_current_level = false;
      draw_current_level();
    }

  SPoint p;
  if (view->pointer_position(&p))
    {
      current_level = view->spectrum()->height_at_point(p);
      draw_current_level();
      showing_current_level = true;
    }
}

// ----------------------------------------------------------------------------
//
void ContourScale::pointer_in(double, double)
  { show_pointer_level(); }
void ContourScale::pointer_move(double, double)
  { show_pointer_level(); }
void ContourScale::pointer_out(double, double)
  { show_pointer_level(); }
