// ----------------------------------------------------------------------------
//
#include <stdio.h>		// Use sprintf()
#include <math.h>		// Use fabs()

#include "assignguess.h"	// Use Assignment_Guess
#include "atom.h"
#include "condition.h"		// Use Condition
#include "crosspeak.h"
#include "group.h"
#include "label.h"		// Use Label
#include "list.h"		// Use List
#include "memalloc.h"		// use new()
#include "notifier.h"		// use Notifier
#include "project.h"		// use Project
#include "reporter.h"		// use Reporter
#include "resonance.h"		// Use Resonance
#include "session.h"		// use Session
#include "spectrum.h"
#include "table.h"		// use Table
#include "uicomponents.h"	// use table_axis_label()
#include "uidialogs.h"		// Use show_guess_dialog(), help_cb()
#include "uidialog.h"		// use Dialog, Dialog_Table
#include "uipeak.h"		// Use show_peak_list_dialog()
#include "winsystem.h"		// use WinSys

static void deferred_resonance_update(CB_Data adialog);
static int closest_axis(CrossPeak *xp, double freq);
static Stringy assignment_name(Assignment_Guess *ag, int dim);
static Stringy resonance_text(Resonance *r);
static int resonance_compare(const void *r1, const void *r2);

// ----------------------------------------------------------------------------
//
class assignment_dialog : public Dialog
{
public:
  assignment_dialog(Session &);
  ~assignment_dialog();

  static assignment_dialog *the(Session &);

  void show(CrossPeak *xp);
  void update(CrossPeak *xp);
  void update_guesses();
  void update_resonance_list();

private:
  Widget table, rlist, guess_menu;
  List guesses;
  CrossPeak *peak;	// Can be NULL.
  Spectrum *spectrum;	// Can be NULL.
  int dimension;

  static void will_delete_spectrum_cb(void *adialog, void *spectrum);
  static void will_delete_crosspeak_cb(void *adialog, void *xpeak);
  static void ornament_selection_cb(void *adialog, void *);
  static void resonance_list_changed_cb(void *adialog, void *);
  static void axis_select_cb(Widget, CB_Data, CB_Data);
  static void update_resonance_list_cb(Widget, CB_Data, CB_Data);
  static void fill_in_resonance_cb(Widget, CB_Data, CB_Data);
  static void guess_options_cb(Widget, CB_Data, CB_Data);
  static void guess_selected_cb(Widget, CB_Data, CB_Data);
  static void resonance_peaks_cb(Widget, CB_Data, CB_Data);

  Widget axis_name_button(int axis);
  Widget axis_type_label(int axis);
  Widget group_text(int axis);
  Widget atom_text(int axis);
  Widget resonance_label(int axis);
  Widget frequency_label(int axis);

  void show_assignment(int axis, Resonance *r);
  Resonance *list_resonance(const Stringy &text);
  void highlight_closest_resonance(int axis);
  void use_guess(const Stringy &guess);
  void apply();
};

// ----------------------------------------------------------------------------
//
void show_assignment_dialog(Session &s, CrossPeak *xp)
  { assignment_dialog::the(s)->show(xp); }
void update_assignment_dialog_guesses(Session &s)
  { if (assignment_dialog::the(s)->shown())
      assignment_dialog::the(s)->update_guesses(); }

// ----------------------------------------------------------------------------
//
assignment_dialog::assignment_dialog(Session &s) : Dialog(s, "assignDialog")
{
  this->peak = NULL;
  this->spectrum = NULL;

  Table_Entry label = TABLE_LABEL;
  Table_Entry value = TABLE_TEXT_FIELD;
  Table_Entry button = TABLE_TOGGLE_BUTTON;
  table = ws.widget_table(dialog, "axes", 5, 5,
			  label, label, label, label, label,
			  button, value, value, label, label,
			  button, value, value, label, label,
			  button, value, value, label, label,
			  button, value, value, label, label);

  for (int a = 0 ; a < DIM ; ++a)
    ws.add_toggle_button_callback(axis_name_button(a), axis_select_cb, this);

  Widget scroller = ws.create_scrolled_list(dialog, "resList", &rlist);
  ws.add_list_selection_callback(rlist, update_resonance_list_cb, this);
  ws.add_list_double_click_callback(rlist, fill_in_resonance_cb, this);

  Widget res_peaks = ws.push_button(dialog, "resPeaks",
				    resonance_peaks_cb, this);
  Widget guess = ws.create_form(dialog, "guess");
  Widget guess_options = ws.push_button(guess, "guessOpt",
					guess_options_cb, this);
  guess_menu = ws.create_menu(guess, "guesses", "guessPane");
  ws.row_attachments(NULL, guess_options, guess_menu, END_OF_WIDGETS);

  Widget separator = ws.create_separator(dialog, "separator");

  Widget controls = ws.button_row(dialog, "controls",
				  "ok", ok_cb, this,
				  "apply", apply_cb, this,
				  "close", close_cb, this,
				  "help", help_cb, &s,
				  NULL);

  ws.column_attachments(scroller, table, scroller, res_peaks, guess,
			separator, controls, END_OF_WIDGETS);

  Notifier &n = session.notifier();
  n.notify_me(nt_will_delete_spectrum, will_delete_spectrum_cb, this);
  n.notify_me(nt_will_delete_crosspeak, will_delete_crosspeak_cb, this);
  n.notify_me(nt_selected_ornaments_changed, ornament_selection_cb, this);
  n.notify_me(nt_added_resonance_to_condition,
	      resonance_list_changed_cb, this);
  n.notify_me(nt_removed_resonance_from_condition,
	      resonance_list_changed_cb, this);
  n.notify_me(nt_changed_resonance, resonance_list_changed_cb, this);
}

// ----------------------------------------------------------------------------
//
assignment_dialog::~assignment_dialog()
{
  session.dialog_table().delete_dialog("assignment_dialog", this);

  Notifier &n = session.notifier();
  n.dont_notify_me(nt_will_delete_spectrum, will_delete_spectrum_cb, this);
  n.dont_notify_me(nt_will_delete_crosspeak, will_delete_crosspeak_cb, this);
  n.dont_notify_me(nt_selected_ornaments_changed, ornament_selection_cb, this);
  n.dont_notify_me(nt_added_resonance_to_condition,
		   resonance_list_changed_cb, this);
  n.dont_notify_me(nt_removed_resonance_from_condition,
		   resonance_list_changed_cb, this);
  n.dont_notify_me(nt_changed_resonance, resonance_list_changed_cb, this);

  free_assignment_guesses(guesses);
}

// ----------------------------------------------------------------------------
// The default assignment_dialog instance.
//
assignment_dialog *assignment_dialog::the(Session &s)
{
  Stringy name = "assignment_dialog";
  Dialog_Table &dt = s.dialog_table();
  if (dt.get_dialog(name) == NULL)
    dt.set_dialog(name, new assignment_dialog(s));
  return (assignment_dialog *) dt.get_dialog(name);
}

// ----------------------------------------------------------------------------
//
void assignment_dialog::will_delete_spectrum_cb(void *adialog, void *spectrum)
{
  assignment_dialog *ad = (assignment_dialog *) adialog;
  Spectrum *sp = (Spectrum *) spectrum;

  if (ad->spectrum == sp)
    ad->update(NULL);
}

// ----------------------------------------------------------------------------
//
void assignment_dialog::will_delete_crosspeak_cb(void *adialog, void *xpeak)
{
  assignment_dialog *ad = (assignment_dialog *) adialog;
  CrossPeak *xp = (CrossPeak *) xpeak;

  if (ad->peak == xp)
    ad->update(NULL);
}

// ----------------------------------------------------------------------------
//
void assignment_dialog::ornament_selection_cb(void *adialog, void *)
{
  assignment_dialog *ad = (assignment_dialog *) adialog;

  if (ad->shown())
    {
      CrossPeak *xp = ad->session.project().single_peak_selected();
      if (xp)
	ad->update(xp);
    }
}

// ----------------------------------------------------------------------------
//
void assignment_dialog::resonance_list_changed_cb(void *adialog, void *)
{
  assignment_dialog *ad = (assignment_dialog *) adialog;

  if (ad->shown())
    ad->ws.add_work_procedure(deferred_resonance_update, adialog);
}

// ----------------------------------------------------------------------------
//
static void deferred_resonance_update(CB_Data adialog)
{
  assignment_dialog *ad = (assignment_dialog *) adialog;

  if (ad->shown())
    {
      ad->update_resonance_list();
      ad->update_guesses();
    }
}

// ----------------------------------------------------------------------------
//
void assignment_dialog::axis_select_cb(Widget w,
				       CB_Data client_data,
				       CB_Data)
{
  assignment_dialog *ad = (assignment_dialog *) client_data;

  for (int a = 0 ; a < DIM ; ++a)
    {
      bool selected = (w == ad->axis_name_button(a));
      ad->ws.set_toggle_button(ad->axis_name_button(a), selected, false);
      if (selected)
	ad->highlight_closest_resonance(a);
    }
}

// ----------------------------------------------------------------------------
//
void assignment_dialog::update_resonance_list_cb(Widget w,
						 CB_Data client_data,
						 CB_Data)
{
  assignment_dialog *ad = (assignment_dialog *) client_data;
  Stringy stext = ad->ws.selected_list_item(w);

  Resonance *r = ad->list_resonance(stext);
  if (r)
    update_peak_list_dialog(ad->session, r);
  else
    ad->session.reporter().message("Resonance list out of date.\n");
}

// ----------------------------------------------------------------------------
//
void assignment_dialog::fill_in_resonance_cb(Widget w,
					     CB_Data client_data,
					     CB_Data)
{
  assignment_dialog *ad = (assignment_dialog *) client_data;

  if (ad->peak == NULL)
    return;

  Stringy stext = ad->ws.selected_list_item(w);
  Resonance *r = ad->list_resonance(stext);
  if (r)
    {
      int axis;
      for (axis = 0 ; axis < ad->dimension ; ++axis)
	if (ad->ws.toggle_button_state(ad->axis_name_button(axis)))
	  break;
      if (axis == ad->dimension)
	axis = closest_axis(ad->peak, r->frequency());
		  
      ad->show_assignment(axis, r);
    }
  else
    ad->session.reporter().message("Resonance list out of date.\n");
}

// ----------------------------------------------------------------------------
//
static int closest_axis(CrossPeak *xp, double freq_ppm)
{
  int min_axis = 0;
  double min_sep = fabs(xp->frequency(min_axis) - freq_ppm);
  for (int axis = 1 ; axis < xp->dimension() ; ++axis)
    {
      double sep = fabs(xp->frequency(axis) - freq_ppm);
      if (sep < min_sep)
	{ min_sep = sep; min_axis = axis; }
    }

  return min_axis;
}

// ----------------------------------------------------------------------------
//
void assignment_dialog::guess_options_cb(Widget,
					 CB_Data client_data, CB_Data)
{
  assignment_dialog *ad = (assignment_dialog *) client_data;

  if (ad->spectrum)
    show_guess_dialog(ad->session, ad->spectrum);
}

// ----------------------------------------------------------------------------
//
void assignment_dialog::guess_selected_cb(Widget w,
					  CB_Data client_data,
					  CB_Data call_data)
{
  assignment_dialog *ad = (assignment_dialog *) client_data;

  Stringy guess = ad->ws.selected_menu_button(w, call_data);
  ad->use_guess(guess);
}

// ----------------------------------------------------------------------------
//
void assignment_dialog::resonance_peaks_cb(Widget,
					   CB_Data client_data,
					   CB_Data)
{
  assignment_dialog *ad = (assignment_dialog *) client_data;

  Stringy text = ad->ws.selected_list_item(ad->rlist);
  if (!text.is_empty())
    {
      Resonance *r = ad->list_resonance(text);
      if (r)
	show_peak_list_dialog(ad->session, r);
    }
  else
    ad->session.reporter().message("No resonance selected.\n");
}

// ----------------------------------------------------------------------------
//
Resonance *assignment_dialog::list_resonance(const Stringy &text)
{
  if (spectrum)
    {
      const List &resonances = spectrum->condition()->resonance_list();
      for (int ri = 0 ; ri < resonances.size() ; ++ri)
	{
	  Resonance *r = (Resonance *) resonances[ri];
	  if (resonance_text(r) == text)
	    return r;
	}
    }

  return NULL;
}

// ----------------------------------------------------------------------------
//
void assignment_dialog::highlight_closest_resonance(int axis)
{
  if (peak == NULL)
    return;

  double freq_ppm = peak->frequency(axis);
  const List &resonances = spectrum->condition()->resonance_list();
  Resonance *r = closest_resonance(resonances, freq_ppm);
  if (r)
    ws.center_list_item(rlist, resonance_text(r), true);
}

// ----------------------------------------------------------------------------
//
Widget assignment_dialog::axis_name_button(int axis)
  { return ws.table_element(table, axis+1, 0); }
Widget assignment_dialog::group_text(int axis)
  { return ws.table_element(table, axis+1, 1); }
Widget assignment_dialog::atom_text(int axis)
  { return ws.table_element(table, axis+1, 2); }
Widget assignment_dialog::resonance_label(int axis)
  { return ws.table_element(table, axis+1, 3); }
Widget assignment_dialog::frequency_label(int axis)
  { return ws.table_element(table, axis+1, 4); }

// ----------------------------------------------------------------------------
//
void assignment_dialog::show(CrossPeak *xp)
{
  update(xp);
  ws.show_dialog(dialog);
  ws.raise_widget(dialog);
}

// ----------------------------------------------------------------------------
//
void assignment_dialog::update(CrossPeak *xp)
{
  bool same_peak = (xp == peak);

  this->peak = xp;
  this->spectrum = (peak ? peak->spectrum() : (Spectrum *)NULL);
  this->dimension = (spectrum ? spectrum->dimension() : 2);

  Stringy dialog_title = Stringy("Assignment");
  if (spectrum)
    dialog_title << Stringy(" ") << spectrum->name();
  ws.set_dialog_title(dialog, dialog_title);

  for (int a = 0 ; a < dimension ; ++a)
    {
      ws.set_toggle_button_text(axis_name_button(a),
				table_axis_label(spectrum, a, ""));

      Resonance *r = (peak ? peak->resonance(a) : (Resonance *)NULL);
      show_assignment(a, r);

      Stringy freq;
      if (peak)
	freq = formatted_string("%.4g", peak->frequency(a));
      ws.set_label_text(frequency_label(a), freq);
    }
  ws.manage_table_children(table, dimension + 1, 5);

  update_resonance_list();
  if (spectrum)
    update_guess_dialog(session, spectrum);
  update_guesses();

  if (peak && !same_peak)
    {
      int axis;
      for (axis = 0 ; axis < dimension ; ++axis)
	if (peak->resonance(axis) == NULL)
	  break;
      if (axis == dimension)
	axis = 0;
      ws.set_toggle_button(axis_name_button(axis), false, false);
      ws.set_toggle_button(axis_name_button(axis), true, true);
    }
}

// ----------------------------------------------------------------------------
//
#define MAX_GUESSES 100
void assignment_dialog::update_guesses()
{
  Widget guess_list = ws.menu_pane(guess_menu);
  ws.delete_menu_buttons(guess_list);
  free_assignment_guesses(guesses);
  guesses.erase();

  if (peak)
    {
      const Guess_Assignments &ga = peak->spectrum()->GetAssignGuessOptions();
      guesses = ga.guesses(peak);
      for (int g = 0 ; g < guesses.size() && g < MAX_GUESSES ; ++g)
	{
	  Assignment_Guess *ag = (Assignment_Guess *) guesses[g];
	  Stringy aname = assignment_name(ag, dimension);
	  ws.add_menu_button(guess_list, aname, guess_selected_cb, this);
	}
    }

  ws.set_sensitive(guess_menu, guesses.size() > 0);
}

// ----------------------------------------------------------------------------
//
static Stringy assignment_name(Assignment_Guess *ag, int dim)
{
  Stringy name;

  for (int axis = 0 ; axis < dim ; ++axis)
    {
      Resonance *res = ag->resonance(axis);
      if (axis > 0)
	name << "-";
      name << (res ? res->name() : Stringy("?"));
    }

  return name;
}

// ----------------------------------------------------------------------------
//
void assignment_dialog::update_resonance_list()
{
  List strings;
  if (spectrum)
    {
      List resonances = spectrum->condition()->resonance_list();
      resonances.sort(resonance_compare);
      for (int ri = 0 ; ri < resonances.size() ; ++ri)
	strings.append(new Stringy(resonance_text((Resonance *) resonances[ri])));
    }

  //
  // Preserve the current top visible line number.
  //
  int top = ws.top_visible_list_position(rlist);
  ws.set_list_items(rlist, strings);
  free_string_list_entries(strings);
  ws.set_top_visible_list_position(rlist, top);
}

// ----------------------------------------------------------------------------
//
static Stringy resonance_text(Resonance *r)
{
  char cs[256];

  sprintf(cs, "%11s %5s %9.4g",
	  r->group()->name().cstring(),
	  r->atom()->name().cstring(),
	  r->frequency());

  return Stringy(cs);
}

// ----------------------------------------------------------------------------
//
void assignment_dialog::use_guess(const Stringy &guess)
{
  Assignment_Guess *ag = NULL;
  for (int g = 0 ; g < guesses.size() ; ++g)
    {
      ag = (Assignment_Guess *) guesses[g];
      if (assignment_name(ag, dimension) == guess)
	break;
    }

  if (ag != NULL)
    for (int axis = 0 ; axis < dimension ; ++axis)
      {
	Resonance *res = ag->resonance(axis);
	if (res)
	  show_assignment(axis, res);
      }
}

// ----------------------------------------------------------------------------
//
void assignment_dialog::show_assignment(int axis, Resonance *r)
{
  ws.set_text_field(group_text(axis), (r ? r->group()->name() : Stringy("")));
  ws.set_text_field(atom_text(axis), (r ? r->atom()->name() : Stringy("")));
  ws.set_label_text(resonance_label(axis),
		 (r ? formatted_string("%.4g", r->frequency()) : Stringy("")));
}

// ----------------------------------------------------------------------------
//
void assignment_dialog::apply()
{
  if (peak == NULL)
    return;

  for (int axis = 0 ; axis < dimension ; ++axis)
    {
      Stringy group_name = ws.text_field_string(group_text(axis));
      Stringy atom_name = ws.text_field_string(atom_text(axis));
      peak->ChangeAssignment(axis, atom_name, group_name);
      if (peak->label() == NULL)
	(void) new Label(peak);
    }

  update(peak);
}

// ----------------------------------------------------------------------------
//
static int resonance_compare(const void *r1, const void *r2)
{
  Resonance *res1 = (Resonance *) r1;
  Resonance *res2 = (Resonance *) r2;

  if (res1->frequency() < res2->frequency())
    return -1;
  else if (res1->frequency() > res2->frequency())
    return 1;

  return 0;
}
