// ----------------------------------------------------------------------------
// Spectrum class manages NMR data and peak markers, peak labels, and lines.
// Might be better to rename this PeakSet and rename the SpectrumData class
// to Spectrum.
//
#ifndef SPECTRUM_HEADER_INCLUDED
#define SPECTRUM_HEADER_INCLUDED

class Spectrum;

#include "attacheddata.h"	// use AttachedData
#include "assignguess.h"	// Use Assignment_Guessing_Options
#include "integrate.h"		// Use Integration_Parameters
#include "list.h"		// Use List
#include "ornament.h"		// Use Ornament_Type
#include "peakpick.h"		// Use Peak_Pick_Parameters
#include "savefile.h"
#include "spectrumdata.h"	// use SpectrumData
#include "spoint.h"		// Use SPoint
#include "stringc.h"		// use Stringy
#include "uiplop.h"		// Use Peak_List_Options

class Condition;
class CrossPeak;
class Molecule;
class NMR_Data;
class Notifier;
class Peak;
class PeakGp;
class Session;

// ----------------------------------------------------------------------------
//
class Spectrum : public SpectrumData
{
public:
  Spectrum(Session &, const Stringy &spectrum_name, Condition *c,
	   const Stringy &save_path, NMR_Data *nmr_data);
  virtual		~Spectrum();

  Stringy		name() const;
  void			set_name(const Stringy &name);
  Stringy		fullname() const;
  Session &		session();
  Notifier &		notifier();

  void			set_ppm_zero(const SPoint &zero_position);

  //
  // Save and restore
  //

  SaveFile &	save_file();
  void		set_save_file(const SaveFile &sf);

  //
  // Ornaments
  //

  void		AddOrnament(Ornament *);
  void		RemoveOrnament(Ornament *);
  const List &	ornaments() const;
  List		ornaments(Ornament_Type t) const;
  List		selected_ornaments() const;
  Ornament *	nth_ornament(Ornament_Type t, int n) const;
  void		unselect_ornaments();

  double		ornament_size(Ornament_Type type) const; // PPM for axis 0
  void		set_ornament_size(Ornament_Type type, double size);
  double		select_size() const;
  double		pointer_size() const;
  double		line_end_size() const;

  //
  // CrossPeaks
  //

  void		add(CrossPeak &p);	// add a crosspeak
  void		remove(PeakGp &p);	// remove a Peak Group
  void		remove(Peak &p);	// remove a Peak

  CrossPeak *	find_assigned_peak(const List &resonances);
  CrossPeak *	find_assigned_peak(const Stringy &assignment);

  // Here crosspeaks means peaks and peakgroups but not child peaks
  // of peakgroups.

  const List &	crosspeaks() const;
  List		selected_crosspeaks() const;
  List		nearby_crosspeaks(const SPoint &p) const;

  List		peaklets() const;
  List		selected_peaklets() const;

  //
  // Molecule and Condition
  //

  Molecule *	molecule() const;
  Condition *	condition() const;
  void		set_condition(Condition *c);

  //
  // Assignments
  //
  const Guess_Assignments &GetAssignGuessOptions() const;
  void		SetAssignGuessOptions(const Guess_Assignments &);

  //
  // Get/Set the default formats for assignment labels
  //
  Stringy		GetDefaultAssignmentFormat();
  Stringy		assignment_format() const;
  void		set_assignment_format(const Stringy &format);

  SPoint		assignment_copy_threshold() const;
  void		set_assignment_copy_threshold(const SPoint &thresh);

  //
  // Extra data attached to spectra and saved in files
  //
  AttachedData &saved_values();

  Integration_Parameters mIntegrate;
  Peak_Pick_Parameters mPeakPick;	/* peak-pick parameters */

  //
  // Public access to this stuff is used just for reading
  // and writing spectrum files.
  //
  Peak_List_Options mPeakListOptions;
  float		mSelectsize;	/* size of select region (ornaments) */
  float		mPointersize;	/* basewidth of label pointer */
  float		mLineendsize;	/* size of line ends (ornaments) */
  double		mSizes[ORNAMENT_TYPE_COUNT];

private:
  Session &	ses;
  Stringy	mName;
  Condition	*mCondition;		// link back to condition
  List		mPeaks;			// set of crosspeaks
  List		mOrnaments;		// Peaks, labels, etc...

  Guess_Assignments	mAssignGuessOptions;	// Assignment Tool options
  Stringy		mAssignFormat;		// Assignment label format
  SPoint		mAssignCopyThreshold;

  AttachedData	attached_data;		// data attached by extensions

  SaveFile	mSaveFile;		// Save File object
};

#endif
