/*
 * Resonance.C:	 Implementation of the Resonance class.
 *
 * In NMR spectra, Atoms resonate at frequencies characteristic of the Atom's
 * themselves and their magnetic and chemical environment. CrossPeaks appear
 * in the spectrum where magnetic energy gets transferred into one Atom
 * then out another Atom.
 *
 * A Resonance object is used to hold the resonant frequency for one Atom. Each
 * Resonance is stored in the Spectrum's Condition and is identified with
 * exactly one Atom, which gives the Resonance its identity (name, id).
 *
 * Resonances link CrossPeak to Atoms through the Condition. That is, when an
 * assigned CrossPeak is linked to at most one Resonance along each of the
 * Spectrum's frequency axes.
 *
 * Initially, the true name of the Atoms may not be known, but as the
 * assignment process proceeds, their identifies become known. Because the
 * CrossPeak links dynamically to the the Atoms through Resonances, when any
 * Atom is re-identified each CrossPeak's assignment is updated.
 *
 */
#include <math.h>		// Use fabs()
#include <ctype.h>		// Use isspace(), isalpha(), isdigit()

#include "atom.h"
#include "condition.h"
#include "group.h"
#include "molecule.h"		// use Molecule
#include "num.h"		// use interval_mod()
#include "resonance.h"
#include "peak.h"
#include "peakgp.h"
#include "session.h"		// use Session
#include "spectrum.h"		// Use Spectrum
#include "spectrumdata.h"	// Use Units
#include "notifier.h"		// use Notifier

static int assignment_count(const CrossPeak *xp, const Resonance *r);

/*
 * Construct the Resonance
 */
Resonance::Resonance(Condition *c, Atom *ap)
{
	mCondition = c;				// set the condition
	freq_ppm = 0;		// frequency if no peaks.
	freq_valid = false;			// frequency must be calculated
	mAtom = ap;				// set the Atom
	c->add_resonance(this);

	Notifier &n = c->molecule()->session().notifier();
	n.send_notice(nt_created_resonance, this);
}

/*
 * Destruct the Resonance
 */
Resonance::~Resonance()
{
  	Notifier &n = condition()->molecule()->session().notifier();
	n.send_notice(nt_will_delete_resonance, this);

	List plist = crosspeaks();
	for (int xp = 0 ; xp < plist.size() ; ++xp)
	  {
	    CrossPeak *x = (CrossPeak *) plist[xp];
	    for (int axis = 0 ; axis < x->dimension() ; ++axis)
	      if (x->resonance(axis) == this)
		x->SetResonance(axis, NULL);
	  }

	condition()->remove_resonance(this);
	mAtom = NULL;
	mCondition = NULL;
}

// ----------------------------------------------------------------------------
// Get the Resonances's Condition
//
Condition *Resonance::condition() const
  { return mCondition; }

// ----------------------------------------------------------------------------
// Get/Set a Resonance's Atom
//
Atom *Resonance::atom() const
  { return mAtom; }

// ----------------------------------------------------------------------------
// Get the Resonance Group, which is the Atom Group
//
Group *Resonance::group() const
  { return mAtom ? mAtom->group() : NULL; }

/*
 * Make an assignment between this Resonance and CrossPeak <x>, along
 * frequency axis <a>.
 */
void Resonance::add_assignment(CrossPeak *x)
{
  if (!mCrossPeaks.contains(x))
    mCrossPeaks.append(x);

  InvalidateFrequency();
}

/*
 * Remove the assignment between this Resonance and CrossPeak <x>, along
 * frequency axis <a>.
 */
void Resonance::remove_assignment(CrossPeak *x)
{
  if (::assignment_count(x, this) == 0)
    mCrossPeaks.erase(x);

  InvalidateFrequency();
}

// ----------------------------------------------------------------------------
//
static int assignment_count(const CrossPeak *xp, const Resonance *r)
{
  int count = 0;

  for (int axis = 0 ; axis < xp->dimension() ; ++axis)
    if (xp->resonance(axis) == r)
      count += 1;

  return count;
}

// ----------------------------------------------------------------------------
// Return the list of peak links.
//
const List &Resonance::crosspeaks() const
  { return mCrossPeaks; }

/*
*/
int Resonance::assignment_count() const
{
	int cnt = 0;

	const List &plist = crosspeaks();
	for (int pi = 0 ; pi < plist.size() ; ++pi)
	  cnt += ::assignment_count((CrossPeak *) plist[pi], this);

	return cnt;
}

/*
 * This non-const function returns the cached frequency value if it is valid.
 * Otherwise it calls mValidateCache() function to calculate the frequency and
 * validate the cache.
 */
double Resonance::frequency()
{
	if (! freq_valid)
		mValidateCache();
	return freq_ppm;
}

/*
 * Compute the resonance frequency and cache the value.
 */
void Resonance::mValidateCache()
{
	/*
	 * Otherwise compute the frequency.
	 */
	int		cnt = 0;
	double		sum = 0.0;

	/*
	 * Add any peak contributions
	 */
	for (int xp = 0 ; xp < crosspeaks().size() ; ++xp) {
	  CrossPeak *x = (CrossPeak *) crosspeaks()[xp];
	  int dim = x->dimension();
	  for (int axis = 0 ; axis < dim ; ++axis)
	    if (x->resonance(axis) == this)
	      {
		sum += x->frequency(axis);
		cnt++;
	      }
	}

	if (cnt) {
		/*
		 * If there are any resonances, recompute the frequency
		 * as the mean.
		 */
		freq_ppm = sum / cnt;
	}

	/*
	 * Leave freq_ppm at the current value if there is
	 * no way to calculate it.
	 */

	freq_valid = true;
}

// ----------------------------------------------------------------------------
//
void Resonance::set_frequency(double freq_ppm)
{
  if (assignment_count() == 0)
    {
      this->freq_ppm = freq_ppm;
      freq_valid = true;
    }
}

/*
 * Return the one SD value for this frequency.
 */
double Resonance::frequencySigma() const
{
	//
	// Return the average of the peak links along all axes
	//
	double		sum = 0.0;
	double		sum2 = 0.0;
	int		cnt = 0;

	for (int xp = 0 ; xp < crosspeaks().size() ; ++xp) {
	  CrossPeak *x = (CrossPeak *) crosspeaks()[xp];
	  for (int axis = 0 ; axis < x->dimension() ; ++axis)
	    if (x->resonance(axis) == this)
	      {
		double v = x->frequency(axis);
		sum += v;
		sum2 += v * v;
		cnt++;
	      }
	}
	if (cnt > 1) {
		double arg = (sum2 / cnt) - (sum / cnt) * (sum / cnt);
		if (arg > 0)
			return sqrt(arg);
	}
	return 0.0;
}

// ----------------------------------------------------------------------------
//
void Resonance::peak_moved(const CrossPeak *)
{
  InvalidateFrequency();
}

/*
 * Called when the frequency cache is no longer valid
 */
void Resonance::InvalidateFrequency()
{
  freq_valid = false;
  Notifier &n = condition()->molecule()->session().notifier();
  n.send_notice(nt_changed_resonance, this);
}

// ----------------------------------------------------------------------------
// Convenience functions passed to the Resonance's Atom
//
Stringy Resonance::name() const
  { return atom()->LongName(); }

// ----------------------------------------------------------------------------
//
Resonance *closest_resonance(const List &res, double freq_ppm)
{
  Resonance *closest = NULL;
  double min_deviation = 0;	// Suppress compiler warning about uninit var

  for (int ri = 0 ; ri < res.size() ; ++ri)
    {
      Resonance *r = (Resonance *) res[ri];
      double deviation = fabs(r->frequency() - freq_ppm);
      if (closest == NULL || deviation < min_deviation)
	{
	  min_deviation = deviation;
	  closest = r;
	}
    }

  return closest;
}

// ----------------------------------------------------------------------------
//
int compare_resonance_frequencies(const void *a, const void *b)
{
  double fa = ((Resonance *) a)->frequency();
  double fb = ((Resonance *) b)->frequency();

  return (fa == fb ? 0 : (fa < fb ? -1 : 1));
}

// ----------------------------------------------------------------------------
//
int compare_resonance_names(const void *res1, const void *res2)
{
  Resonance *r1 = (Resonance *) res1;
  Resonance *r2 = (Resonance *) res2;

  int order = compare_ints(r1->group()->number(), r2->group()->number());
  if (order != 0)
    return order;

  order = compare_strings(r1->group()->name(), r2->group()->name());
  if (order != 0)
    return order;

  order = compare_strings(r1->atom()->name(), r2->atom()->name());
  return order;
}

// ----------------------------------------------------------------------------
// Comparison function for sorting resonances by aliased frequency.
//
class compare_aliased_resonances: public ListOrder
{
public:
  compare_aliased_resonances(double alias_min, double alias_max)
    {
      this->alias_min = alias_min;
      this->alias_max = alias_max;
    }
  virtual int operator()(const void *r1, const void *r2) const;
private:
  double alias_min, alias_max;
};

// ----------------------------------------------------------------------------
//
int compare_aliased_resonances::operator()(const void *res1,
					   const void *res2) const
{
  Resonance *r1 = (Resonance *) res1;
  Resonance *r2 = (Resonance *) res2;
  return compare_doubles(interval_mod(r1->frequency(), alias_min, alias_max),
			 interval_mod(r2->frequency(), alias_min, alias_max));
}

// ----------------------------------------------------------------------------
//
void sort_aliased_resonances(List &rlist, double alias_min, double alias_max)
{
  rlist.sort(compare_aliased_resonances(alias_min, alias_max));
}

// ----------------------------------------------------------------------------
//
class compare_resonance_distance : public ListOrder
{
public:
  compare_resonance_distance(double center,
			     double alias_min, double alias_max)
    {
      this->alias_min = alias_min;
      this->alias_max = alias_max;
      this->center = center;
    }
  virtual int operator()(const void *r1, const void *r2) const;
private:
  double alias_min, alias_max, center;
};

// ----------------------------------------------------------------------------
//
int compare_resonance_distance::operator()(const void *res1,
					   const void *res2) const
{
  Resonance *r1 = (Resonance *) res1;
  Resonance *r2 = (Resonance *) res2;

  double pos1 = interval_mod(r1->frequency(), alias_min, alias_max);
  double pos2 = interval_mod(r2->frequency(), alias_min, alias_max);

  return compare_doubles(fabs(pos1 - center), fabs(pos2 - center));
}

// ----------------------------------------------------------------------------
//
List closest_n_resonances(const List &rlist, double alias_min,
			  double alias_max, double pos_ppm, int n)
{
  if (rlist.size() <= n)
    return rlist;

  List res = rlist;
  res.sort(compare_resonance_distance(pos_ppm, alias_min, alias_max));
  return res.sublist(0, n);
}
