/*
 * Support for printing of spectra in postscript format.
 */
#include <math.h>		// use fabs
#include <stdlib.h>		// use getenv()
#include <time.h>

#include "axismap.h"		// use Axis_Map
#include "color.h"
#include "condition.h"	// Use Condition
#include "contour.h"
#include "paths.h"		// use installation_path(), SPARKY_PRINT
#include "label.h"		// Use postscript_string()
#include "list.h"	// Use List
#include "num.h"
#include "print.h"
#include "project.h"		// use Project
#include "reporter.h"		// use Reporter
#include "resonance.h"
#include "session.h"		// use Session
#include "spectrum.h"		// Use Spectrum
#include "spoint.h"	// Use SRegion
#include "stringc.h"	// Use Stringy
#include "system.h"	// Use get_username(), start_process(), run_command()
#include "uimain.h"		// Use query()
#include "uirpanel.h"		// Use resonance_positioning()
#include "uislice.h"		// Use Slice
#include "uiview.h"
#include "version.h"		// use SPARKY_VERSION, SPARKY_RELEASE
#include "wait.h"		// use Wait
#include "winsystem.h"		// use color_rgb()

#define DEFAULTPRINT		"sparky.ps"

#define DPI		300
#define UPI		72

#define P_GRID		0
#define P_TICK		1

#define BANNER_SIZE	1.75
#define SLICE_SIZE	1.0
#define RES_SIZE	1.0

#define RESFONTNAME	"Helvetica"
#define RESFONTSIZE	12
#define RESHEIGHT	80

#define MAX_LINE_LENGTH	4096

#define inch_to_cm(x)		((x) * 2.54)
#define cm_to_inch(x)		((x) / 2.54)

typedef	struct	{
	char	*name;				/* printer name */
	float	width, height;
	float	left_margin, bottom_margin;	/* margin to imagable area */
	float	right_margin, top_margin;	/* size of imagable area */
}	paper_t;

typedef struct
{
  // Temporaries for doing printing

	double		start[2];	// BL of print region (view units)
	double		end[2];		// TR of print region (view units)
	double		labelincr[2];	// label every <labelincr>
	int		ticks[2];	// tick every <ticks>
	int		pages[2];
} Print_Layout;

static void resonance_print(FILE *fp, View *view, Axis a,
			    const SRegion &region, int dpi, double scale);
static void print_contour_level(float level, const Color &color, FILE *fp);
static void print_contour_path(float *xy, bool switched, int len,
			       double xoffset, double yoffset,
			       double xscale, double yscale, FILE *fp);
static void makeaxis(double start, double end, double delta,
		     double *roriginp, int *rcountp);
static int decimals_after_point(double step);
static void ornament_print_region(FILE *fp, View *vp, Rectangle rect,
				  double xsc, double ysc,
				  bool peakgroupshow, bool peakshow,
				  bool blacklabel);
static void print_ornaments(Ornament_Type type, View *view, const Rectangle &rect,
			    double xsc, double ysc, FILE *fp);
static double nice_scale_step(double max_step);
static int number_of_ticks(double incr);
static void print_contours(View *vp, const SRegion &region,
			   double xscale, double yscale, FILE *fp, Wait *wait);
static void slice_print(FILE *fp, View *view, Axis a, const SRegion &region,
			double scale, int thickness);
static void slice_range_labels(FILE *fp, double hmin, double hmax,
			       double yoffset, double yscale);
static void draw_slice_curve(FILE *fp, double iscale, double ioffset,
			     double yscale, double yoffset,
			     float *data, int len);
static int slice_print_position(double x, double offset, double scale);
static Stringy postscript_previewer();

// ----------------------------------------------------------------------------
//
static paper_t	paper_types[]	= {
	{ "A",  8.5, 11, 1, 1, 1, 1},		/* 8.5x11 in */
	{ "AC", 8.5, 11, 1, 2, 1, 1.6},		/* 8.5x11 in, color */
	{ 0 }
};
static paper_t	*_paper = &paper_types[0];

static const char *print_color_list[] =
{"black", "white", "gray", "red", "green", "blue",
 "yellow", "magenta", "cyan", "light blue",
 "light green", "light yellow", "gold", "orange",
 "pink", "beige", "turquoise", "coral", "maroon",
 "purple", "tomato", "dark orange", "chartreuse",
 "spring green", "deep sky blue",
 NULL};

// ----------------------------------------------------------------------------
//
class Printing_Routines : public Drawing_Routines
{
public:
  Printing_Routines(View *view, int xaxis, int yaxis,
		    double xoffset, double yoffset,	// max PPM
		    double xscale, double yscale,	// dots per PPM
		    FILE *fp)
    {
      this->view = view;
      this->xaxis = xaxis;
      this->yaxis = yaxis;
      Spectrum *sp = view->spectrum();
      this->index_xscale = xscale * sp->scale(1.0, xaxis, INDEX, PPM);
      this->index_yscale = yscale * sp->scale(1.0, yaxis, INDEX, PPM);
      this->index_xoffset = sp->map(xoffset, xaxis, PPM, INDEX);
      this->index_yoffset = sp->map(yoffset, yaxis, PPM, INDEX);
      this->fp = fp;
    }
  virtual void set_contour_level(float level)
    {
      const Contour_Parameters &cp = view->contour_parameters(level);
      print_contour_level(level, cp.level_color(level), fp);
    }
  virtual void draw_contour_path(float *xy, int len)
    {
      print_contour_path(xy, xaxis > yaxis, len,
			 index_xoffset, index_yoffset,
			 index_xscale, index_yscale, fp);
    }

private:
  View *view;
  int xaxis, yaxis;
  double index_xscale, index_yscale;	// printer dots per INDEX unit
  double index_xoffset, index_yoffset;	// min INDEX
  FILE *fp;
};

/*
 * Get the location of the bottom left (XY) corner, returning the
 * result in <xp> and <yp>.
 *
 * Get the size of the print image, returning the width in <wp>
 * and the height in <hp>.
 */
static void get_image_xywh(double *x, double *y, double *w, double *h,
			   bool slices, bool rpanels, const Print_Options &po)
{
	if (!po.landscape) {
		*x = _paper->left_margin;
		*y = _paper->bottom_margin;
	}
	else {
		*x = _paper->top_margin;
		*y = _paper->left_margin;
		if (po.show_banner)
			*x += BANNER_SIZE;
	}
	if (slices)
		*y += SLICE_SIZE;
	if (rpanels)
		*y += RES_SIZE;

	*w = _paper->width - _paper->left_margin - _paper->right_margin;
	*h = _paper->height - _paper->top_margin - _paper->bottom_margin;

	if (slices) {
		*w -= SLICE_SIZE;
		*h -= SLICE_SIZE;
	}
	if (rpanels) {
		*w -= RES_SIZE;
		*h -= RES_SIZE;
	}
	if (po.show_banner)
		*h -= BANNER_SIZE;
	if (po.landscape)
		swap(w, h);
}

/*
 * Set the number of pages needed to print with fixed scale printing.
 */
#define MAX_PRINT_LABELS 100
#define DEFAULT_PRINT_LABELS 7

static Print_Layout compute_layout(View *vp, Rectangle r,
				   const Print_Options &po)
{
  Print_Layout layout;

  Units u = vp->axis_units();

  layout.start[0] = vp->map(r.max(X), X, PPM, u);
  layout.start[1] = vp->map(r.max(Y), Y, PPM, u);
  layout.end[0]   = vp->map(r.min(X), X, PPM, u);
  layout.end[1]   = vp->map(r.min(Y), Y, PPM, u);

  double rx = fabs(layout.end[0] - layout.start[0]);
  double sx = vp->scale(po.label_spacing[0], X, PPM, u);
  layout.labelincr[0] = (sx * MAX_PRINT_LABELS > rx ? sx :
			 nice_scale_step(rx / DEFAULT_PRINT_LABELS));

  double ry = fabs(layout.end[1] - layout.start[1]);
  double sy = vp->scale(po.label_spacing[1], Y, PPM, u);
  layout.labelincr[1] = (sy * MAX_PRINT_LABELS > ry ? sy :
			 nice_scale_step(ry / DEFAULT_PRINT_LABELS));

  layout.ticks[X] = number_of_ticks(layout.labelincr[X]);
  layout.ticks[Y] = number_of_ticks(layout.labelincr[Y]);

  if (po.fixed_scale)
    {
      double x, y, width, height;
      View_Settings s = vp->settings();
      get_image_xywh(&x, &y, &width, &height,
		     s.show_slices, s.show_resonance_panels, po);
      width = inch_to_cm(width);
      height = inch_to_cm(height);

      double x_page_size = vp->scale(po.fixed_scales[0], X, PPM, u) * width;
      double x_size = fabs(layout.start[0]-layout.end[0]);
      layout.pages[0] = (x_page_size > 0 ? (int) ceil(x_size/x_page_size) : 1);

      double y_page_size = vp->scale(po.fixed_scales[1], Y, PPM, u) * height;
      double y_size = fabs(layout.start[1]-layout.end[1]);
      layout.pages[1] = (y_page_size > 0 ? (int) ceil(y_size/y_page_size) : 1);
    }
  else
    {
      layout.pages[0] = po.pages[0];
      layout.pages[1] = po.pages[1];
    }
  layout.pages[0] = max(1, layout.pages[0]);
  layout.pages[1] = max(1, layout.pages[1]);

  return layout;
}

// ----------------------------------------------------------------------------
//
static double nice_scale_step(double max_step)
{
  long exp;
  double mant, decade, f;

  scientific_form(max_step, &mant, &exp, &decade);
  if (mant <= .2)
    f = .2;
  else if (mant <= .5)
    f = .5;
  else
    f = 1;

  return (f * decade);
}

/*
 * Determine a good number of ticks to use for the given label increment.
 */
static int number_of_ticks(double incr)
{
	int ticks = 5;			/* the default value */

	incr = fabs(incr);

	if (incr == 0)
	  return ticks;

	// Get the leading base 10 digit
	while (incr < 1)
	  incr *= 10;
	while (incr >= 10)
	  incr /= 10;

	/*
	 * Now map to a nice number of ticks.
	 */
	switch ((int) incr) {
		case 2: case 4: case 8: ticks = 4; break;
		case 3: case 6: ticks = 6; break;
		case 7: ticks = 7; break;
		case 9: ticks = 9; break;
	}

	return ticks;
}

#define GSAVE(fp)	fprintf(fp, "gsave\n")
#define GRESTORE(fp)	fprintf(fp, "grestore\n")
#define CLIP(fp)	fprintf(fp, "0 0 WIDTH HEIGHT rectpath clip newpath\n");

//
// Print one axis
//
static void
axis_scale_print(
	FILE	*fp,			// file to receive print
	View *	vp,			// view (for data conversion)
	double	xstart,			// start axis value (view units)
	double	xend,			// end axis value (view units)
	double	xinc,			// axis increment (view units)
	int	prec,			// printing precision for numbers
	Axis	axis,			// which axis (X or Y)
	double	scale,			// printer dots per ppm < 0
	double	range,			// range in ppm < 0
	int	ticks,			// number of intervals between numbers
	int	doGrid)			// true if doing a grid
{
	double	pos,			// position in printer space
		x,			// position in data space
		xstart_dots,		// xstart in printer space
		rfirst,			// first label position
		rinc_dots;		// increment between label positions
	int	rsteps,			// number of labels
		i,			// general integer
		count,			// count of number of labels
		before;			// number of ticks before first label

	Units	u = vp->axis_units();

	makeaxis(xstart, xend, xinc, &rfirst, &rsteps);
	if (xend < xstart) {
		rfirst += xinc * rsteps;
		xinc = -xinc;
		while (rfirst > xstart) {
			rfirst += xinc;
			rsteps--;
		}
	}
	else {
		while (rfirst < xstart) {
			rfirst += xinc;
			rsteps--;
		}
	}

	xstart_dots = vp->map(xstart, axis, u, PPM) * scale;
	rinc_dots = (vp->map(rfirst + xinc, axis, u, PPM)
		     - vp->map(rfirst, axis, u, PPM)) * scale;

	/*
	 * Move to the beginning of the labels.
	 */
	GSAVE(fp);
	if (axis == X) {
	  pos = vp->map(rfirst, axis, u, PPM) * scale - xstart_dots;
	  fprintf(fp, "%f xslice_y", pos);
	}
	else
	  {
	    pos = vp->map(rfirst, axis, u, PPM) * scale - xstart_dots;
	    fprintf(fp, "0 %f\n", pos);
	  }
			
	fprintf(fp, " translate 0 0 moveto\n");

	/*
	 * Spit out the labels, counting as you go, and putting them out
	 * in reverse order.
	 */
	count = 0;
	for (i = rsteps; i >= 0; i--) {
		x = rfirst + xinc * i;
		pos = vp->map(x, axis, u, PPM) * scale - xstart_dots;
		if (0 <= pos && pos <= range * scale) {
			fprintf(fp, "(%.*f)\n", prec, x);
			count++;
		}
	}

	/*
	 * Call the PostScript routine to do the labeling
	 */
	if (axis == X)
	  fprintf(fp, "%d HEIGHT xslice_y sub %f xlabel\n", count, rinc_dots);
	else
	  fprintf(fp, "%d yslice_x %f ylabel\n", count, rinc_dots);
	GRESTORE(fp);


	/*
	 * Put axis labels on
	 */
	fprintf(fp, "(%s) (%s) (%d) %saxislabel\n",
		index_name(Unit_Names, vp->axis_units()).cstring(),
		vp->spectrum()->nucleus_type(vp->axis(axis)).cstring(),
		vp->axis(axis) + 1,
		(axis == X ? "x" : "y"));

	ticks = max(1, ticks);
	xinc /= ticks;
	rinc_dots /= ticks;

	before = 0;
	x = rfirst - xinc;
	while (vp->map(x, axis, u, PPM) * scale - xstart_dots >= 0) {
		x -= xinc;
		before++;
	}

	/*
	 * Spit out the ticks
	 */
	GSAVE(fp);
	x = vp->map(rfirst - before * xinc, axis, u, PPM)*scale - xstart_dots;
	if (axis == X) {
		fprintf(fp, "%f 0 translate\n", x);
		if (doGrid) {
			GSAVE(fp);
			CLIP(fp);
			fprintf(fp, "%d %d HEIGHT %f WIDTH %f sub grid\n",
				before % ticks, ticks, rinc_dots, x);
			GRESTORE(fp);
		}

		/*
		 * Put out the tick marks
		 */
		fprintf(fp, "[[xslice_y 1] %s] %d %d %f WIDTH %f sub xTick\n",
			doGrid ? "" : "[0 1] [HEIGHT -1]",
			before % ticks, ticks, rinc_dots, x);
	}
	else {
		fprintf(fp, "-90 rotate -1 1 scale %f 0 translate\n", x);

		/*
		 * Put out the grid.
		 */
		if (doGrid) {
			GSAVE(fp);
			fprintf(fp, "0 0 HEIGHT WIDTH rectpath clip newpath\n");
			fprintf(fp, "%d %d WIDTH %f HEIGHT %f sub grid\n",
				before % ticks, ticks, rinc_dots, x);
			GRESTORE(fp);
		}
		fprintf(fp, "[%s [yslice_x -1]] %d %d %f HEIGHT %f sub yTick\n",
			doGrid ? "" : "[0 1] [WIDTH -1]",
			before % ticks, ticks, rinc_dots, x);
	}
	GRESTORE(fp);
}

//
// Print a page
//
static void
_printPage(
	View *		vp,		// the current view (for data conv.)
	const SRegion & region,		// the printed region (ppm)
	const Print_Options &opt,	// which print options
	const Print_Layout &layout,
	FILE		*fp,		// who receives output
	const Stringy	&who,		// the person doing the print
	time_t		clock,		// when the print started
	Wait *wait)
{
  char		*timestring;	// the current date & time
  int		prec;		// the axis numbering position
  int		dim;		// dimension of the view
  int		xaxis = vp->axis(X);
  int		yaxis = vp->axis(Y);
  View_Settings s = vp->settings();
  int		slices = s.show_slices;
  int		resonances = s.show_resonance_panels;
  int		landscape = opt.landscape;
  Units		u = vp->axis_units();

  if (slices)
    fprintf(fp, "/SLICE %.1f def\n", SLICE_SIZE * DPI);
  if (resonances)
    fprintf(fp, "/RESONANCE %.1f def\n", RES_SIZE * DPI);

  double bl_x = vp->map(region.max[xaxis], X, PPM, u);
  double bl_y = vp->map(region.max[yaxis], Y, PPM, u);

  double tr_x = vp->map(region.min[xaxis], X, PPM, u);
  double tr_y = vp->map(region.min[yaxis], Y, PPM, u);

  double dx = region.size(xaxis);
  double dy = region.size(yaxis);

  double		width;		// width  of printed region
  double		height;		// height of printed region

  double	imageLeft, imageBottom, imageWidth, imageHeight;  // Inches
  get_image_xywh(&imageLeft, &imageBottom, &imageWidth, &imageHeight,
		 slices, resonances, opt);

  //
  // If FIXED_SCALE, the size of the print is fixed and we just
  // have to worry about centering it
  //
  if (opt.fixed_scale) {
    double scx = vp->scale(max(opt.fixed_scales[0], .001), X, PPM, u);
    double scy = vp->scale(max(opt.fixed_scales[1], .001), Y, PPM, u);
    width = cm_to_inch(fabs(bl_x - tr_x) / scx);
    height = cm_to_inch(fabs(bl_y - tr_y) / scy);
  }

  //
  // If not FIXED_SCALE, fit the print into as much area as possible.
  // This is done by finding the scale that will have one side
  // as long as the printable area.
  //
  else {
    // On screen aspect ratio in physical distance (eg inches)
    double aspect = (dx / dy) * vp->aspect();

    if (aspect * imageHeight < imageWidth) {
      width = aspect * imageHeight;
      height = imageHeight;
    }
    else {
      width = imageWidth;
      height = imageWidth / aspect;
    }
  }

  // offsets from paper edge to region in inches
  double left = imageLeft + (imageWidth - width) / 2;
  double bottom = imageBottom + (imageHeight - height) / 2;

  /*
   * Determine the scale parameters for converting from data
   * units to printer units.
   */
  double xscale = width / dx * DPI;	// printer dots per PPM
  double yscale = height / dy * DPI;	// printer dots per PPM

  /*
   * Output some needed constants
   */
  fprintf(fp, "/WIDTH %d def\n", (int) (width * DPI));
  fprintf(fp, "/HEIGHT %d def\n", (int) (height * DPI));

  /*
   * Set up the translate, scale and rotation for the view
   */
  GSAVE(fp);
  if (landscape)
    fprintf(fp, "-90 rotate %.2f I 0 translate\n", -_paper->height);
  fprintf(fp, "%.2f I %.2f I translate\n", left, bottom);
  fprintf(fp, "SCALE dup scale\n");


  /*
   * Do the axes.
   */
  fprintf(fp, "setAxisFont 0 setgray\n");

  if (s.show_scales)
    {
      int doGrid = opt.show_grids;
      /*
       * The precision has to be at least large enough to show the
       * labelincr without loss of resolution.
       */
      prec = decimals_after_point(layout.labelincr[X]);
      axis_scale_print(fp, vp, bl_x, tr_x, layout.labelincr[X],
		       prec, X, -xscale, -dx, layout.ticks[X], doGrid);

      prec = decimals_after_point(layout.labelincr[Y]);
      axis_scale_print(fp, vp, bl_y, tr_y, layout.labelincr[Y],
		       prec, Y, -yscale, -dy, layout.ticks[Y], doGrid);
    }

  /*
   * Clip to the data and contour it.
   */
  if (opt.show_contours) {
    print_contours(vp, region, xscale, yscale, fp, wait);
  }

  /*
   * Clip the region and do the ornaments.
   */
  GSAVE(fp);
  CLIP(fp);
  ornament_print_region(fp, vp, vp->flatten(region), xscale, yscale,
			opt.show_peakgroups, opt.show_peaks, opt.black_labels);
  GRESTORE(fp);

  if (slices) {
    int thickness = (int) (SLICE_SIZE * DPI + 0.5);
    slice_print(fp, vp, X, region, xscale, thickness);
    slice_print(fp, vp, Y, region, yscale, thickness);

    // Draw a crosshair at the slice position
    GSAVE(fp);
    CLIP(fp);
    double y = vp->slice(X)->section()[yaxis];
    double x = vp->slice(Y)->section()[xaxis];
    fprintf(fp, "%f 0 moveto 0 HEIGHT rlineto\n",
	    (region.max[xaxis] - x) * xscale);
    fprintf(fp, "0 %f moveto WIDTH 0 rlineto stroke\n",
	    (region.max[yaxis] - y) * yscale);
    GRESTORE(fp);
  }

  if (resonances) {
    resonance_print(fp, vp, X, region, DPI, xscale);
    resonance_print(fp, vp, Y, region, DPI, yscale);
  }

  /*
   * For greater than 2-D spectra, place box with the text
   * that describes the plane that is being looked at.
   */
  if ((dim = vp->spectrum()->dimension()) > 2) {
    IPoint axis_order = vp->axis_order();
    fprintf(fp, "[");
    for (int k = 2 ; k < dim ; ++k) {
      int	oa = axis_order[k];

      double	zplane = vp->spectrum()->map(region.min[oa], oa, PPM, u);
      int precision = (u == INDEX ? 0 : 3);

      if (u == INDEX)
	zplane++;

      fprintf(fp, "[(%s) (%s) (%d) (%.*f)]",
	      index_name(Unit_Names, u).cstring(),
	      vp->spectrum()->nucleus_type(oa).cstring(),
	      oa + 1,
	      precision,
	      zplane);
    }
    fprintf(fp, "] planebanner\n");
  }


  /*
   * Edge the view
   */
  fprintf(fp, "edgeview\n");


  /*
   * Pop the saved state (for translation and scale)
   */
  GRESTORE(fp);

  /*
   * Set up for the banner by translating to a an 'X''Y' location
   * which will be in the bottom, center of the banner.
   */
  GSAVE(fp);
  if (opt.show_banner) {
    fprintf(fp, "%.2f I %.2f 1.75 add I translate\n",
	    landscape ? (bottom + height / 2) : (left + width / 2),
	    landscape ? (_paper->height -left) : (bottom + height));

    fprintf(fp, "[\n");
    if (!opt.title.is_empty())
      fprintf(fp, "[(Title:  %s   --   ", opt.title.cstring());
    else
      fprintf(fp, "[(Spectrum:  ");

    fprintf(fp, "%s)]\n", vp->spectrum()->name().cstring());

    timestring = ctime(&clock);
    timestring[24] = '\0';
    fprintf(fp, "[(User:  %s     Date:  %s)]\n", who.cstring(), timestring);
	
    const Contour_Levels &pos_levels = vp->positive_contours().levels;
    if (pos_levels.levels) {
      fprintf(fp, "[(Positive contours: low %.2e  ", pos_levels.lowest);
      fprintf(fp, "levels %2d  factor %.2f",
	      pos_levels.levels, pos_levels.factor);
      fprintf(fp, ")]\n");
    }

    const Contour_Levels &neg_levels = vp->negative_contours().levels;
    if (neg_levels.levels) {
      fprintf(fp, "[(Negative contours: low %.2e  ", neg_levels.lowest);
      fprintf(fp, "levels %2d  factor %.2f",
	      neg_levels.levels, neg_levels.factor);
      fprintf(fp, ")]\n");
    }

    if (opt.fixed_scale) {
      fprintf(fp, "[(X scale: %.04f ppm / cm    ", opt.fixed_scales[0]);
      fprintf(fp, " Y scale: %.04f ppm / cm)]\n", opt.fixed_scales[1]);
    }

    fprintf(fp, "] banner\n");
  }
  GRESTORE(fp);


  /*
   * Done. Restore and show the page and flush the output
   */
  fprintf(fp, "showpage\n");
  fflush(fp);
}

// ----------------------------------------------------------------------------
//
static void print_contours(View *vp, const SRegion &region,
			   double xscale, double yscale, FILE *fp, Wait *wait)
{
  GSAVE(fp);
  CLIP(fp);

  int xaxis = vp->axis(X);
  int yaxis = vp->axis(Y);
  double xoffset = region.max[xaxis];
  double yoffset = region.max[yaxis];
  const View_Settings &s = vp->settings();
  Printing_Routines pr(vp, xaxis, yaxis, xoffset, yoffset, xscale, yscale, fp);
  SRegion r = flat_region(region, xaxis, yaxis);
  Project &proj = vp->session().project();
  Contour_Cache *cc = proj.contour_cache();
  draw_contours(vp->spectrum(),	r,
		s.pos_contours.levels, s.neg_contours.levels,
		s.subtract_fit_peaks, pr, cc, wait);

  // Overlaid views.
  List vlist = proj.overlaid_views(vp);
  for (int vi = 0 ; vi < vlist.size() ; ++vi)
    {
      View *v = (View *) vlist[vi];
      Spectrum *sp = v->spectrum();
      Axis_Map axismap;
      if (identity_axis_map(sp, vp->spectrum(), &axismap) ||
	  unique_axis_map(sp, vp->spectrum(), &axismap))
	{
	  SRegion vregion = axismap.invert(region);
	  int vxaxis = axismap.invert(xaxis), vyaxis = axismap.invert(yaxis);
	  const View_Settings &s = v->settings();
	  Printing_Routines pr(v, vxaxis, vyaxis,
			       xoffset, yoffset, xscale, yscale, fp);
	  SRegion r = flat_region(vregion, vxaxis, vyaxis);
	  draw_contours(sp, r, s.pos_contours.levels, s.neg_contours.levels,
			s.subtract_fit_peaks, pr, cc, wait);
	}
    }

  GRESTORE(fp);
}

/*
 * Print to FILE <fp> the a-axis resonance list for given ppm limits
 */
static void resonance_print(FILE *fp, View *view, Axis a,
			    const SRegion &region,
			    int dpi,
			    double scale)  // dots per PPM
{
  //
  // I don't use the actual resonance panel font height because
  // the printed resonance panel thickness is fixed (ie not equal to the
  // on screen thickness) so it makes sense to fix the font height too.
  //
  double font_inches = RESFONTSIZE / 72.0;
  double ppm_per_inch = dpi / scale;
  double font_height = ppm_per_inch * font_inches;
  double ppm_spacing = font_height;
  RPanel *rpan = view->resonance_panel(a);
  List pstrings = rpan->resonance_positions(region, ppm_spacing);

  fprintf(fp, (a == X ? "xres_init\n" : "yres_init\n"));
  fprintf(fp, "0 0 RESONANCE %s rectpath clip newpath\n",
	  (a == X ? "WIDTH" : "HEIGHT"));

  /*
   * Initialize the font
   */
  int font_size = nearest_int(72 * font_height * scale / dpi); // Point size
  fprintf(fp, "/XfR /%s bindFont XfR\n", RESFONTNAME);
  fprintf(fp, "/resScale %d SC def\n", font_size);
//  fprintf(fp, "/resScale %d SC def\n", RESFONTSIZE);

  double ppm_max = region.max[view->axis(a)];
  for (int k = 0 ; k < pstrings.size() ; k++)
    {
      double p, plabel, pbot, pshift;
      Panel_String *ps = (Panel_String *) pstrings[k];

      p = (ppm_max - ps->tick_position) * scale;
      plabel = (ppm_max - ps->label_position) * scale;
      pbot = (ppm_max - ps->label_position - font_height/4) * scale;
      pshift = plabel - p;

      fprintf(fp, "%f %f {%s} %f PaintRes\n",
	      p, pshift, postscript_string(ps->label).cstring(), pbot);
    }

  fprintf(fp, (a == X ? "xres_done\n" : "yres_done\n"));

  free_panel_string_list_entries(pstrings);
}

/*
 * Map the <r,b,g> triplet into a black&white value for printing
 */
static double rgb_to_grey(double r, double g, double b)
{
	return (r * 0.302 + g * 0.584 + b * 0.114);
}

/*
 * Print the colormap to file <fp> as a PostScript array of
 * [ b&w r g b ] entries, all elements between 0 and 1.
 */
static void cmap_print(WinSys &ws, FILE *fp)
{
	fprintf(fp, "/colors [\n");
	for (int ci = 0 ; print_color_list[ci] ; ++ci)
	  {
	    double r = 0, g = 0, b = 0;
	    ws.color_rgb(print_color_list[ci], &r, &g, &b);
	    fprintf(fp, "[%.2f %.2f %.2f %.2f]\n",
		    rgb_to_grey(r, g, b), r, g, b);
	  }
	fprintf(fp, "] def\n");
}

/*
 * Set up the colors used for printing contours, both filled and
 * stroked contours.
 */
static void cmap_contoursetup(
	FILE	*fp,
	int	poslevels,
	int	posfill,
	int	neglevels,
	int	negfill)
{
	int	i;

	fprintf(fp, "/PositiveLevels %d def\n/NegativeLevels %d def\n",
		poslevels, neglevels);
	fprintf(fp, "/strokecolors [\n");
	for (i = 0; i < poslevels; i++)
		fprintf(fp, "[0 1 0 0]\n");
	for (i = 0; i < neglevels; i++)
		fprintf(fp, "[0 0 1 0]\n");
	fprintf(fp, "] def\n");

	fprintf(fp, "/fillcolors [\n");
	for (i = 0; i < poslevels; i++)
		fprintf(fp, "%s\n", posfill ? "[.9 1 0 0]" : "null");
	for (i = 0; i < neglevels; i++)
		fprintf(fp, "%s\n", negfill ? "[.8 0 1 0]" : "null");
	fprintf(fp, "] def\n");

	fprintf(fp, "/dashpatterns [\n");
	fprintf(fp, "[[] []]\n");		// Positive contours
	fprintf(fp, "[[DASH] []]\n");		// Negative contours
	fprintf(fp, "] def\n");
}

/*
 * Send the PostScript prolog file to the printed output.
 */
static int _printProlog(FILE *to, View *vp)
{
  Stringy prolog = file_path(installation_path(SPARKY_LIB), SPARKY_PRINT);

	FILE	*fp = fopen(prolog.cstring(), "r");	// Postscript header
	if (fp == NULL) {
	  Reporter &rr = vp->session().reporter();
	  rr.warning( "Can't open printer init file %s", prolog.cstring());
	  return false;
	}

	/*
	 * Copy then close init file.
	 */
	char	buf[MAX_LINE_LENGTH];
	while (fgets(buf, sizeof buf, fp))
		fputs(buf, to);
	fclose(fp);

	/*
	 * Set up needed constants.
	 */
	fprintf(to, "/SparkyMajorVersion %d def\n",  SPARKY_VERSION);
	fprintf(to, "/SparkyMinorVersion %d def\n",  SPARKY_RELEASE);
	fprintf(to, "/SCALE %f def\n", (double) UPI / DPI);
	fprintf(to, "/PEAKLINEWIDTH %d def\n", nearest_int(0.015 * DPI));

	/*
	 * Set up the color array which maps pixels to colors.
	 */
	cmap_print(vp->session().window_system(), to);
	cmap_contoursetup(to, vp->positive_contours().levels.levels, 0,
			  vp->negative_contours().levels.levels, 0);

	/*
	 * End the file prolog.
	 */
	fputs("%%EndProlog\n\n", to);

	return !ferror(to);
}

/*
 * Open the printer, return a FILE pointer if successful or NULL if failed
 */
static FILE *_openPrinter(const Print_Options &po, Reporter &rr)
{
  Stringy path = (po.print_to_file ?
		  tilde_expand(po.print_file) : po.temp_file);

  FILE *ps = fopen(path.cstring(), "w");
  if (ps == NULL)
    rr.warning("Can't open file %s for printing", path.cstring());

  return ps;
}

/*
 * Close the printer.
 */
static bool _finalizePrinter(FILE *ps, const Print_Options &po, Reporter &rr)
{
  bool success = ! ferror(ps);
  fclose(ps);
  if (success)
    {
      if (po.preview)
	{
	  Stringy previewer = postscript_previewer();
	  Stringy cmd = previewer + " " + po.temp_file;
	  if (!start_process(cmd))
	    rr.warning("Couldn't start postscript previewer %s",
		       previewer.cstring());
	  remove_file_at_exit(po.temp_file);
	}
      else if (po.print_to_file)
	;
      else
	{
	  Stringy cmd = po.print_command;
	  if (cmd.is_empty())
	    cmd = default_print_command();
	  cmd = cmd +  " " + po.temp_file;
	  if (!start_process(cmd))
	    rr.warning("Couldn't execute print command %s", cmd.cstring());
	  remove_file_at_exit(po.temp_file);
	}
    }
  if (!success)
    rr.warning("Printing failed.");

  return success;
}

/*
 * Initialize the output device or file, returning a file pointer if
 * successful, otherwise return NULL.
 */
static FILE *_initializePrinter(const Print_Options &po, View *vp)
{
  Reporter &rr = vp->session().reporter();
  FILE *ps = _openPrinter(po, rr);

	if (ps == NULL)
		return NULL;

	if (_printProlog(ps, vp) == false) {
		_finalizePrinter(ps, po, rr);
		ps = NULL;
		return NULL;
	}

	return ps;
}

/*
 * _doPrint helper function, needed because _doPrint can run in
 * the foreground or background.
 */
static bool _doPrintHelper(View *vp, Rectangle r,
			   const Print_Options &po, Wait *wait)
{
  Print_Layout	layout = compute_layout(vp, r, po);

  int nrows = layout.pages[1];
  int ncols = layout.pages[0];
  int npages = nrows * ncols;

  /*
   * Warn if the number of pages is large.
   */
  if (npages > 9) {
    char		buf[MAX_LINE_LENGTH];
    sprintf(buf,
	    "WARNING!!  You will be printing %d pages\n\n"
	    "Are you sure you want to proceed?",
	    npages);
    if (query(vp->session(), buf, "Yes, print them", "No, cancel print") == 2)
      return false;
  }

  const char *action = (po.print_to_file ? "Saving" :
		  (po.preview ? "Previewing" : "Printing"));
  Reporter &rr = vp->session().reporter();
  rr.message("%s %d page%s", action, npages, (npages > 1 ? "s" : ""));

  Stringy who = get_username();		// who is starting the print
  if (who.is_empty())
    who = "unknown";
  time_t	clock = time(NULL);	// get the current time


  /*
   * Initialize the printer, returning an open file pointer
   */
  FILE *ps = _initializePrinter(po, vp);
  if (ps == NULL)
    return false;

  double	diffx = (layout.end[X] - layout.start[X]) / ncols;
  double	diffy = (layout.end[Y] - layout.start[Y]) / nrows;

  int	page = 1;			// printing first page

  Print_Layout lay = layout;
  lay.start[Y] = layout.start[Y];
  for (int row = 0; row < nrows; row++, lay.start[Y] += diffy) {
    if (diffy < 0)
      lay.end[Y] = max(lay.start[Y] + diffy, layout.end[Y]);
    else
      lay.end[Y] = min(lay.start[Y] + diffy, layout.end[Y]);

    lay.start[X] = layout.start[X];
    for (int col = 0; col < ncols; col++, lay.start[X] += diffx) {
      if (diffx < 0)
	lay.end[X] =max(lay.start[X]+diffx,layout.end[X]);
      else
	lay.end[X] =min(lay.start[X]+diffx,layout.end[X]);

      fprintf(ps, "%%%%Page: %d %d\n", page, page);

      //
      // Determine the printed region.
      //
      Units aunits = vp->axis_units();
      Rectangle rect(vp->map(lay.end[X], X, aunits, PPM),
		     vp->map(lay.end[Y], Y, aunits, PPM),
		     vp->map(lay.start[X], X, aunits, PPM),
		     vp->map(lay.start[Y], Y, aunits, PPM));

      rect.clip(vp->full_view());

      //
      // Print this region
      //
      SRegion region = vp->thicken(rect);
      _printPage(vp, region, po, lay, ps, who, clock, wait);

      if (wait && wait->was_stop_requested()) {
	_finalizePrinter(ps, po, rr);
	return false;
      }

      page++;
    }
  }

  return _finalizePrinter(ps, po, rr);
}


//
// This routine handles setup of the printer type, forks the background
// print process (if desired), prints the prolog and colormap and contour
// information, then prints each page.
//
bool print_view(View *vp, Rectangle r, const Print_Options &po, Wait *wait)
{
	if (_paper == NULL) {
	  Reporter &rr = vp->session().reporter();
	  rr.warning( "print: can't find paper type");
	  return false;
	}

	if (wait)
	  wait->begin_waiting("Printing contours.", true);
	bool status = _doPrintHelper(vp, r, po, wait);
	if (wait)
	  wait->end_waiting();

	Reporter &rr = vp->session().reporter();
	rr.message(status ? "\n" : " FAILED\n");

	return status;
}

//
// Initialize the printers. This routine builds a list of all printer names
// then sets the various print defaults.
//
Print_Options::Print_Options()
{
	this->print_command = default_print_command();

	this->pages[X] = this->pages[Y] = 1;

	this->landscape = false;
	this->black_labels = false;
	this->fixed_scale = false;
	this->show_banner = true;
	this->show_grids = false;
	this->show_peaks = true;
	this->show_peakgroups = true;
	this->show_contours = true;

	this->fixed_scales[0] = 1.0;
	this->fixed_scales[1] = 1.0;

	this->label_spacing[0] = 0;
	this->label_spacing[1] = 0;

	this->print_file = "";
	this->temp_file = temporary_file_name();
	this->title = "";
}

// ----------------------------------------------------------------------------
//
static Stringy postscript_previewer()
{
  return getenv("POSTSCRIPT_PREVIEWER");
}

// ----------------------------------------------------------------------------
//
bool print_previewer_exists()
{
  Stringy previewer = postscript_previewer();
  Stringy exec = first_token(previewer, NULL);
  return (!previewer.is_empty() && file_exists(exec));
}

// ----------------------------------------------------------------------------
//
static void print_contour_level(float level, const Color &color, FILE *fp)
{
  int print_level = (level >= 0 ? 0 : 1);
  fprintf(fp, "/level %d def\n", print_level);
  fprintf(fp, "/levelcolor %d def\n", color_index(color));
}

// ----------------------------------------------------------------------------
//

/*
 * This routine takes a contour line segment of <n> pieces, whose X-
 * and Y-positions are x[0],y[0]; x[1],y[1]; ... x[n-1],y[n-1] and
 * scales and prints this segment to file pointer <p>->fp.
 *
 * A contour is a set of moveto, lineto, lineto, ... coordinates
 * which when printed map directly into a PostScript object called a
 * "path". Once you have a PostScript path you can "stroke" the path
 * with a given color and line texture.
 *
 * When printing to a file or printer a PostScript header file is
 * prepended to each printed output. This header file is named
 * /usr/local/lib/nmr/sparky/Init/print.<majorver>.<minorver>.ps
 * and it defines many PostScript operations including the following:
 *
 *	"S" sets the stroke color and line texture, then strokes the path.
 *	"M" is a moveto.
 *	"L" is a lineto.
 *
 * Unused features:
 *
 *	"F" closes the path, sets the fill color and fills the path, then
 *		calls "S" to set the stroke color and line texture and
 *		stroke the path. If the contour filling is not selected,
 *		the "F" operator closes the path then calls "S".
 *	"C" closes the path from the current point to the starting "moveto".
 */

/*
 * For the PostScript output produced by Sparky there is a maximum length
 * to a PostScript path, which is the following MAX_PS_PATH variable.
 */
 #define MAX_PS_PATH 1456

static void print_contour_path(float *xy, bool switched, int len,
			       // min INDEX
			       double index_xoffset, double index_yoffset,
			       // dots per INDEX
			       double index_xscale, double index_yscale,
			       FILE *fp)
{
  if (len > MAX_PS_PATH)
    {
      print_contour_path(xy, switched, MAX_PS_PATH,
			 index_xoffset, index_yoffset,
			 index_xscale, index_yscale, fp);
      print_contour_path(xy + 2*MAX_PS_PATH, switched, len - MAX_PS_PATH,
			 index_xoffset, index_yoffset,
			 index_xscale, index_yscale, fp);
      return;
    }

  float *x = (switched ? xy + 1 : xy);
  float *y = (switched ? xy : xy + 1);

  /*
   * Print the moveto point. The "M" PostScript operator
   * does a "moveto".
   */
  int px1 = nearest_int((x[0] - index_xoffset) * index_xscale);
  int py1 = nearest_int((y[0] - index_yoffset) * index_yscale);
  fprintf(fp, "%d %d M\n", px1, py1);

  for (int s = 1 ; s < len ; ++s)
    {
      int px = nearest_int((x[2*s] - index_xoffset) * index_xscale);
      int py = nearest_int((y[2*s] - index_yoffset) * index_yscale);
      fprintf(fp, "%d %d L\n", px, py);
    }

  fprintf(fp, "S\n");
}


/*
 * Given a <start> point, an <end> point and a <delta> between points returns:
 */
static void
makeaxis(
	double	start,		/* first data point */
	double	end,		/* last data point */
	double	delta,		/* delta a point */
	double	*roriginp,	/* first point on repetitive part of axis */
	int	*rcountp)	/* number of steps on repetitive part */
{
	double	rend;

	/* Force start to be less than end
	 */
	if (start > end) {
		double temp = start;
		start = end;
		end = temp;
	}

	/* The first point and last points on the repetitive part of the axis
	 * always lie inside "start" and "end".
	 */
	if (start < 0)
		*roriginp = -delta * (int) (-start / delta);
	else
		*roriginp = delta * (int) (start / delta);
	if (end < 0)
		rend = -delta * (int) (-end / delta);
	else
		rend = delta * (int) (end / delta);


	/* The stepsize is just "delta", so the number of steps on
	 * the repetitive part is the integer part of the repetitive
	 * range divided by the delta.
	 */
	*rcountp = (int) ((rend - *roriginp) / delta + 0.5);

	/*
	 * Sanity check in case numbers entered are bogus
	 */
	*rcountp = bounded(0, *rcountp, 1024);
}

/*
 * Starting with precision estimate <prec>, return the precision that
 * will allow the <delta> to be displayed with no loss.
 */
static int decimals_after_point(double step)
{
  int prec = 0;
  for (step = fabs(step) ; step < .999 && prec <= 8 ; step *= 10)
    prec += 1;

  return prec;
}

/*
 * Print the ornaments within a region.  
 *
 * We cannot use a scale factor as does the routine that builds the
 * display list because we use different PostScript linewidths, which
 * are scaled as the scale factor.  The scale factor is different in
 * the two dimensions, causing distorted peaks.
 */
static void ornament_print_region(
	FILE		*fp,
	View *		vp,
	Rectangle	rect,		// PPM
	double		xsc,		// printer dots per PPM
	double		ysc,		// printer dots per PPM
	bool		peakgroupshow,
	bool		peakshow,
	bool		/* blacklabel */)
{
  fprintf(fp, "gsave 0 setlinewidth 0 setgray\n");

  View_Settings s = vp->settings();
  if (s.show_ornaments)
    {
	if (s.show_grids) {
		print_ornaments(grid, vp, rect, xsc, ysc, fp);
	}

	if (peakshow && s.show_peaks) {
		fprintf(fp, "gsave PEAKLINEWIDTH setlinewidth\n");
		print_ornaments(peak, vp, rect, xsc, ysc, fp);
		fprintf(fp, "grestore\n");
	}

	if (peakgroupshow && s.show_peakgroups) {
		fprintf(fp, "gsave PEAKLINEWIDTH setlinewidth\n");
		print_ornaments(peakgroup, vp, rect, xsc, ysc, fp);
		fprintf(fp, "grestore\n");
	}

	if (s.show_lines) {
		fprintf(fp, "/LineEndSizeX %f def\n",
			vp->spectrum()->line_end_size() * xsc);
		fprintf(fp, "/LineEndSizeY %f def\n",
			vp->spectrum()->line_end_size() * ysc /vp->aspect());
		print_ornaments(line, vp, rect, xsc, ysc, fp);
	}

	if (s.show_labels) {
	  char *standard = "Standard Width";
	  double width, ascent, descent;
	  vp->text_size(standard, &width, &ascent, &descent);
	  fprintf(fp, "LabelFont findfont setfont\n");
	  fprintf(fp,
		  "currentfont %f (%s) stringwidth pop div"
		  " scalefont setfont\n",
		  width * xsc, standard);
	  print_ornaments(label, vp, rect, xsc, ysc, fp);
	}
      }

  fprintf(fp, "grestore\n");
}

// ----------------------------------------------------------------------------
//
static void print_ornaments(Ornament_Type type, View *view,
			    const Rectangle &rect,	// in PPM
			    double xsc, double ysc,	// dots per PPM
			    FILE *fp)
{
  List olist = view->spectrum()->ornaments(type);
  for (int oi = 0 ; oi < olist.size() ; ++oi)
    ((Ornament *) olist[oi])->print(view->ornament_drawing_routines(),
				    fp, xsc, ysc, rect);
}

// ----------------------------------------------------------------------------
// Write postscript to print a slice panel
//
static void slice_print(FILE *fp, View *view, Axis a,
			const SRegion &region,
			double scale,	// ppm to print units
			int thickness)	// print units
{
  Slice *s = view->slice(a);
  int axis = view->axis(a);
  double pmin = region.min[axis];
  double pmax = region.max[axis];
  double hmin, hmax;
  s->height(&hmin, &hmax);
  int i1, i2;
  float *sdata, *fdata, *seldata, *ddata;
  s->get_slice_data(pmin, pmax, &i1, &i2, &sdata, &fdata, &seldata, &ddata);
  int len = i2 - i1 + 1;

  double iscale = scale * view->scale(1.0, a, INDEX, PPM);
  double ioffset = (view->map((double) i1, a, INDEX, PPM) - pmax) * scale;
  double yscale = 0.90 * thickness / (hmax - hmin);
  double yoffset = hmin * yscale - thickness * 0.05;

  fprintf(fp, (a == X ? "xslice_init\n" :  "yslice_init\n"));

  slice_range_labels(fp, hmin, hmax, yoffset, yscale);

  // Set up clipping
  fprintf(fp, "0 0 %s SLICE rectpath clip newpath\n",
	  (a == X ? "WIDTH" : "HEIGHT"));

  // Draw the zero line, fit peaks curve and spectrum data curve.
  fprintf(fp, "[DASH dup add] 0 setdash\n");
  fprintf(fp, "0 %d M %s x stroke\n",
	  slice_print_position(0, yoffset, yscale),
	  (a == X ? "WIDTH" : "HEIGHT"));
  fprintf(fp, "[DASH] 0 setdash\n");
  draw_slice_curve(fp, iscale, ioffset, yscale, yoffset, fdata, len);
  fprintf(fp, "[] 0 setdash\n");
  draw_slice_curve(fp, iscale, ioffset, yscale, yoffset, sdata, len);

  fprintf(fp, (a == X ? "xslice_done\n" :  "yslice_done\n"));

  delete [] sdata;
  delete [] fdata;
  delete [] seldata;
  delete [] ddata;
}

// ----------------------------------------------------------------------------
// Show labels for range of data heights and zero position.
//
static void slice_range_labels(FILE *fp, double hmin, double hmax,
			       double yoffset, double yscale)
{
	fprintf(fp, "(%.1e) 0 %d tickrshow\n",
		hmin, slice_print_position(hmin, yoffset, yscale));
	if (slice_print_position(0, yoffset, yscale) >
	    slice_print_position(hmin, yoffset, yscale) + 12 &&
	    slice_print_position(0, yoffset, yscale) <
	    slice_print_position(hmax, yoffset, yscale) - 12)
	  fprintf(fp, "(0) 0 %d tickrshow\n",
		  slice_print_position(0, yoffset, yscale));
	fprintf(fp, "(%.1e) 0 %d tickrshow\n",
		hmax, slice_print_position(hmax, yoffset, yscale));
}

/*
 * Draw the slice defined in <data> to file <fp>
 */
static void draw_slice_curve(
	FILE		*fp,
	double		iscale,		// index to print units
	double		ioffset,
	double		yscale,		// data height to print units
	double		yoffset,	// data height offset
	float		*data,
	int		len)		// the number of data points
{
	for (int i = 0; i < len; i++) {
		fprintf(fp, "%d %d %s\n",
			slice_print_position(i, ioffset, iscale),
			slice_print_position(data[i], yoffset, yscale),
			(i == 0 ? "M" : "L"));
	}
	if (len > 0)
		fputs("stroke\n", fp);
}

// ----------------------------------------------------------------------------
//
static int slice_print_position(double x, double offset, double scale)
  { return (int) (x * scale - offset + 0.5); }

// ----------------------------------------------------------------------------
//
int color_index(const Color &color)
{
  for (int ci = 0 ; print_color_list[ci] ; ++ci)
    if (color.name() == print_color_list[ci])
      return ci;

  return 0;				// index for black 
}

