// ----------------------------------------------------------------------------
// Convert NMR data from NIH format (produced by nmrPipe) to UCSF format.
//
#include <stdio.h>		// use stderr
#include <stdlib.h>		// use exit()
#include <string.h>		// Use strlen()

#include "memcache.h"		// use Memory_Cache
#include "nihfile.h"		// Use nih_nmr_data()
#include "nmrdata.h"		// Use NMR_Data
#include "spoint.h"		// Use IPoint
#include "stringc.h"		// Use Stringy
#include "ucsffile.h"		// Use write_ucsf_nmr_data()

#define MEMORY_CACHE_SIZE 8000000

static bool override_axis_order(char *axis_order, IPoint *new_order);

// ----------------------------------------------------------------------------
//
static char *usage = 
"Syntax: pipe2ucsf [-<axis-order>] <pipe-file> <ucsf-file>\n"
"  where <axis-order> is a string of digits (eg. 312)\n"
"\n"
"Example:\n"
"\n"
"        % pipe2ucsf noesy150.pipe noesy150.ucsf\n"
"\n"
"The nmrPipe file can be 2D, 3D, or 4D data.  The nmrPipe data must\n"
"be in a single file.  See documentation for the xyz2pipe command that\n"
"comes with nmrPipe for how to combine planes of a 3D or 4D spectrum\n"
"into a single file. For more information see the Sparky documentation\n"
"manual/files.html.\n";

// ----------------------------------------------------------------------------
//
int main(int argc, char **argv)
{
	char *infile = NULL, *outfile = NULL, *axis_order = NULL;

	if(argc == 3) {
	  infile = argv[1];
	  outfile = argv[2];
	} else if (argc == 4) {
	  axis_order = argv[1];
	  infile = argv[2];
	  outfile = argv[3];
	} else
	  { fputs(usage, stderr); exit(1); }

	Stringy error_msg;
	Memory_Cache mcache(MEMORY_CACHE_SIZE);
	NMR_Data *nih_data = nih_nmr_data(infile, &error_msg, &mcache);
	if (nih_data == NULL)
	  {
	    fprintf(stderr, "pipe2ucsf: %s\n", error_msg.cstring());
	    exit(1);
	  }

	if (axis_order)
	  {
	    IPoint new_order(nih_data->dimension());
	    if (override_axis_order(axis_order, &new_order))
	      nih_data = permute_nmr_axes(nih_data, new_order);
	    else
	      { fputs(usage, stderr); exit(1); }
	  }

	Stringy err_msg;
	if (!write_ucsf_nmr_data(nih_data, outfile, &err_msg, &mcache))
	  {
	    fprintf(stderr, "pipe2ucsf: %s\n", err_msg.cstring());
	    exit(1);
	  }
}

// ----------------------------------------------------------------------------
//
static bool override_axis_order(char *axis_order, IPoint *new_order)
{
  int dim = new_order->dimension();
  if (axis_order[0] != '-' || (int) strlen(axis_order) != dim + 1)
    return false;

  for (int d = 0 ; d < dim ; ++d)
    {
      int a = axis_order[d+1] - '1';
      if (a < 0 || a >= dim)
	return false;
      for (int dprev = 0 ; dprev < d ; ++dprev)
	if (a == (*new_order)[dprev])
	  return false;
      (*new_order)[d] = a;
    }

  return true;
}
