// ----------------------------------------------------------------------------
// Create a UCSF format spectrum file containing a specified list of
// Gaussian peaks.
//
#include <iostream>		// use cin, cerr
using std::istream;

#include <math.h>		// use exp()
#include <stdlib.h>		// use atoi(), exit()

#include "blockfile.h"		// use Block_File
#include "list.h"		// use List
#include "mathconst.h"		// use MC_LN2
#include "memalloc.h"		// use new()
#include "memcache.h"		// use Memory_Cache
#include "nmrdata.h"		// use NMR_Data
#include "spoint.h"		// use IPoint, SPoint
#include "stringc.h"		// use Stringy
#include "subarray.h"		// use Subarray_Iterator
#include "ucsffile.h"		// use write_ucsf_nmr_data()

#define MEMORY_CACHE_SIZE 8000000

// ----------------------------------------------------------------------------
//
class gaussian_parameters
{
public:
  gaussian_parameters(const SPoint &position, double height,
		      const SPoint &linewidth);

  double value(const SPoint &p);
  SRegion region(double sd_range);

  SPoint position, linewidth;
  double height;
};

// ----------------------------------------------------------------------------
//
class Peak_NMR_Data : public NMR_Data
{
public:
  Peak_NMR_Data(const IPoint &size, const SPoint &origin,
		const List &nuclei, const SPoint &freq, const SPoint &swidth,
		const List &gaussians, Memory_Cache *);
  virtual ~Peak_NMR_Data();

  virtual bool read_value(const IPoint &position, float *value);
  virtual bool read_values(const IRegion &region, float *values);

private:
  Block_File *bf;

  IPoint block_size(const IPoint &size);
};

// ----------------------------------------------------------------------------
//
class Peak_Block_Data : public Block_File
{
public:
  Peak_Block_Data(const IPoint &size, const IPoint &block_size,
		  const List &gaussians, Memory_Cache *);
  virtual ~Peak_Block_Data();

  virtual bool read_block(const IPoint &block, float *values);
  virtual bool write_block(const IPoint &block, float *values);
  virtual int cache_size();
private:
  List gaussians;
};

// ----------------------------------------------------------------------------
//
static NMR_Data *peak_spectrum(istream &params, Stringy *err_msg,
			       Memory_Cache *);
static bool read_parameters(istream &params, IPoint *size,
			    SPoint *origin, List *nuclei, SPoint *freq,
			    SPoint *swidth, List *gaussians, Stringy *err_msg);static Stringy parameter_line(istream &params);
static bool read_dimension(const Stringy &line, int *dim, Stringy *err_msg);
static bool read_size(const Stringy &line, int dim,
		      IPoint *size, Stringy *err_msg);
static bool read_origin(const Stringy &line, int dim,
			SPoint *origin, Stringy *err_msg);
static bool read_nuclei(const Stringy &line, int dim,
			List *nuclei, Stringy *err_msg);
static bool read_spectrometer_frequency(const Stringy &line, int dim,
					SPoint *freq, Stringy *err_msg);
static bool read_spectral_width(const Stringy &line, int dim,
				SPoint *swidth, Stringy *err_msg);
static IPoint read_ipoint(const Stringy &line, int dim);
static SPoint read_spoint(const Stringy &line, int dim);
static bool read_gaussians(istream &params,
			   const IPoint &size, const SPoint &sfreq,
			   const SPoint &swidth, const SPoint &origin,
			   List *gaussians, Stringy *err_msg);

// ----------------------------------------------------------------------------
//
static char *usage =
"Syntax: peaks2ucsf output-file < parameter-file\n"
"\n"
"Creates a UCSF format spectrum file from a list of Gaussian peaks.\n"
"The parameter file has the following format.\n"
"\n"
"2				# dimension\n"
"1024 2048			# matrix size\n"
"8.37 9.21			# ppm at index 0,0\n"
"H H				# nuclei (H, N or C)\n"
"500.123 500.123		# nucleus frequencies (MHz)\n"
"4000.732 5200.183		# spectral widths (Hz)\n"
"4.37 2.15 1.95e6 13.2 15.1	# Gaussian center (ppm), height, linewidth (Hz)\n"
"    .\n"
"    .\n"
"    .\n";

// ----------------------------------------------------------------------------
//
int main(int argc, char **argv)
{
  if (argc != 2)
    { std::cerr << usage; exit(1); }

  Stringy err_msg;
  Memory_Cache mcache(MEMORY_CACHE_SIZE);
  NMR_Data *nmr_data = peak_spectrum(std::cin, &err_msg, &mcache);
  if (nmr_data == NULL)
    {
      std::cerr << "peaks2ucsf: " << err_msg.cstring() << std::endl;
      exit(1);
    }
  
  Stringy outfile = argv[1];
  if (!write_ucsf_nmr_data(nmr_data, outfile, &err_msg, &mcache))
    {
      std::cerr << "peaks2ucsf: " << err_msg.cstring() << std::endl;
      exit(1);
    }

  return 0;
}

// ----------------------------------------------------------------------------
//
static NMR_Data *peak_spectrum(istream &params, Stringy *err_msg,
			       Memory_Cache *mcache)
{
  IPoint size;
  SPoint origin, freq, swidth;
  List nuclei;
  List gaussians;

  if (! read_parameters(params, &size, &origin, &nuclei,
			&freq, &swidth, &gaussians, err_msg))
    return NULL;

  return new Peak_NMR_Data(size, origin, nuclei, freq, swidth, gaussians,
			   mcache);
}

// ----------------------------------------------------------------------------
//
static bool read_parameters(istream &params, IPoint *size,
			    SPoint *origin, List *nuclei, SPoint *freq,
			    SPoint *swidth, List *gaussians, Stringy *err_msg)
{
  int dim;
  return (read_dimension(parameter_line(params), &dim, err_msg) &&
	  read_size(parameter_line(params), dim, size, err_msg) &&
	  read_origin(parameter_line(params), dim, origin, err_msg) &&
	  read_nuclei(parameter_line(params), dim, nuclei, err_msg) &&
	  read_spectrometer_frequency(parameter_line(params), dim,
				      freq, err_msg) &&
	  read_spectral_width(parameter_line(params), dim, swidth, err_msg) &&
	  read_gaussians(params, *size, *freq, *swidth, *origin,
			 gaussians, err_msg));
}

// ----------------------------------------------------------------------------
// Skip blank lines and comment lines beginning with #
//
static Stringy parameter_line(istream &params)
{
  Stringy line;
  while (stream_line(params, &line))
    {
      bool comment = (line.is_empty() || line[0] == '#');
      if (!comment)
	return line;
    }
  return "";
}

// ----------------------------------------------------------------------------
//
static bool read_dimension(const Stringy &line, int *dim, Stringy *err_msg)
{
  int d = atoi(line.cstring());
  if (d < 1 || d > 4)
    { *err_msg = formatted_string("Bad dimension %d", d); return false; }
  *dim = d;
  return true;
}

// ----------------------------------------------------------------------------
//
static bool read_size(const Stringy &line, int dim,
		      IPoint *size, Stringy *err_msg)
{
  IPoint s = read_ipoint(line, dim);
  for (int a = 0 ; a < dim ; ++a)
    if (s[a] <= 0)
      { *err_msg = "Bad size: " + line; return false; }
  *size = s;
  return true;
}

// ----------------------------------------------------------------------------
//
static bool read_origin(const Stringy &line, int dim,
			SPoint *origin, Stringy *)
{
  *origin = read_spoint(line, dim);
  return true;
}

// ----------------------------------------------------------------------------
//
static bool read_nuclei(const Stringy &line, int dim,
			List *nuclei, Stringy *)
{
  Stringy nucleus_line = line;
  for (int a = 0 ; a < dim ; ++a)
    {
      Stringy nucleus = first_token(nucleus_line, &nucleus_line);
      nucleus = standard_nucleus_name(nucleus);
      nuclei->append(new Stringy(nucleus));
    }
  return true;
}

// ----------------------------------------------------------------------------
//
static bool read_spectrometer_frequency(const Stringy &line, int dim,
					SPoint *freq, Stringy *err_msg)
{
  SPoint f = read_spoint(line, dim);
  for (int a = 0 ; a < dim ; ++a)
    if (f[a] <= 0)
      {	*err_msg = "Bad spectrometer frequency: " + line; return false; }
  *freq = f;
  return true;
}

// ----------------------------------------------------------------------------
//
static bool read_spectral_width(const Stringy &line, int dim,
				SPoint *swidth, Stringy *err_msg)
{
  SPoint sw = read_spoint(line, dim);
  for (int a = 0 ; a < dim ; ++a)
    if (sw[a] <= 0)
      {	*err_msg = "Bad spectral width: " + line; return false; }
  *swidth = sw;
  return true;
}

// ----------------------------------------------------------------------------
//
static IPoint read_ipoint(const Stringy &line, int dim)
{
  Stringy iline = line;
  IPoint i(dim);
  for (int a = 0 ; a < dim ; ++a)
    {
      Stringy e = first_token(iline, &iline);
      i[a] = atoi(e.cstring());
    }
  return i;
}

// ----------------------------------------------------------------------------
//
static SPoint read_spoint(const Stringy &line, int dim)
{
  Stringy sline = line;
  SPoint s(dim);
  for (int a = 0 ; a < dim ; ++a)
    {
      Stringy e = first_token(sline, &sline);
      s[a] = atof(e.cstring());
    }
  return s;
}

// ----------------------------------------------------------------------------
// Return Gaussians with positions and linewidths in index units.
//
static bool read_gaussians(istream &params,
			   const IPoint &size, const SPoint &sfreq,
			   const SPoint &swidth, const SPoint &origin,
			   List *gaussians, Stringy *)
{
  int dim = size.dimension();
  SPoint index_per_ppm(dim), index_per_hz(dim);
  for (int a = 0 ; a < dim ; ++a)
    {
      index_per_hz[a] = size[a] / swidth[a];
      double ppm_width = swidth[a] / sfreq[a];
      index_per_ppm[a] = size[a] / ppm_width;
    }

  double half_height_to_gaussian_linewidth = 1 / (2 * sqrt(MC_LN2));
  Stringy line;
  while ((line = parameter_line(params), !line.is_empty()))
    {
      SPoint position(dim), linewidth(dim);
      for (int a = 0 ; a < dim ; ++a)
	position[a] = atof(first_token(line, &line).cstring());
      position = (origin - position) * index_per_ppm;
      double height = atof(first_token(line, &line).cstring());
      for (int a = 0 ; a < dim ; ++a)
	linewidth[a] = atof(first_token(line, &line).cstring());
      linewidth = linewidth * index_per_hz;
      linewidth = linewidth * half_height_to_gaussian_linewidth;
      gaussians->append(new gaussian_parameters(position, height, linewidth));
    }
  return true;
}

// ----------------------------------------------------------------------------
//
gaussian_parameters::gaussian_parameters(const SPoint &position, double height,
					 const SPoint &linewidth)
{
  this->position = position;
  this->height = height;
  this->linewidth = linewidth;
}

// ----------------------------------------------------------------------------
//
double gaussian_parameters::value(const SPoint &p)
{
  double  v = height;
  for (int a = 0 ; a < position.dimension() ; ++a)
    {
      double delta = (p[a] - position[a]) / linewidth[a];
      v *= exp(-delta * delta);
    }
  return v;
}

// ----------------------------------------------------------------------------
//
SRegion gaussian_parameters::region(double sd_range)
{
  return SRegion(position - (linewidth*sd_range),
		 position + (linewidth*sd_range));
}

// ----------------------------------------------------------------------------
//
Peak_NMR_Data::Peak_NMR_Data(const IPoint &size,
			     const SPoint &origin,
			     const List &nuclei,
			     const SPoint &freq,
			     const SPoint &swidth,
			     const List &gaussians,
			     Memory_Cache *mcache)
  : NMR_Data("", size, block_size(size), freq, swidth, origin, nuclei)
{
  bf = new Peak_Block_Data(size, block_size(size), gaussians, mcache);
}

// ----------------------------------------------------------------------------
//
Peak_NMR_Data::~Peak_NMR_Data()
{
  delete bf;
  bf = NULL;
}

// ----------------------------------------------------------------------------
//
bool Peak_NMR_Data::read_value(const IPoint &position, float *value)
  { return bf->read_value(position, value); }
bool Peak_NMR_Data::read_values(const IRegion &region, float *values)
  { return bf->read_values(region, values); }

// ----------------------------------------------------------------------------
//
#define BLOCK_POINT_LIMIT (1 << 22)
IPoint Peak_NMR_Data::block_size(const IPoint &size)
{
  IPoint bsize = size;
  while (bsize.product() > BLOCK_POINT_LIMIT)
    for (int a = 0 ; a < size.dimension() ; ++a)
      bsize[a] = bsize[a] / 2;
  return bsize;
}

// ----------------------------------------------------------------------------
//
Peak_Block_Data::Peak_Block_Data(const IPoint &size, const IPoint &block_size,
				 const List &gaussians, Memory_Cache *mcache)
  : Block_File("", size, block_size, mcache)
{
  this->gaussians = gaussians;
}

// ----------------------------------------------------------------------------
//
Peak_Block_Data::~Peak_Block_Data()
{
}

// ----------------------------------------------------------------------------
//
#define MAX_RANGE_SD 5
bool Peak_Block_Data::read_block(const IPoint &block, float *values)
{
  IRegion region = block_region(block);

  offset_type volume = block_volume();
  for (offset_type k = 0 ; k < volume ; ++k)
    values[k] = 0;

  for (int gi = 0 ; gi < gaussians.size() ; ++gi)
    {
      gaussian_parameters *g = (gaussian_parameters *) gaussians[gi];
      IRegion gr = g->region(MAX_RANGE_SD).rounded();
      gr.clip(region);
      if (!gr.empty())
	{
	  Subarray_Iterator ri(gr, region, true);
	  for ( ; !ri.finished() ; ri.next())
	    values[ri.index()] += g->value(ri.position());
	}
    }

  return true;
}

// ----------------------------------------------------------------------------
//
bool Peak_Block_Data::write_block(const IPoint &, float *)
{
  return false;		// Not implemented.
}

// ----------------------------------------------------------------------------
//
int Peak_Block_Data::cache_size()
{
 return 1;
}
