// ----------------------------------------------------------------------------
// Program for read a data matrix of big-endian floats and create a UCSF
// format NMR file.  The header parameters are supplied as command line
// arguments.
//

#include <iostream>		// use cerr
#include <stdio.h>		// Use sscanf()
#include <stdlib.h>		// Use exit()
#include <string.h>		// Use strcmp()

#include "blockfile.h"		// use Block_File
#include "memalloc.h"		// use new()
#include "memcache.h"		// use Memory_Cache
#include "nmrdata.h"		// Use NMR_Data
#include "spoint.h"		// Use IPoint, SPoint
#include "stringc.h"		// Use Stringy
#include "ucsffile.h"		// Use write_ucsf_nmr_data()

#define MEMORY_CACHE_SIZE 8000000

// ----------------------------------------------------------------------------
// Command line options.
//
struct Options
{
  Stringy matrix_input;
  Stringy ucsf_output;

  IPoint size;
  SPoint frequency;
  SPoint spectral_width;
  SPoint origin;
  List axis_names;
};

// ----------------------------------------------------------------------------
//
static bool parse_args(int argc, char **argv, Options *opt);
static NMR_Data *nmr_data_out(const Options &options, Memory_Cache *mcache);

// ----------------------------------------------------------------------------
//
const char *use =
"Usage: matrix2ucsf -n1 # -n2 # [-n3 #] [-n4 #]\n"
"                [-f1 #] [-f2 #] [-f3 #] [-f4 #]\n"
"                [-sw1 #] [-sw2 #] [-sw3 #] [-sw4 #]\n"
"                [-o1 #] [-o2 #] [-o3 #] [-o4 #]\n"
"                [-a1 name] [-a2 name] [-a3 name] [-a4 name]\n"
"                -o output-file input-file\n"
"\n"
"Creates a UCSF format NMR file from a matrix of floating point values.\n"
"On reading through the matrix file the w1 index should vary slowest.\n"
"The floating point numbers must have big-endian byte order.\n"
"Header parameters are supplied on the command line.\n"
"\n"
"  -nN size\n"
"	specifies matrix size for axis N\n"
"  -fN freq\n"
"	spectrometer frequency in Mhz for axis N\n"
"  -swN sweepwidth\n"
"	sweep width in Hz for axis N\n"
"  -oN origin\n"
"	position of matrix index 0 in ppm for axis N\n"
"  -aN axis-nucleus-name\n"
"	nucleus name (eg 1H, 15N, 13C) for axis N\n";

// ----------------------------------------------------------------------------
//
int main(int argc, char **argv)
{
  Options options;
  if (!parse_args(argc, argv, &options))
    { std::cerr << use; exit(1); }

  Memory_Cache mcache(MEMORY_CACHE_SIZE);
  NMR_Data *nmr_data = nmr_data_out(options, &mcache);
  if (nmr_data == NULL)
    { std::cerr << use; exit(1); }

  if (options.ucsf_output.is_empty())
    { std::cerr << use; exit(1); }

  Stringy err_msg;
  if (!write_ucsf_nmr_data(nmr_data, options.ucsf_output, &err_msg, &mcache))
    { std::cerr << "ucsfdata: " << err_msg.cstring() << std::endl; exit(1); }

  delete nmr_data;

  return 0;
}

// ----------------------------------------------------------------------------
//
static bool parse_args(int argc, char **argv, Options *options)
{
  int d, size[DIM];
  Stringy output_path, input_path, axis_name[DIM];
  double origin[DIM], spectral_width[DIM], frequency[DIM];

  for (int a = 0 ; a < DIM ; ++a)
    {
      size[a] = 0;
      origin[a] = 0;
      spectral_width[a] = 1;
      frequency[a] = 1;
    }

  for (int a = 1 ; a < argc ; ++a)
    if (sscanf(argv[a], "-n%d", &d) == 1 && d >= 1 && d <= DIM)
      if (a+1 < argc)
	{ a += 1; size[d-1] = atoi(argv[a]); }
      else
	return false;
    else if (sscanf(argv[a], "-f%d", &d) == 1 && d >= 1 && d <= DIM)
      if (a+1 < argc)
	{ a += 1; frequency[d-1] = atof(argv[a]); }
      else
	return false;
    else if (sscanf(argv[a], "-sw%d", &d) == 1 && d >= 1 && d <= DIM)
      if (a+1 < argc)
	{ a += 1; spectral_width[d-1] = atof(argv[a]); }
      else
	return false;
    else if (sscanf(argv[a], "-o%d", &d) == 1 && d >= 1 && d <= DIM)
      if (a+1 < argc)
	{ a += 1; origin[d-1] = atof(argv[a]); }
      else
	return false;
    else if (sscanf(argv[a], "-a%d", &d) == 1 && d >= 1 && d <= DIM)
      if (a+1 < argc)
	{ a += 1; axis_name[d-1] = argv[a]; }
      else
	return false;
    else if (strcmp(argv[a], "-o") == 0)
      if (a+1 < argc) { a += 1; output_path = argv[a]; } else return false;
    else if (a == argc - 1)
      input_path = argv[a];
    else
      return false;

  int dim;
  for (dim = 0 ; dim < DIM ; ++dim)
    if (size[dim] == 0)
      break;

  options->matrix_input = input_path;
  options->ucsf_output = output_path;

  options->size = IPoint(dim, size);
  options->frequency = SPoint(dim, frequency);
  options->spectral_width = SPoint(dim, spectral_width);
  options->origin = SPoint(dim, origin);
  for (int a = 0 ; a < dim ; ++a)
    options->axis_names.append(new Stringy(axis_name[a]));

  return true;
}

// ----------------------------------------------------------------------------
//
static NMR_Data *nmr_data_out(const Options &options, Memory_Cache *mcache)
{
  Block_File *bf = packed_block_file(options.matrix_input, "r",
				     options.size, options.size,
				     0, true,
				     mcache);
  if (bf == NULL)
    return NULL;

  NMR_Data *nmr_output = block_file_nmr_data(bf,
					     options.frequency,
					     options.spectral_width,
					     options.origin,
					     options.axis_names);
  return nmr_output;
}
