/*
 * Grid.C:		Implementation of the Grid ornament class.
 *
 * A Grid is an ornament centered at a point (the center is not so important)
 * and extending essentially to infinity parallel to one axis.
 */
#include <stdlib.h>		// Use atoi()

#include "axismap.h"		// Use Axis_Map
#include "color.h"
#include "grid.h"
#include "memalloc.h"		// use new()
#include "notifier.h"		// use Notifier
#include "print.h"		// use color_index()
#include "spectrum.h"		// Use Spectrum

//
// Constructors
//
// Call the superclass constructor
//
Grid::Grid(Spectrum *sp, const SPoint &p, int axis) : Ornament(sp)
{
  this->axis = axis;
  location = p;
  SetColor("white");

  sp->notifier().send_notice(nt_created_ornament, (Ornament *) this);
}

// ----------------------------------------------------------------------------
//
Grid::~Grid()
{
  Notifier &n = spectrum()->notifier();
  n.send_notice(nt_will_delete_ornament, (Ornament *) this);
  n.send_notice(nt_will_delete_grid, this);
}

// ----------------------------------------------------------------------------
//
Ornament *Grid::copy(Spectrum *to, const Axis_Map &axismap)
{
  SPoint loc = axismap.map(location);
  int to_axis = axismap.map(axis);
  Grid *grid = new Grid(to, loc, to_axis);

  grid->lock(IsLocked());
  grid->SetNote(GetNote());
  grid->SetColor(GetColor());

  return grid;
}

// ----------------------------------------------------------------------------
//
Ornament_Type Grid::type() const { return ::grid; }
const char *Grid::type_name() const { return "grid"; }

//
// Return the bounding volume of the grid.
//
SRegion Grid::erasure_region(ODraw &dr) const
{
  SRegion r(location, location);

  r.min[axis] = 0;
  r.max[axis] = spectrum()->ppm_region().max[axis];

  add_selection_padding(&r, dr);

  return r;
}

//
// Move the ornament by <dr>
//
void Grid::IncrementLocation(const SPoint &dr)
{
  Notifier &n = spectrum()->notifier();
  n.send_notice(nt_will_change_ornament, (Ornament *) this);
  location += dr;
  n.send_notice(nt_changed_ornament, (Ornament *) this);
}

// ----------------------------------------------------------------------------
// Draw a grid line.
//
void Grid::display(ODraw &dr) const
{
  SRegion r(location, location);
  SRegion full = spectrum()->ppm_region();
  r.min[axis] = full.min[axis];
  r.max[axis] = full.max[axis];

  if (r.intersect(dr.clip_region()))
    {
      dr.set_drawing_color(GetColor());
      dr.draw_line(r.min[dr.axis(X)], r.min[dr.axis(Y)],
		   r.max[dr.axis(X)], r.max[dr.axis(Y)]);

      Ornament::display(dr);	// Handle selection highlighting.
    }
}

// ----------------------------------------------------------------------------
// Send a Postscript command to draw a grid line.
//
void Grid::print(ODraw &dr, FILE *fp,
		 double xsc, double ysc, Rectangle r) const
{
  if (!dr.is_visible(this, r))
    return;

  int color;
  double xdata, ydata, x1, y1, x2, y2;

  dr.xy_point(location, &xdata, &ydata);
  color = color_index(GetColor());

  if (axis == dr.axis(X))			// Horizontal line
    {
      x1 = 0;
      x2 = r.size(X) * xsc;
      y1 = y2 = (r.max(Y) - ydata) * ysc;
    }
  else if (axis == dr.axis(Y))		// Vertical line
    {
      x1 = x2 = (r.max(X) - xdata) * xsc;
      y1 = 0;
      y2 = r.size(Y) * ysc;
    }
  else
    return;

  fprintf(fp, "%d %f %f %f %f PG\n", color, x1, y1, x2-x1, y2-y1);
}
