/*
 * ContourPlane.C:  	Implementation of the ContourPlane class
 *
 * A contour plane is a 2-dimensional plane, located at <mBL>, <mTR>.
 * Each contour planes stores the parameters under which plane was contoured.
 *
 * When a ContourPlane is asked to display its contours, it determines
 * whether its parameters match the view's contour parameters. If they
 * do, the ContourPlane just displays its contours. Otherwise, the
 * ContourPlane removes its old contours, recomputes the new contours
 * then displays its contours.
 */

#include <float.h>		// use FLT_MAX

#include "contourplane.h"
#include "contourstream.h"
#include "dataregion.h"		// Use Data_Region
#include "linefit.h"		// use add_fit_peaks()
#include "memalloc.h"		// use new()
#include "spectrum.h"		// use Spectrum
#include "spoint.h"		// Use IRegion
#include "table.h"		// Use hash_combine()
#include "utility.h"		// Use fatal_error()

static bool contour(Data_Region &dr, const Contour_Levels &, ContourStream *);
static void contour_path_cb(float level, int length,
			    float *path_x, float *path_y, void *stream);
static unsigned long hash_region(const SRegion &r);
static unsigned long hash_contour_levels(const Contour_Levels &);

// ----------------------------------------------------------------------------
//
ContourPlane::ContourPlane(Spectrum *spectrum, const IRegion &tile,
			   const Contour_Levels &pos,
			   const Contour_Levels &neg,
			   bool subtract_fit_peaks)
{
  mSpectrum = spectrum;
  mRegion = tile;
  mPosLevels = pos;
  mNegLevels = neg;
  mSubtractPeaks = subtract_fit_peaks;
  mStream = NULL;
}

ContourPlane::~ContourPlane()
{
  if (mStream)
    delete mStream;
}

// ----------------------------------------------------------------------------
//
void ContourPlane::display(Drawing_Routines &dr)
{
  compute();

  mStream->display(dr);
}

void ContourPlane::compute()
{
  if (mStream)
    return;

	Data_Region	plane(spectrum(), region());

	mStream = new ContourStream;

	if (mSubtractPeaks)
	  add_fit_peaks(spectrum()->peaklets(), -1, region(), plane.data());

	contour(plane, mPosLevels, mStream);
	contour(plane, mNegLevels, mStream);

	mStream->free_extra_memory();
}

// ----------------------------------------------------------------------------
//
#define data_index(r, c) ((c)*h + (r))
#define is_crossable(r, c, vertical) \
  (((vertical) ? vert_crossable : horz_crossable)[data_index((r),(c))])
#define set_crossable(r, c, vertical, state) \
  (((vertical) ? vert_crossable : horz_crossable)[data_index((r),(c))] = state)

class calculate_contours
{
public:

  calculate_contours(Data_Region &dr,
		     void (*path_cb)(float level, int length,
				     float *x, float *y,
				     void *cb_data),
		     void *cb_data);
  ~calculate_contours();
  void compute_paths(float level);

private:

  const float *data;
  int w, h;
  float x_offset, y_offset;
  void (*path_cb)(float level, int length, float *x, float *y, void *cb_data);
  void *cb_data;
  bool *above, *horz_crossable, *vert_crossable;
  float *path_x, *path_y;
  int path_length;

/*
  int data_index(int r, int c)
    { return c*h + r; }
  bool is_crossable(int r, int c, bool vertical)
    { return (vertical ? vert_crossable : horz_crossable)[data_index(r,c)]; }
  void set_crossable(int r, int c, bool vertical, bool state)
    { (vertical ? vert_crossable : horz_crossable)[data_index(r,c)] = state; }
*/
  bool exiting_boundary(int r, int c, bool vertical, bool right)
    { return (vertical ?
	      (right ? c == w-1 : c == 0) :
	      (right ? r == h-1 : r == 0)); }

  void mark_crossable(float level);
  void mark_above(float level);
  void trace_contour(float level, int r, int c, bool vertical, bool right);
  void add_edge_crossing(float level, int r, int c, bool vertical);
  bool next_edge_crossing(int *r, int *c, bool *vertical, bool *right);
};

// ----------------------------------------------------------------------------
//
calculate_contours::calculate_contours(Data_Region &dr,
				       void (*path_cb)(float level, int length,
						       float *x, float *y,
						       void *cb_data),
				       void *cb_data)
{
  IRegion region = dr.region();
  int axis1, axis2;
  if (!region.plane_axes(&axis1, &axis2))
    fatal_error("calculate_contours(): Region is not 2 dimensional.\n");

  this->data = dr.data();
  this->w = region.size(axis1);
  this->h = region.size(axis2);
  this->x_offset = region.min[axis1];
  this->y_offset = region.min[axis2];
  this->path_cb = path_cb;
  this->cb_data = cb_data;

  this->above = new bool [w*h];
  this->horz_crossable = new bool [w*h];
  this->vert_crossable = new bool [w*h];
  this->path_x = new float [2*w*h];
  this->path_y = new float [2*w*h];
}

// ----------------------------------------------------------------------------
//
calculate_contours::~calculate_contours()
{
  delete [] above;
  delete [] horz_crossable;
  delete [] vert_crossable;
  delete [] path_x;
  delete [] path_y;
}

// ----------------------------------------------------------------------------
//
void calculate_contours::compute_paths(float level)
{
  mark_crossable(level);

  //
  // Contours entering top and bottom sides.
  //
  for (int c = 0 ; c < w-1 ; ++c)
    {
      trace_contour(level, 0, c, false, true);
      trace_contour(level, h-1, c, false, false);
    }

  //
  // Contours entering left and right sides.
  //
  for (int r = 0 ; r < h-1 ; ++r)
    {
      trace_contour(level, r, 0, true, true);
      trace_contour(level, r, w-1, true, false);
    }

  //
  // Internal contours starting on horizontal edges.
  //
  for (int r = 1 ; r < h-1 ; ++r)
    for (int c = 0 ; c < w-1 ; ++c)
      if (is_crossable(r, c, false))			// Optimization.
	trace_contour(level, r, c, false, true);
}

// ----------------------------------------------------------------------------
// This code is speed critical.  Iteration using row and column indices
// was replaced with the single index iteration.  Speed up of a factor of 2.
//
void calculate_contours::mark_crossable(float level)
{
  mark_above(level);

  bool *above1 = above, *above2 = above + data_index(0,1);
  int last = data_index(h-1, w-2);
  for (int k = 0 ; k <= last ; ++k)
    horz_crossable[k] = (above1[k] != above2[k]);

  above1 = above;
  above2 = above + data_index(1,0);
  last = data_index(h-2, w-1);
  for (int k = 0 ; k <= last ; ++k)
    vert_crossable[k] = (above1[k] != above2[k]);
}

// ----------------------------------------------------------------------------
// This code is speed critical.  Iteration using row and column indices
// was replaced with the single index iteration.  Speed up of a factor of 2.
//
void calculate_contours::mark_above(float level)
{
  int last = data_index(h-1, w-1);
  for (int k = 0 ; k <= last ; ++k)
    above[k] = (data[k] > level);
}

// ----------------------------------------------------------------------------
// Trace a contour starting from vertical or horizontal edge based at (r,c)
// crossing in the specified direction.
//
void calculate_contours::trace_contour(float level, int r, int c,
				       bool vertical, bool right)
{
  if (!is_crossable(r, c, vertical))
    return;

  path_length = 0;

  int r_start = r, c_start = c;
  bool vertical_start = vertical;

  add_edge_crossing(level, r, c, vertical);
  while (next_edge_crossing(&r, &c, &vertical, &right))
    {
      add_edge_crossing(level, r, c, vertical);
      if (r == r_start && c == c_start && vertical == vertical_start)
	break;
      set_crossable(r, c, vertical, false);
    }
  set_crossable(r_start, c_start, vertical_start, false);

  if (path_length < 2)
    fatal_error("trace_contour(): Dead end contour.\n");

  (*path_cb)(level, path_length, path_x, path_y, cb_data);
}

// ----------------------------------------------------------------------------
//
void calculate_contours::add_edge_crossing(float level,
					   int r, int c, bool vertical)
{
  float x, y;

  if (vertical)
    {
      float a = data[data_index(r,c)];
      float b = data[data_index(r+1,c)];
      x = c;
      y = r + (level - a) / (b - a);
    }
  else
    {
      float a = data[data_index(r,c)];
      float b = data[data_index(r,c+1)];
      x = c + (level - a) / (b - a);
      y = r;
    }

  path_x[path_length] = x + x_offset;
  path_y[path_length] = y + y_offset;
  path_length += 1;
}

// ----------------------------------------------------------------------------
//
bool calculate_contours::next_edge_crossing(int *r, int *c,
					    bool *vertical, bool *right)
{
  if (exiting_boundary(*r, *c, *vertical, *right))
    return false;

  int nr, nc;
  bool nv, nright;
  if (*vertical)
    if (*right)		// Right crossing vertical edge.
      if (is_crossable(*r+1, *c, false))
	{ nr = *r+1; nc = *c; nv = false; nright = true; }
      else if (is_crossable(*r, *c, false))
	{ nr = *r; nc = *c; nv = false; nright = false; }
      else if (is_crossable(*r, *c+1, true))
	{ nr = *r; nc = *c+1; nv = true; nright = true; }
      else
	{
	  // Suppress compiler warning about unitialized var.
	  nr = 0; nc = 0; nv = false; nright = false;
	  fatal_error("next_edge_crossing(): Dead end contour.\n");
	}
    else		// Left crossing vertical edge.
      if (is_crossable(*r, *c-1, false))
	{ nr = *r; nc = *c-1; nv = false; nright = false; }
      else if (is_crossable(*r+1, *c-1, false))
	{ nr = *r+1; nc = *c-1; nv = false; nright = true; }
      else if (is_crossable(*r, *c-1, true))
	{ nr = *r; nc = *c-1; nv = true; nright = false; }
      else
	{
	  // Suppress compiler warning about unitialized var.
	  nr = 0; nc = 0; nv = false; nright = false;
	  fatal_error("next_edge_crossing(): Dead end contour.\n");
	}
  else
    if (*right)		// Up crossing horizontal edge.
      if (is_crossable(*r, *c+1, true))
	{ nr = *r; nc = *c+1; nv = true; nright = true; }
      else if (is_crossable(*r, *c, true))
	{ nr = *r; nc = *c; nv = true; nright = false; }
      else if (is_crossable(*r+1, *c, false))
	{ nr = *r+1; nc = *c; nv = false; nright = true; }
      else
	{
	  // Suppress compiler warning about unitialized var.
	  nr = 0; nc = 0; nv = false; nright = false;
	  fatal_error("next_edge_crossing(): Dead end contour.\n");
	}
    else		// Down crossing horizontal edge.
      if (is_crossable(*r-1, *c, true))
	{ nr = *r-1; nc = *c; nv = true; nright = false; }
      else if (is_crossable(*r-1, *c+1, true))
	{ nr = *r-1; nc = *c+1; nv = true; nright = true; }
      else if (is_crossable(*r-1, *c, false))
	{ nr = *r-1; nc = *c; nv = false; nright = false; }
      else
	{
	  // Suppress compiler warning about unitialized var.
	  nr = 0; nc = 0; nv = false; nright = false;
	  fatal_error("next_edge_crossing(): Dead end contour.\n");
	}

  *r = nr;
  *c = nc;
  *vertical = nv;
  *right = nright;

  return true;
}

// ----------------------------------------------------------------------------
//
static bool contour(Data_Region &dr, const Contour_Levels &levels,
		    ContourStream *stream)
{
	calculate_contours cc(dr, contour_path_cb, stream);
	double	threshold = levels.lowest;
	for (int lev = 0; lev < levels.levels; ++lev)
	  if (threshold >= FLT_MAX || threshold <= - FLT_MAX)
	    break;
	  else
	    {
	      cc.compute_paths(threshold);
	      threshold *= levels.factor;
	    }

	return true;
}

// ----------------------------------------------------------------------------
//
static void contour_path_cb(float level, int length,
			    float *path_x, float *path_y, void *stream)
  { ((ContourStream *)stream)->addContour(level, length, path_x, path_y); }

// ----------------------------------------------------------------------------
//
bool equal_contour_planes(const void *cp1, const void *cp2)
{
  ContourPlane *p1 = (ContourPlane *) cp1;
  ContourPlane *p2 = (ContourPlane *) cp2;

  return (p1->spectrum() == p2->spectrum() &&
	  p1->region() == p2->region() &&
	  p1->positive_levels() == p2->positive_levels() &&
	  p1->negative_levels() == p2->negative_levels());
}

// ----------------------------------------------------------------------------
//
unsigned long contour_plane_hash(const void *cp)
{
  ContourPlane *p = (ContourPlane *) cp; 
  unsigned long h = (unsigned long) p->spectrum();
  h = hash_combine(h, hash_region(p->region()));
  h = hash_combine(h, hash_contour_levels(p->positive_levels()));
  h = hash_combine(h, hash_contour_levels(p->negative_levels()));
  return h;
}

// ----------------------------------------------------------------------------
//
static unsigned long hash_region(const SRegion &r)
{
  unsigned long h = r.dimension();
  for (int a = 0 ; a < r.dimension() ; ++a)
    {
      h = hash_combine(h, (unsigned long) r.min[a]);
      h = hash_combine(h, (unsigned long) r.max[a]);
    }
  return h;
}

// ----------------------------------------------------------------------------
//
static unsigned long hash_contour_levels(const Contour_Levels &levels)
{
  unsigned long h = 0;
  h = hash_combine(h, (unsigned long) levels.levels);
  h = hash_combine(h, (unsigned long) levels.lowest);
  h = hash_combine(h, (unsigned long) (levels.factor * 1000));
  return h;
}

// ----------------------------------------------------------------------------
//
size_t ContourPlane::memory_use()
{
  compute();

  return sizeof(ContourPlane) + mStream->memory_use();
}
