// ----------------------------------------------------------------------------
//
#include <limits.h>		// use INT_MAX
#include <stdarg.h>		// Use ...

#include "assigncopy.h"		// use crossdiagonal_assignment_copy()
#include "axismap.h"		// use Axis_Map
#include "command.h"
#include "condition.h"		// Use Condition
#include "crosspeak.h"		// use CrossPeak
#include "format.h"		// Use save_project()
#include "group.h"		// use Group
#include "integrate.h"		// Use integrate_list()
#include "label.h"		// use Label, label_peaks()
#include "memalloc.h"		// use new()
#include "ornament.h"		// Use Ornament_Type
#include "peakgp.h"		// use PeakGp
#include "peak.h"		// use Peak
#include "project.h"		// use Project
#include "reporter.h"		// use Reporter
#include "resonance.h"		// use Resonance
#include "session.h"		// Use Session, request_coredump()
#include "spectrum.h"		// Use Spectrum
#include "spectrumdata.h"	// Use alias_onto_spectrum()
#include "stringc.h"		// Use Stringy
#include "table.h"		// use Table
#include "uidialogs.h"		// Use show_*_dialog()
#include "uifile.h"		// Use show_*_open_dialog()
#include "uimain.h"		// Use status_line()
#include "uimenu.h"		// Use add_extension_menu_entry()
#include "uimode.h"		// Use show_mode_buttons()
#include "uipeak.h"		// Use show_peak_list_dialog()
#include "uiwait.h"		// Use wait_callbacks()
#include "undo.h"		// Use Undo
#include "uiview.h"		// Use View, View_Settings
#include "winsystem.h"		// use add_timer(), key_pressed()

#define ZOOM_IN_FACTOR .5
#define ZOOM_OUT_FACTOR 2

// ----------------------------------------------------------------------------
//
class Accelerator
{
public:
  Accelerator(Session &s, const Stringy &accel, const Stringy &menu_text,
	      Command_Proc cmd, void *cmd_data);
  ~Accelerator();

  const Stringy &accelerator();

  void invoke();

private:
  Session &session;
  Stringy accel, menu_text;
  Command_Proc cmd;
  void *cmd_data;
};

// ----------------------------------------------------------------------------
//
class Accelerator_List
{
public:
  ~Accelerator_List();

  void add_accelerator(Accelerator *a);
  void remove_accelerator(Accelerator *a);
  Accelerator *find(const Stringy &accel);
  bool prefix_exists(const Stringy &prefix);

private:
  List list;
};

static void add_standard_menu_accelerators(Command_Dispatcher &);
static void add_builtin_accelerators(Command_Dispatcher &, char *accel, ...);
static bool builtin_menu_accelerator(Session &, const Stringy &accel, void *);
static void toggle_ornament_shown(View *view, Ornament_Type type);
static void toggle_view_setting(View *view, bool View_Settings::*member);
static void zoom(View *view, double factor);
static void z_step(View *view, double planes);
static void zoom_previous(View *view);
static void zoom_next(View *view);
static void zoom_remember(View *view);
static void show_resonance_peaks(Session &, CrossPeak *xp, int axis);
static void integrate_box(Spectrum *sp, Reporter &rr);
static void integrate_ellipse(Spectrum *sp, Reporter &rr);
static void integrate_gaussian(Spectrum *sp, Reporter &rr);
static void integrate_lorentzian(Spectrum *sp, Reporter &rr);
static bool peak_integrate(View *);
static void key_timeout_cb(CB_Data);
static bool ornament_undoable_delete(const List &olist, Project &);
static List destruction_dependents(const List &olist);
static bool unintegrate_peaks(Spectrum *sp);
static bool peak_select_all(Spectrum *sp, Project &);
static bool peak_select_fully_assigned(Spectrum *, Project &);
static bool peak_select_partially_assigned(Spectrum *, Project &);
static bool peak_select_unassigned(Spectrum *, Project &);
static bool peak_select_colors(Spectrum *sp);
static bool peak_invert_selection(Spectrum *sp);
static bool peak_select_aliased(Spectrum *sp, Project &);
static bool peak_select_range(Spectrum *sp, Project &, int rmin, int rmax);
static int assignment_range(CrossPeak *xp);
static bool peak_estimate(Spectrum *sp);
static bool peak_height_reset(Spectrum *sp);
static void alias_selected_peaks(Spectrum *sp, int a, double multiples);
static void reflect_selected_peaks(Spectrum *sp, int a, bool downfield);
static void unalias_selected_peaks(Spectrum *sp);
static bool peak_lock(Spectrum *sp);
static bool peak_unlock(Spectrum *sp);
static bool peak_center(Spectrum *sp, Session &);
static bool center_view_on_peak(View *view, CrossPeak *xp);
static void axis_origin(View *view);
static void selected_peak_list(Session &);
static void request_quit(Session &);
static void delete_unused_resonances(Condition *c, Reporter &rr);
static bool define_peak_group(Spectrum *sp, Reporter &rr);
static void crossdiagonal_ornament_copy(Spectrum *sp, Project &proj,
					Reporter &rr);
static void renumber_view(View *v);

// ----------------------------------------------------------------------------
//
Command_Dispatcher::Command_Dispatcher(Session &s) : session(s)
{
  accel_list = new Accelerator_List();
  ignore_key_commands = false;
  add_standard_menu_accelerators(*this);
}

// ----------------------------------------------------------------------------
//
Command_Dispatcher::~Command_Dispatcher()
{
  delete accel_list;
  accel_list = NULL;
}

// ----------------------------------------------------------------------------
//
Accelerator::Accelerator(Session &s,
			 const Stringy &accel, const Stringy &menu_text,
			 Command_Proc cmd, void *cmd_data) : session(s)
{
  this->accel = accel;
  this->menu_text = menu_text;
  this->cmd = cmd;
  this->cmd_data = cmd_data;
  if (!menu_text.is_empty())
    add_extension_menu_entry(s, menu_text, accel);
}

// ----------------------------------------------------------------------------
//
Accelerator::~Accelerator()
{
  if (!menu_text.is_empty())
    remove_extension_menu_entry(session, menu_text);
}

// ----------------------------------------------------------------------------
//
const Stringy &Accelerator::accelerator()
  { return accel; }

// ----------------------------------------------------------------------------
//
void Accelerator::invoke()
{
  cmd(session, accel, cmd_data);
}

// ----------------------------------------------------------------------------
//
Accelerator_List::~Accelerator_List()
{
  for (int ai = 0 ; ai < list.size() ; ++ai)
    delete (Accelerator *) list[ai];
  list.erase();
}

// ----------------------------------------------------------------------------
//
void Accelerator_List::add_accelerator(Accelerator *a)
  { list.append(a); }
void Accelerator_List::remove_accelerator(Accelerator *a)
  { list.erase(a); }

// ----------------------------------------------------------------------------
//
Accelerator *Accelerator_List::find(const Stringy &accel)
{
  for (int ai = 0 ; ai < list.size() ; ++ai)
    {
      Accelerator *a = (Accelerator *) list[ai];
      if (a->accelerator() == accel)
	return a;
    }
  return NULL;
}

// ----------------------------------------------------------------------------
//
bool Accelerator_List::prefix_exists(const Stringy &prefix)
{
  for (int ai = 0 ; ai < list.size() ; ++ai)
    {
      Accelerator *a = (Accelerator *) list[ai];
      if (has_prefix(a->accelerator(), prefix))
	return true;
    }
  return false;
}

// ----------------------------------------------------------------------------
//
void Command_Dispatcher::add_accelerator(const Stringy &accel,
					 const Stringy &menu_text,
					 Command_Proc cmd, void *cmd_data)
{
  remove_accelerator(accel);
  Accelerator *a = new Accelerator(session, accel, menu_text, cmd, cmd_data);
  accel_list->add_accelerator(a);
}

// ----------------------------------------------------------------------------
//
bool Command_Dispatcher::remove_accelerator(const Stringy &accel)
{
  Accelerator *a = accel_list->find(accel);
  if (a)
    {
      accel_list->remove_accelerator(a);
      delete a;
    }

  return (a != NULL);
}

// ----------------------------------------------------------------------------
//
bool Command_Dispatcher::execute_menu_accelerator(const Stringy &accel)
{
  Accelerator *a = accel_list->find(accel);

  if (a)
    a->invoke();

  return a != NULL;
}

// ----------------------------------------------------------------------------
//
static void add_standard_menu_accelerators(Command_Dispatcher &cd)
{
  add_builtin_accelerators(cd,
    "",
    "!d",
    "A1","A2","A3","A4","a1","a2","a3","a4","aD","aP","aX",
      "ac","ag","ap","as","at",
    "co","cr","ct",
    "d1","d2","d3","d4","dr",
    "eu",
    "fa","fo","fs",
    "gs",
    "he","hf","hh","hi","ho","hp","hs","ht","hx",
    "ib","ic","ie","ig","il","is","it",
    "ja","jc","jo","js",
    "kt",
    "lb","lB","lr", "ls","lt", "lu",
    "mi","MA",
    "nl","nt",
    "oX", "oc","ol","op","os","ot","ov","oz",
    "pa","pA","pc","pC","pd","pe","pf","pF","pg","ph", "pi","pI","pk",
      "pl","pL","pm","pM","pN","pP","pR","ps","pS","pt","pu","pU","pv",
    "qt",
    "r1","r2","r3","r4","ra","rf","rgX","rl","rr","rt","rv",
    "sa","sc","sl","sf","st",
    "tb","tt",
    "u1","u2","u3","u4","ua",
    "vC","vD","vR","vS","vb","vc","vd","vf","vh","vp","vr","vs","vt","vz",
    "xD","xO","xa","xd","xh","xp","xr","xs","xx",
    "ya","yd","yr","yt",
    "zd","zf","zi","zn","zo","zp","zr","zu",
    NULL);
}

// ----------------------------------------------------------------------------
//
static void add_builtin_accelerators(Command_Dispatcher &cd, char *accel, ...)
{
  va_list args;

  va_start(args, accel);
  for (; accel ; accel = va_arg(args, char *))
    cd.add_accelerator(accel, "", builtin_menu_accelerator, NULL);

  va_end(args);
}

// ----------------------------------------------------------------------------
//
static bool builtin_menu_accelerator(Session &s, const Stringy &accel, void *)
{
  Reporter &rr = s.reporter();
  Project &proj = s.project();
  WinSys &ws = s.window_system();
  View *view = selected_view(s);
  Spectrum *sp = (view ? view->spectrum() : NULL);
  Condition *c = (sp ? sp->condition() : NULL);
  CrossPeak *xp = proj.selected_peak();

  if (accel == "") ornament_undoable_delete(proj.selected_ornaments(), proj);
  else if (accel == "lb") label_peaks(proj.selected_crosspeaks());
  else if (accel == "lB") unlabel_peaks(proj.selected_crosspeaks());
  else if (accel == "pl") show_label_dialog(s, xp);
  else if (accel == "at") show_assignment_dialog(s, xp);
  else if (accel == "ot") show_ornament_dialog(s);
  else if (accel == "nt") show_ornament_note_dialog(s);
  else if (accel == "yt") show_sync_dialog(s);
  else if (accel == "ya") show_sync_dialog(s);
  else if (accel == "yd") show_sync_dialog(s);
  else if (accel == "!d") request_coredump(s);
  else if (accel == "pf") show_preference_dialog(s);
  else if (accel == "pv") show_view_list_dialog(s);
  else if (accel == "pd") show_predefined_resonance_dialog(s);
  else if (accel == "rr") show_rename_dialog(s);
  else if (accel == "fo") show_spectrum_open_dialog(s);
  else if (accel == "jo") show_project_open_dialog(s);
  else if (accel == "js") save_project(s, false);
  else if (accel == "ja") show_project_saveas_dialog(s);
  else if (accel == "jc") proj.unload_query();
  else if (accel == "qt") request_quit(s);
  else if (accel == "ic") ws.iconify_dialog(ws.main_widget());
  else if (accel == "pm") show_mode_buttons(s);
  else if (accel == "ol") show_overlay_dialog(s);
  else if (accel == "eu") proj.DoUndo();
  else if (accel == "ra") show_resonance_peaks_dialog(s);
  else if (accel == "r1") show_resonance_peaks(s, xp, 0);
  else if (accel == "r2") show_resonance_peaks(s, xp, 1);
  else if (accel == "r3") show_resonance_peaks(s, xp, 2);
  else if (accel == "r4") show_resonance_peaks(s, xp, 3);
  else if (accel == "sl") selected_peak_list(s);
  else if (accel == "cr") show_ornament_color_dialog(s);
  else if (accel == "oz") show_ornament_size_dialog(s, sp);
  else if (accel == "he") show_help(s, "manual");
  else if (accel == "hi") show_help(s, "intro");
  else if (accel == "hf") show_help(s, "files");
  else if (accel == "hs") show_help(s, "spectra");
  else if (accel == "hp") show_help(s, "peaks");
  else if (accel == "hx") show_help(s, "extensions");
  else if (accel == "ct") show_contour_dialog(s, view);
  else if (accel == "it") show_integration_dialog(s, sp);
  else if (accel == "kt") show_pick_dialog(s, sp);
  else if (accel == "tt") show_view_dialog(s, view);
  else if (accel == "pt") show_print_dialog(s, view);
  else if (accel == "rt") show_region_dialog(s, view);
  else if (accel == "rgX") show_region_dialog(s, view);
  else if (accel == "st") show_spectrum_dialog(s, sp);
  else if (accel == "vt") show_view_dialog(s, view);
  else if (accel == "vz") show_view_depth_dialog(s, view);
  else if (accel == "co") show_assignment_copy_dialog(s, sp);
  else if (accel == "mi") show_midas_dialog(s, sp);
  else if (accel == "tb") show_resonance_table_dialog(s, sp);
  else if (accel == "ag") show_guess_dialog(s, sp);
  else if (accel == "MA") allocation_report();

  else if (view == NULL)
    {
      rr.message("No view selected.\n");
      return true;
    }
  else if (accel == "pc") peak_center(sp, s);
  else if (accel == "pe") peak_estimate(sp);
  else if (accel == "pg") define_peak_group(sp, rr);
  else if (accel == "ph") peak_height_reset(sp);
  else if (accel == "pi") peak_integrate(view);
  else if (accel == "pk") peak_lock(sp);
  else if (accel == "pa") peak_select_all(sp, proj);
  else if (accel == "pF") peak_select_fully_assigned(sp, proj);
  else if (accel == "pP") peak_select_partially_assigned(sp, proj);
  else if (accel == "pN") peak_select_unassigned(sp, proj);
  else if (accel == "pC") peak_select_colors(sp);
  else if (accel == "pI") peak_invert_selection(sp);
  else if (accel == "pA") peak_select_aliased(sp, proj);
  else if (accel == "pR") peak_select_range(sp, proj, 0, 0);
  else if (accel == "pS") peak_select_range(sp, proj, 1, 1);
  else if (accel == "pM") peak_select_range(sp, proj, 2, 4);
  else if (accel == "pL") peak_select_range(sp, proj, 5, INT_MAX);
  else if (accel == "pu") peak_unlock(sp);
  else if (accel == "pU") unintegrate_peaks(sp);
  else if (accel == "ac") proj.assignment_copy(sp);
  else if (accel == "aD") proj.assignment_delete(sp);
  else if (accel == "aP") proj.assignment_paste(sp, false);
  else if (accel == "ap") proj.assignment_paste(sp, true);
  else if (accel == "aX")
    crossdiagonal_assignment_copy(sp->selected_crosspeaks(), true, rr);
  else if (accel == "gs") toggle_ornament_shown(view, grid);
  else if (accel == "ls") toggle_ornament_shown(view, label);
  else if (accel == "is") toggle_ornament_shown(view, line);
  else if (accel == "os")
    toggle_view_setting(view, &View_Settings::show_ornaments);
  else if (accel == "ps") toggle_ornament_shown(view, peak);
  else if (accel == "xs") toggle_ornament_shown(view, peakgroup);
  else if (accel == "xd") view->set_axis_units(INDEX);
  else if (accel == "xh") view->set_axis_units(HZ);
  else if (accel == "xp") view->set_axis_units(PPM);
  else if (accel == "xO") axis_origin(view);
  else if (accel == "xr") view->roll_axes();
  else if (accel == "xx") view->swap_axes();
  else if (accel == "vc") center_view_on_peak(view, xp);
  else if (accel == "vs")
    toggle_view_setting(view, &View_Settings::show_scales);
  else if (accel == "vb")
    toggle_view_setting(view, &View_Settings::show_scrollbars);
  else if (accel == "xa")
    toggle_view_setting(view, &View_Settings::show_nucleus_names);
  else if (accel == "vR")
    toggle_view_setting(view, &View_Settings::show_resonance_panels);
  else if (accel == "rf")
    toggle_view_setting(view, &View_Settings::filter_resonances);
  else if (accel == "vS")
    toggle_view_setting(view, &View_Settings::show_slices);
  else if (accel == "sa")
    toggle_view_setting(view, &View_Settings::slice_auto_scale);
  else if (accel == "sf")
    toggle_view_setting(view, &View_Settings::slice_subtract_peaks);
  else if (accel == "vp")
    toggle_view_setting(view, &View_Settings::show_peak_info);
  else if (accel == "vC")
    toggle_view_setting(view, &View_Settings::show_contour_scale);
  else if (accel == "vf")
    toggle_view_setting(view, &View_Settings::subtract_fit_peaks);
  else if (accel == "hh")
    toggle_view_setting(view, &View_Settings::show_crosshair);
  else if (accel == "ht")
    toggle_view_setting(view, &View_Settings::show_transpose_crosshair);
  else if (accel == "ho")
    toggle_view_setting(view, &View_Settings::show_crosshair_from_other_views);
  else if (accel == "vd") view->duplicate(view->view_rectangle());
  else if (accel == "vD") destroy_view_window(view);
  else if (accel == "vh") view->show_view(false);
//  else if (accel == "vP") print_view(view, wait_callbacks(s));
  else if (accel == "vr") view->redraw();
  else if (accel == "lu") view->unoverlap_labels();
  else if (accel == "lr") view->right_offset_labels();
  else if (accel == "lt") show_peak_list_dialog(s, sp, sp->mPeakListOptions);
  else if (accel == "nl") new_peak_list_dialog(s, sp, sp->mPeakListOptions);
  else if (accel == "rl") show_resonance_list_dialog(s, *c);
  else if (accel == "fa") show_spectrum_saveas_dialog(view);
  else if (accel == "yr") proj.synchronize_match_views(view);
  else if (accel == "zp") zoom_previous(view);
  else if (accel == "zn") zoom_next(view);
  else if (accel == "zr") zoom_remember(view);
  else if (accel == "zi") zoom(view, ZOOM_IN_FACTOR);
  else if (accel == "zo") zoom(view, ZOOM_OUT_FACTOR);
  else if (accel == "zf") view->zoom(view->full_view());
  else if (accel == "zd") z_step(view, 1);
  else if (accel == "zu") z_step(view, -1);
  else if (accel == "ov") orthogonal_views(view);
  else if (accel == "sc") orthogonal_views(view);
  else if (accel == "ib") integrate_box(sp, rr);
  else if (accel == "ie") integrate_ellipse(sp, rr);
  else if (accel == "ig") integrate_gaussian(sp, rr);
  else if (accel == "il") integrate_lorentzian(sp, rr);
  else if (accel == "a1") alias_selected_peaks(sp, 0, 1);
  else if (accel == "a2") alias_selected_peaks(sp, 1, 1);
  else if (accel == "a3") alias_selected_peaks(sp, 2, 1);
  else if (accel == "a4") alias_selected_peaks(sp, 3, 1);
  else if (accel == "A1") alias_selected_peaks(sp, 0, -1);
  else if (accel == "A2") alias_selected_peaks(sp, 1, -1);
  else if (accel == "A3") alias_selected_peaks(sp, 2, -1);
  else if (accel == "A4") alias_selected_peaks(sp, 3, -1);
  else if (accel == "d1") reflect_selected_peaks(sp, 0, true);
  else if (accel == "d2") reflect_selected_peaks(sp, 1, true);
  else if (accel == "d3") reflect_selected_peaks(sp, 2, true);
  else if (accel == "d4") reflect_selected_peaks(sp, 3, true);
  else if (accel == "u1") reflect_selected_peaks(sp, 0, false);
  else if (accel == "u2") reflect_selected_peaks(sp, 1, false);
  else if (accel == "u3") reflect_selected_peaks(sp, 2, false);
  else if (accel == "u4") reflect_selected_peaks(sp, 3, false);
  else if (accel == "ua") unalias_selected_peaks(sp);
  else if (accel == "oc") proj.ornament_copy(sp);
  else if (accel == "op") proj.ornament_paste(sp);
  else if (accel == "oX") crossdiagonal_ornament_copy(sp, proj, rr);
  else if (accel == "fs") save_spectrum(s, sp, false);
  else if (accel == "dr") delete_unused_resonances(c, rr);
  else if (accel == "rv") renumber_view(view);
  else
    return false;

  return true;
}

// ----------------------------------------------------------------------------
//
static void toggle_view_setting(View *view, bool View_Settings::*member)
{
  View_Settings s = view->settings();
  s.*member = !(s.*member);
  view->configure(s);
}

// ----------------------------------------------------------------------------
//
static void toggle_ornament_shown(View *view, Ornament_Type type)
{
  View_Settings s = view->settings();
  s.show_ornament(type, !s.show_ornament(type));
  view->configure(s);
}

// ----------------------------------------------------------------------------
//
static void zoom(View *view, double factor)
{
  Rectangle r = view->view_rectangle();
  r.scale(factor);
  view->zoom(r);
}

// ----------------------------------------------------------------------------
//
static void z_step(View *view, double planes)
{
  Spectrum *sp = view->spectrum();
  if (sp->dimension() == 3)
    {
      int z_axis = view->axis_order()[2];
      double plane_step = sp->scale(1, z_axis, INDEX, PPM);
      double step = planes * plane_step;
      SRegion r = view->view_region();
      r.min[z_axis] += step;
      r.max[z_axis] += step;
      view->zoom(r);
    }
}

// ----------------------------------------------------------------------------
//
static void zoom_previous(View *view)
  { view->zoom_previous(); }
static void zoom_next(View *view)
  { view->zoom_next(); }
static void zoom_remember(View *view)
  { view->zoom(view->view_region()); }

// ----------------------------------------------------------------------------
//
static void show_resonance_peaks(Session &s, CrossPeak *xp, int axis)
{
  if (xp && axis < xp->dimension() && xp->resonance(axis))
    show_peak_list_dialog(s, xp->resonance(axis));
}

/*
 * Switch to integrating isolated peaks by boxing.
 */
static void integrate_box(Spectrum *sp, Reporter &rr)
{
	sp->mIntegrate.integration_method = INTEGRATE_BOX;
	rr.message("Isolated peaks now integrated by boxing.\n");
}

/*
 * Switch to integrating isolated peaks by ellipsing.
 */
static void integrate_ellipse(Spectrum *sp, Reporter &rr)
{
	sp->mIntegrate.integration_method = INTEGRATE_ELLIPSE;
	rr.message("Isolated peaks now integrated by ellipsing.\n");
}

/*
 * Switch to integrating isolated peaks by gaussian fitting.
 */
static void integrate_gaussian(Spectrum *sp, Reporter &rr)
{
	sp->mIntegrate.integration_method = INTEGRATE_GAUSSIAN;
	rr.message("Isolated peaks now integrated by Gaussian fit.\n");
}

/*
 * Switch to integrating isolated peaks by gaussian fitting.
 */
static void integrate_lorentzian(Spectrum *sp, Reporter &rr)
{
	sp->mIntegrate.integration_method = INTEGRATE_LORENTZIAN;
	rr.message("Isolated peaks now integrated by Lorentzian fit.\n");
}

//
// Integrate the selected peaks
//
static bool peak_integrate(View *v)
{
	Session &s = v->session();
	Spectrum *sp = v->spectrum();
	List peaks = sp->selected_peaklets();
	if (!peaks.empty()) {
	  s.project().MakeUndoIntegrate(peaks);
	  double pos, neg;
	  v->lowest_contour_levels(&pos, &neg);
	  integrate_list(sp, peaks, SRegion(), pos, neg,
			 wait_callbacks(s), s.reporter());
	  return true;
	}
	Reporter &rr = s.reporter();
	rr.message("No selected peaks to integrate\n");
	return false;
}

// ----------------------------------------------------------------------------
//
void Command_Dispatcher::ignore_keys(bool ignore)
{
  ignore_key_commands = ignore;
}

// ----------------------------------------------------------------------------
//
void Command_Dispatcher::parse_key_event(void *event)
{
  char c;
  int f;
  bool shift;
  WinSys &ws = session.window_system();
  if (ws.key_pressed(event, &c))
    parse_key(c);
  else if (ws.function_key_pressed(event, &f, &shift))
    parse_function_key(f, shift);
}

// ----------------------------------------------------------------------------
//
void Command_Dispatcher::parse_key(char key)
{
	if (ignore_key_commands)
	  return;

	if (key == '')	// escape clears key buffer
	  {
	    key_buffer = "";
	    status_line(session, "");
	    return;
	  }

	WinSys &ws = session.window_system();
	long timeout = session.project().preferences.key_timeout_interval;
	if (timeout > 0 && key_buffer.is_empty())
	  ws.add_timer(timeout, key_timeout_cb, this);

	key_buffer << key;
	status_line(session, "%s", key_buffer.cstring());
	Accelerator *a = accel_list->find(key_buffer);
	if (a)
	  {
	    ws.remove_timer(timeout, key_timeout_cb, this);
	    key_buffer = "";
	    a->invoke();
	  }
	else if (!accel_list->prefix_exists(key_buffer))
	  {
	    ws.remove_timer(timeout, key_timeout_cb, this);
	    Reporter &rr = session.reporter();
	    rr.message("No accelerator for %s\n", key_buffer.cstring());
	    status_line(session, "%s unknown accelerator", key_buffer.cstring());
	    key_buffer = "";
	  }
}

// ----------------------------------------------------------------------------
//
void Command_Dispatcher::clear_key_buffer()
{
  if (!key_buffer.is_empty())
    {
      key_buffer = "";
      status_line(session, "");
    }
}

// ----------------------------------------------------------------------------
//
void key_timeout_cb(CB_Data cmd_dispatcher)
{
  Command_Dispatcher *cd = (Command_Dispatcher *) cmd_dispatcher;
  cd->clear_key_buffer();
}

/*
 * A function key <key> was pressed.
 */
void Command_Dispatcher::parse_function_key(int key, bool shifted)
{
  if (ignore_key_commands)
    return;

  if (1 <= key && key <= 12 && !shifted) {	/* a function key */
    Pointer_Mode modes[] = 
    {
      MODE_SELECT,			/* F1 */
      MODE_CENTER,			/* F2 */
      MODE_GRIDBOTH,			/* F3 */
      MODE_GRIDHOR,			/* F4 */
      MODE_GRIDVERT,			/* F5 */
      MODE_LABEL,			/* F6 */
      MODE_LINE,			/* F7 */
      MODE_PEAK,			/* F8 */
      MODE_SELECT,			/* F9 */
      MODE_INTEGRATE,			/* F10 */
      MODE_ZOOM,			/* F11 */
      MODE_DUPZOOM			/* F12 */
    };
    set_pointer_mode(session, modes[key - 1]);
  }
  else if (1 <= key && key <= 2 && shifted) { /* shift function key */
    Pointer_Mode shift_modes[] = 
    {
      MODE_COPYASSIGNMENT,		/* shift-F1 */
      MODE_GUESSASSIGNMENT		/* shift-F2 */
    };
    set_pointer_mode(session, shift_modes[key - 1]);
  }
}

/*
 */
static bool ornament_undoable_delete(const List &olist, Project &proj)
{
  List dlist = olist;
  dlist.append(destruction_dependents(olist));

	/*
	 * We can undo this delete.
	 */
	Undo	*undo = proj.MakeUndoDelete();
	for (int oi = 0 ; oi < dlist.size() ; ++oi)
	  undo->add((Ornament *) dlist[oi]);
	undo->finish();

	for (int oi = 0 ; oi < dlist.size() ; ++oi)
	  delete (Ornament *) dlist[oi];

  return true;
}

// ----------------------------------------------------------------------------
// If a peak is being deleted make sure its label is too.
// If all the peaklets of a peakgroup are being deleted and the
// peakgroup is not on the list add it.
//
static List destruction_dependents(const List &olist)
{
  List dd;

  Table dtable;
  for (int oi = 0 ; oi < olist.size() ; ++oi)
    dtable.insert((TableKey) olist[oi], (TableData) NULL);

  //
  // Add peak group to destruction list if all peaklets are being destroyed.
  //
  for (int oi = 0 ; oi < olist.size() ; ++oi)
    {
      Ornament *op = (Ornament *) olist[oi];
      if (op->type() == peak)
	{
	  Peak *pk = (Peak *) op;
	  PeakGp *pg = pk->peakgp();
	  const List &peaklets = pg->peaklets();
	  if (pg && peaklets.size() > 0 && (Peak *) peaklets[0] == pk &&
	      !dtable.find((TableKey)(Ornament *)pg, NULL))
	    {
	      int dcount = 0;
	      for (int pi = 0 ; pi < peaklets.size() ; ++pi)
		if (dtable.find((TableKey)(Ornament *)(Peak *) peaklets[pi],
				NULL))
		  dcount += 1;
	      if (dcount == peaklets.size())
		{
		  dd.append(pg);
		  dtable.insert((TableKey)(Ornament *) pg, (TableData) NULL);
		}
	    }
	}
    }

  //
  // Delete labels for peaks
  //
  List dlist = dtable.keys();
  for (int oi = 0 ; oi < dlist.size() ; ++oi)
    {
      Ornament *op = (Ornament *) dlist[oi];
      if (is_crosspeak(op))
	{
	  Label *label = ((CrossPeak *) op)->label();
	  if (label && !dtable.find((Ornament *)label, NULL))
	    {
	      dd.append(label);
	      dtable.insert((TableKey)(Ornament *) label, (TableData) NULL);
	    }
	}
    }

  return dd;
}

//
// Forget the integrals and linewidths for the selected peaks
//
static bool unintegrate_peaks(Spectrum *sp)
{
  List peaks = sp->selected_peaklets();
  for (int pi = 0 ; pi < peaks.size() ; ++pi)
    {
      Peak *pk = (Peak *) peaks[pi];
      pk->no_volume();
      pk->linewidth("", SPoint(pk->dimension()));
    }

  return true;
}

// ----------------------------------------------------------------------------
//
static bool peak_select_all(Spectrum *sp, Project &proj)
{
  proj.unselect_all_ornaments();
  List plist = sp->crosspeaks();
  plist.append(sp->peaklets());
  for (int pi = 0 ; pi < plist.size() ; ++pi)
    {
      CrossPeak *xp = (CrossPeak *) plist[pi];
      xp->select(true);
    }

  return true;
}

// ----------------------------------------------------------------------------
//
static bool peak_select_fully_assigned(Spectrum *sp, Project &proj)
{
  proj.unselect_all_ornaments();
  List plist = sp->crosspeaks();
  for (int pi = 0 ; pi < plist.size() ; ++pi)
    {
      CrossPeak *xp = (CrossPeak *) plist[pi];
      if (xp->is_fully_assigned())
	xp->select(true);
    }

  return true;
}

// ----------------------------------------------------------------------------
//
static bool peak_select_partially_assigned(Spectrum *sp, Project &proj)
{
  proj.unselect_all_ornaments();
  List plist = sp->crosspeaks();
  for (int pi = 0 ; pi < plist.size() ; ++pi)
    {
      CrossPeak *xp = (CrossPeak *) plist[pi];
      if (xp->IsAssigned() && !xp->is_fully_assigned())
	xp->select(true);
    }

  return true;
}

// ----------------------------------------------------------------------------
//
static bool peak_select_unassigned(Spectrum *sp, Project &proj)
{
  proj.unselect_all_ornaments();
  List plist = sp->crosspeaks();
  for (int pi = 0 ; pi < plist.size() ; ++pi)
    {
      CrossPeak *xp = (CrossPeak *) plist[pi];
      if (!xp->IsAssigned())
	xp->select(true);
    }

  return true;
}

// ----------------------------------------------------------------------------
// Select all peaks in spectrum that have the same color of some already
// selected peak in the spectrum.
//
static bool peak_select_colors(Spectrum *sp)
{
  Table colornames(equal_strings, hash_string);
  List plist = sp->selected_crosspeaks();
  for (int pi = 0 ; pi < plist.size() ; ++pi)
    {
      CrossPeak *xp = (CrossPeak *) plist[pi];
      Stringy c = xp->GetColor().name();
      if (! colornames.find((TableKey) &c, NULL))
	colornames.insert((TableKey) new Stringy(c), NULL);
    }

  plist = sp->crosspeaks();
  for (int pi = 0 ; pi < plist.size() ; ++pi)
    {
      CrossPeak *xp = (CrossPeak *) plist[pi];
      Stringy c = xp->GetColor().name();
      if (colornames.find((TableKey) &c, NULL))
	xp->select(true);
    }

  List namelist = colornames.keys();
  free_string_list_entries(namelist);

  return true;
}

// ----------------------------------------------------------------------------
//
static bool peak_invert_selection(Spectrum *sp)
{
  const List &plist = sp->crosspeaks();
  for (int pi = 0 ; pi < plist.size() ; ++pi)
    {
      CrossPeak *xp = (CrossPeak *) plist[pi];
      xp->select(! xp->IsSelected());
    }

  return true;
}

// ----------------------------------------------------------------------------
//
static bool peak_select_aliased(Spectrum *sp, Project &proj)
{
  proj.unselect_all_ornaments();
  const List &plist = sp->crosspeaks();
  for (int pi = 0 ; pi < plist.size() ; ++pi)
    {
      CrossPeak *xp = (CrossPeak *) plist[pi];
      if (xp->IsAliased())
	xp->select(true);
    }
  return true;
}

// ----------------------------------------------------------------------------
//
static bool peak_select_range(Spectrum *sp, Project &proj, int rmin, int rmax)
{
  proj.unselect_all_ornaments();
  const List &plist = sp->crosspeaks();
  for (int pi = 0 ; pi < plist.size() ; ++pi)
    {
      CrossPeak *xp = (CrossPeak *) plist[pi];
      int r = assignment_range(xp);
      if (r >= rmin && r <= rmax)
	xp->select(true);
    }
  return true;
}

// ----------------------------------------------------------------------------
//
static int assignment_range(CrossPeak *xp)
{
  int min_group = INT_MAX;
  int max_group = INT_MIN;
  int dim = xp->spectrum()->dimension();
  for (int a = 0 ; a < dim ; ++a)
    {
      Resonance *r = xp->resonance(a);
      if (r)
	{
	  int g = r->group()->number();
	  if (g)
	    {
	      if (g < min_group)
		min_group = g;
	      if (g > max_group)
		max_group = g;
	    }
	}
    }
  return (max_group >= min_group ? max_group - min_group : -1);
}

/*
 * Set the peak linewidth according to the 1/2 max.
 */
static bool peak_estimate(Spectrum *sp)
{
	List peaks = sp->selected_peaklets();
	for (int pi = 0 ; pi < peaks.size() ; ++pi)
	  ((Peak *) peaks[pi])->estimate_linewidth();

	return true;
}

// ----------------------------------------------------------------------------
// Reset the cached peak heights of selected peaks.
//
static bool peak_height_reset(Spectrum *sp)
{
	List peaks = sp->selected_peaklets();
	for (int pi = 0 ; pi < peaks.size() ; ++pi)
	  {
	    Peak *pk = (Peak *) peaks[pi];
	    pk->DataHeight(sp->height_at_point(pk->position()));
	  }

	return true;
}

/*
 * Shift the amount of alias for all selected peak(lets)
 */
static void alias_selected_peaks(Spectrum *sp, int a, double multiples)
{
  if (a >= sp->dimension())
    return;

  SPoint delta(sp->dimension());
  delta[a] = multiples * sp->ppm_sweep_width()[a];

  List list = sp->selected_ornaments();
  for (int oi = 0 ; oi < list.size() ; ++oi)
    {
      Ornament *op = (Ornament *) list[oi];
      if (is_crosspeak(op))
	{
	  CrossPeak *xp = (CrossPeak *) op;
	  xp->set_alias(xp->alias() + delta);
	}
    }
}

// ----------------------------------------------------------------------------
// Reflect frequency of selected peaks and peaklets about a given ppm value.
// Spectra acquired by the STATES method show this type of aliasing.
// The peak marker stays at the same ppm position.
//
static void reflect_selected_peaks(Spectrum *sp, int a, bool downfield)
{
  if (a >= sp->dimension())
    return;

  double center_ppm = sp->ppm_region().max[a] -
    (downfield ? 0 : sp->ppm_spectrum_width()[a]);

  List olist = sp->selected_ornaments();
  for (int oi = 0 ; oi < olist.size() ; ++oi)
    {
      Ornament *op = (Ornament *) olist[oi];
      if (is_crosspeak(op))
	{
	  CrossPeak *xp = (CrossPeak *) op;
	  SPoint delta(sp->dimension());
	  delta[a] = 2 * (center_ppm - xp->frequency(a));
	  xp->set_alias(xp->alias() + delta);
	}
    }
}

// ----------------------------------------------------------------------------
//
static void unalias_selected_peaks(Spectrum *sp)
{
  List olist = sp->selected_ornaments();
  SPoint zero(sp->dimension());
  for (int oi = 0 ; oi < olist.size() ; ++oi)
    {
      Ornament *op = (Ornament *) olist[oi];
      if (is_crosspeak(op))
	((CrossPeak *) op)->set_alias(zero);
    }
}

/*
 * Lock all selected peaks
 */
static bool peak_lock(Spectrum *sp)
{
	List list = sp->selected_peaklets();

	for (int pi = 0 ; pi < list.size() ; ++pi)
	  {
	    CrossPeak *xp = (CrossPeak *) list[pi];
	    xp->lock(true);
	  }
	return true;
}

/*
 * Unlock all selected peaks
 */
static bool peak_unlock(Spectrum *sp)
{
	List list = sp->selected_peaklets();

	for (int pi = 0 ; pi < list.size() ; ++pi)
	  {
	    CrossPeak *xp = (CrossPeak *) list[pi];
	    xp->lock(false);
	}
	return true;
}


/*
 * Center all selected peaks on the maximum intensity, preserving
 * their integration mode.
 */
static bool peak_center(Spectrum *sp, Session &s)
{
	List list = sp->selected_peaklets();

	if (!list.empty()) {

		/*
		 * Place on undo move.
		 */
		Undo	*undo = s.project().MakeUndoMove();

		for (int pi = 0 ; pi < list.size() ; ++pi)
		      undo->add((Peak *) list[pi]);
		undo->finish();

		/*
		 * Now center each peaklet
		 */
		for (int pi = 0 ; pi < list.size() ; ++pi)
		  if (!((Peak *) list[pi])->MoveToMaximum())
		    {
		      Reporter &rr = s.reporter();
		      rr.message("Peak is locked, cannot be moved\n");
		    }
	}

	return true;
}

// ----------------------------------------------------------------------------
// Center a view on a peak possibly from a different spectrum.
//
// The main difficulty is how to match the axes for the two spectra.
//
// Each view axis is centered when there is a unique crosspeak axis
// with matching nucleus type.  If the wN view axis matches the nucleus
// type of multiple crosspeak axes and the wN crosspeak axis is one of
// these then center with that axis.  This second rule makes mapping
// between 2D homonuclear spectra (HH->HH) work.  Also it assures that
// the centering will always work if the peak and view have the same
// spectrum.
//
static bool center_view_on_peak(View *view, CrossPeak *xp)
{
  if (xp == NULL)
    {
      Reporter &rr = view->session().reporter();
      rr.message("No selected peak to center view on\n");
      return false;
    }

  Spectrum *view_spectrum = view->spectrum();
  Spectrum *peak_spectrum = xp->spectrum();
  SPoint center = view->center();
  for (int a = 0 ; a < view_spectrum->dimension() ; ++a)
    {
      Stringy nucleus = view_spectrum->nucleus_type(a);
      int axis;
      if (peak_spectrum->is_nucleus_unique(nucleus, &axis))
	center[a] = xp->frequency()[axis];
      else if (a < xp->dimension() &&
	       peak_spectrum->nucleus_type(a) == nucleus)
	center[a] = xp->frequency()[a];
    }
  center = alias_onto_spectrum(center, view_spectrum);
  view->set_center(center);

  return true;
}

/*
 * Get the current location of the pointer, convert it into ppm units
 * and make it the new origin.
 */
static void axis_origin(View *view)
{
  SPoint p;
  if (view->pointer_position(&p))
    {
      Spectrum	*sp = view->spectrum();
      SPoint zero = SPoint(sp->dimension());
      zero[view->axis(X)] = p[view->axis(X)];
      zero[view->axis(Y)] = p[view->axis(Y)];
      sp->set_ppm_zero(zero);

      // Move ppm scale, not contours, in spectrum views
      Project &proj = view->session().project();
      shift_views(proj.view_list(sp), -zero);
    }
}

// ----------------------------------------------------------------------------
//
static void selected_peak_list(Session &s)
{
  List plist = s.project().selected_crosspeaks();
  show_peak_list_dialog(s, plist, default_peak_list_format());
}

// ----------------------------------------------------------------------------
//
static void request_quit(Session &s)
{
  if (s.ok_to_end_session())
    s.end_session();
}

/*
 * Remove the unreferenced resonances.
 */
static void delete_unused_resonances(Condition *c, Reporter &rr)
{
	List rlist = c->resonance_list();
	for (int ri = 0 ; ri < rlist.size() ; ++ri) {
		Resonance *r = (Resonance *) rlist[ri];
		if (r->assignment_count() == 0) {
			rr.message("Condition %s: Removing resonance %s\n",
				   c->name().cstring(), r->name().cstring());
			delete r;
		}
	}
}

/*
 * Define a peak group for the selected peaks that are not already
 * part of a peak group.
 */
static bool define_peak_group(Spectrum *sp, Reporter &rr)
{
	List speaks = sp->selected_peaklets();
	if (speaks.empty()) {
		rr.warning("No peaks are selected!");
		return false;
	}

	/*
	 * Throw out peaks that are part of an existing peak group.
	 */
	List peaks;
	for (int pi = 0 ; pi < speaks.size() ; ++pi) {
		Peak *pk = (Peak *) speaks[pi];
		if (!pk->peakgp())
		  peaks.append(pk);
	}
	if (peaks.empty()) {
	  rr.warning("All selected peaks already belong to peak groups.\n");
	  return false;
	}

	/*
	 * Get rid of the peak labels.
	 */
	bool label_deleted = false;
	for (int pi = 0 ; pi < peaks.size() ; ++pi) {
		Peak *pk = (Peak *) peaks[pi];
		if (pk->label()) {
		  delete pk->label();
		  label_deleted = true;
		}
	}

	PeakGp *pg = new PeakGp(sp, peaks);
	pg->select(true);

	/*
	 * Show assignment label.
	 */
	if (label_deleted && pg->label() == NULL && pg->IsAssigned())
	  (void) new Label(pg);

	return true;
}

// ----------------------------------------------------------------------------
//
static void crossdiagonal_ornament_copy(Spectrum *sp, Project &proj,
					Reporter &rr)
{
  if (sp->dimension() != 2 || !sp->is_homonuclear(0, 1))
    {
      rr.warning("Crossdiagonal ornament copy requires "
		 "a homonuclear 2D spectrum");
      return;
    }

  IPoint swap(2);
  swap[0] = 1;
  swap[1] = 0;

  Axis_Map swap_axes;
  swap_axes.set_map(swap);

  List olist = sp->selected_ornaments();
  proj.unselect_all_ornaments();
  List copies = ornament_copy(olist, sp, swap_axes);
  select_ornaments(copies);
}

// ----------------------------------------------------------------------------
//
static void renumber_view(View *v)
{
  Stringy vname = v->spectrum()->name();
  Project &proj = v->session().project();
  vname = proj.unique_view_name(vname, v->is_top_level_window());
  v->set_name(vname);
}
