#!/bin/env python
# -----------------------------------------------------------------------------
# Look for RCS subdirectories in the current directory and recursively in
# subdirectories.  Print out revision comments for all RCS files sorted by
# date, most recent first.
#
# Syntax: rcsview <directory>
#
import os
import string
import sys

# -----------------------------------------------------------------------------
#
class RCS_Revision:

  def __init__(self, path, date, comments):

    self.path = path
    self.date = date
    self.comments = comments
    
# -----------------------------------------------------------------------------
#
def sorted_rcs_changes(directory, output):

  revisions = recurse_rcs_revisions(directory)
  revisions.sort(lambda r1, r2: cmp(r1.date, r2.date))
  revisions.reverse()
  for r in revisions:
    output.write('%s %s\n' % (r.date, r.path))
    for c in r.comments:
      output.write('  ' + c)

# -----------------------------------------------------------------------------
#
def recurse_rcs_revisions(directory):

  revisions = rcs_revisions(directory)
  files = os.listdir(directory)
  paths = map(lambda f, d=directory: os.path.join(d, f), files)
  directories = filter(is_directory, paths)
  for d in directories:
    revisions = revisions + recurse_rcs_revisions(d)
  return revisions

# -----------------------------------------------------------------------------
#
def is_directory(path):

  import stat
  s = os.stat(path)
  return stat.S_ISDIR(s[stat.ST_MODE])

# -----------------------------------------------------------------------------
#
def rcs_revisions(rcs_directory):

  files = os.listdir(rcs_directory)
  rcs_files = filter(lambda f: f[-2:] == ',v', files)
  rcs_paths = map(lambda f, d=rcs_directory: os.path.join(d, f), rcs_files)
  if rcs_paths:
    paths = string.join(rcs_paths, ' ')
    command = 'rlog ' + paths
    rlog_output = os.popen(command, 'r')
    revisions = revision_list(rlog_output)
    rlog_output.close()
  else:
    revisions = []
  return revisions

# -----------------------------------------------------------------------------
#
def revision_list(input):
  
  path = None
  date = None
  comments = []
  revisions = []

  while 1:
    line = input.readline()
    rf = 'RCS file: '
    if has_prefix(line, rf):
      path = line[len(rf):-1]
    elif has_prefix(line, 'revision '):
      date_line = input.readline()
      dt = 'date: '
      date = date_line[len(dt):len(dt)+19]
      while 1:
        comment = input.readline()
        if (has_prefix(comment, '----------------------------') or
            has_prefix(comment, '============================')):
          break
        comments.append(comment)
      r = RCS_Revision(path, date, comments)
      revisions.append(r)
      date = None
      comments = []
    elif line == '':
      break

  return revisions

# -----------------------------------------------------------------------------
#
def has_prefix(string, prefix):

  return string[:len(prefix)] == prefix

# -----------------------------------------------------------------------------
#
if len(sys.argv) == 2:
  dir = sys.argv[1]
else:
  m = ('Syntax: rcsview <directory>\n' +
       'Shows rcs changes for all files under RCS below given directory\n' +
       'sorted by date, most recent first.\n')
  sys.stderr.write(m)
  sys.exit(1)
             
sorted_rcs_changes(dir, sys.stdout)
