////////////////////////////////////////////////////////////////////////////////
/// controllers.js
///
/// DISCLAIMER
///
/// Copyright 2010 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Claudius Weinberger
/// @author Copyright 2009-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

// controllers functions -----------------------------------------------

if(!window['voc']){ 
  // a object voc to namespace the code and hold the config is expected
  // in global scope.
  // has to be defined before including controller.js and/or triBase.js
  throw "Please define a global object named voc."; 
} 


(function(voc){ 
  // stack is an augmented array holding functions   
  // each controller has to manage the page it displays. 
  // if the controller-functions does not have a property 'inited' set to true
  // stack will try to call the property function 'init()' on it when adding.
  // TODO for production this should be minified (along with widgets.js) into voclib.js 
  voc.stack = []; 
  voc.stack.add = function(controller, args){
    args = args || [];
    if(controller.inited || controller.init()){
      controller.apply(null, args);  
      this.push({fun: controller, args: args}); 
    }     
  };
  voc.stack.back = function(){ 
    this.pop(); // discard upmost controller
    var c = this[this.length-1];  // get the top controller
    c.fun.apply(null, c.args || []);    // execute
  }; 
  voc.stack.clear = function(){ 
    while(this.lenth>0){ 
      this.pop();
    } 
  }; 
})(voc); 


(function(voc){  
  // everything is wrapped in a self-calling function to prevent namespace 
  // leakage and increase minifiability.
  // function is called with global object voc as parameter. other global 
  // objects are not used. 


  // the possible chart-typed are defined here.  
  var chartTypes = [
    { 
      name: 'rest', 
      label: 'RESTful Requests', 
      available: ["rest", "restValueDelete", "restValuePut", "restValuePost", "restValueGet", "restInvalid", "restFlush", "restPrefix", "restVersion", "restStats"], 
      values:  ['rest','restValuePost', 'restValuePut', 'restValueGet']
    }, 
    {
      name: 'memcache', 
      label: 'Memcache Requests',
      available: ["memcache", "memcacheInvalid", "memcacheAdd", "memcacheAppend", "memcacheCas", "memcacheDecr", "memcacheDelete", "memcacheFlushAll", "memcacheGet", "memcacheGets", "memcacheIncr", "memcachePGet", "memcachePGets", "memcachePrepend", "memcacheReplace", "memcacheSet", "memcacheStat", "memcacheVersion"], 
      values: ['memcacheInvalid', 'memcacheAdd', 'memcacheStat']
    }, 
    { 
      name: 'postfix', 
      label: 'Postfix Requests',
      available: ["postfix", "postfixInvalid", "postfixGet", "postfixPut"],
      values: ['postfixGet', 'postfixPut']
    },
    { 
      name: 'connection', 
      label: 'Number of Connections',
      available: ["connections", "httpConnections", "lineConnections"],
      values:  ["httpConnections", "lineConnections"],
      type: 'connection', 
      _getValueFromDatasource: function(valuename, datasource){
        // this function works for the listView 
        // but breaks the chart view (kind-of)
        var period = 'minute'; 
        var start = datasource['runtime'][period].start;
        var items = datasource['connection'][period][valuename]['count']; 
        return [start[items.length-1], items[items.length-1]]; 
      } 
    }, 
    { 
      name: 'memory', 
      label: 'Memory Usage',
      available: 'numberKeys,limitKeys,totalKeySize,limitKeySize,limitTotalKeySize,totalDataSize,limitDataSize,limitTotalDataSize,limitTotalSize'.split(','), 
      values:  ['numberKeys', 'totalKeySize'],
      getValueFromDatasource: function(valuename,datasource){ 
        // this function get called from the flotChart Widget and the values
        // get appended to 
        var val = datasource.memory[valuename]; 
        return [ new Date().getTime(), val ];
      }, 
      //maxValuesOnXAxis : 10 // never used again?
    }, 
    { 
      name: 'logouts', 
      label: 'Logout', 
      available: ['count'], 
      values: ['count'], 
      getDatasetsFromDataSource: function(data, period, values, type){ 
        // this function will replace the more generic function of the same
        // name in widgets.js
        var sets = []; 
        var start = data.start; // all the timestamps
        for(var iV=0; iV<values.length; iV++){
          var valueName = values[iV];
          var items = data.logouts[valueName];
          var set = []; 
          for(var i=0; i<items.length; i++){ 
            if(start[i] > 0){  // sometimes there are timestamps 0
              set.push([ start[i] * 1000, items[i]]); 
            } 
          } 
          sets.push(set); 
        }  
        return sets;        
      } 
    }, 
    { 
      name: 'duration', 
      label: 'Duration', 
      available: ['mean', 'min', 'max'], 
      values: ['mean', 'min', 'max'], 
      getDatasetsFromDataSource: function(data, period, values, type){ 
        // this function will replace the more generic function of the same
        // name in widgets.js
        var sets = []; 
        var start = data.start; // all the timestamps
        for(var iV=0; iV<values.length; iV++){
          var valueName = values[iV];
          var items = data.logouts[valueName]; 

          var set = []; 
          for(var i=0; i<items.length; i++){ 
            items[i] = parseFloat(items[i]); 
            if(isNaN(items[i])){ items[i] = 0; } 
            if(start[i] > 0){  // sometimes there are timestamps 0
              set.push([ start[i] * 1000, items[i]]); 
            } 
          } 
          sets.push(set); 
        }  
        return sets;        
      }, 
      linefillFun: function(){  return 0; } 
    }, 
    { 
      name: 'deviation', 
      label: 'Deviation', 
      available: ['deviation'], 
      values: ['deviation'], 
      showLines: false, 
      showBars: true, 
      xaxis_autoscale: 0.1,
      getDatasetsFromDataSource: function(data, period, values, type){ 
        // this function will replace the more generic function of the same
        // name in widgets.js
        var sets = []; 
        var start = data.start; // all the timestamps
        for(var iV=0; iV<values.length; iV++){
          var valueName = values[iV];
          var items = data.logouts[valueName]; 

          var set = []; 
          for(var i=0; i<items.length; i++){ 
            items[i] = parseFloat(items[i]); 
            if(isNaN(items[i])){ items[i] = 0; } 
            if(start[i] > 0){  // sometimes there are timestamps 0
              set.push([ start[i] * 1000, items[i]]); 
            } 
          } 
          sets.push(set); 
        }  
        return sets;        
      }, 
      linefillFun: function(){  return 0; } 
    },
    { 
      name: 'distribution', 
      label: 'Distribution', 
      available: ['60', '120', '300', '600', '1800', '3600', '7200', '+'], 
      values: ['60', '120', '300', '600', '1800', '3600', '7200', '+'], 
      showLines: false, 
      showBars: true, 
      stack: 1,
      lineSteps: true, 
      xaxis_autoscale: 0.1,
      seriesColors: ['#FA0202', '#0217FA', '#28FA02', '#FFFB1F', '#1FFFFB', '#F41FFF', '#FF441F','#B8B8B8', '#4F0680' ], 
      getDatasetsFromDataSource: function(data, period, values, type){ 
        // this function will replace the more generic function of the same
        // name in widgets.js
        //
        //  that means 'this' will change to the flotChart object 
        var sets = []; 
        var start = data.start; // all the timestamps
        var items = data.logouts.distribution; 
        var set = []; 
        for(var i=0; i<items.length; i++){ 
          if(start[i] > 0){  // sometimes there are timestamps 0
            for(var ii=0; ii<this.options.values.length; ii++){ 
              sets[ii] = sets[ii] || []; 
              sets[ii].push([start[i]*1000,parseFloat(items[i][ii])]); 
            } 
          } 
        } 
        return sets;        
      }, 
      linefillFun: function(){ return 0; },
      initType: function(flotChart){ 
        // initType will be called by flotChart when the type is applied, after the values of type
        // are mixed-in with chartType.options.
        if(flotChart.options.datasource){ 
          this.values = this.available = $.map(flotChart.options.datasource.cuts, function(elm){ return String(elm) });
          this.available.push('+'); 
        } 
      } 
    }
  ];  




  // controllers is an object holding functions that bring the application 
  // in a certain desired state. like pages. 
  // every controller-function can have a init-member-function that will be
  // called by the controllerstack when the controller is called for the 
  // first time. 
  // added to the global voc object.
  voc.controllers = {}; 

  // controller for monitor-page with small charts ------------------------
  voc.controllers.monitor = function(){ 
    $('.page').hide(); 
    // TODO trigger an update of the charts here
    //voc.controllers.monitor.updateAllCharts(function(){
      $('#monitorPage').show();
      $('.flotChart').each(function(){
        $(this).flotChart('updateFromDatasource');
      });
    //});
    $('#overlay').empty();
  }; 
  voc.controllers.monitor.filterChartTypes = function(filter){ 
    // filter the chartTypes-array above by the configured 
    // available charttypes in voc.conf.charts.chartTypes
    chartTypes = $.map(chartTypes, function(ct,index){ 
      for(var i=0; i<filter.length; i++){ 
        if(filter[i]===index){ 
          return ct; 
        } 
      } 
      return null;
    }); 
    return chartTypes; 
  }; 

  voc.controllers.monitor.updateAllCharts = function (cb){
    // this get called in an interval, gets the data from the server
    // and updates the charts
      var statsurl = voc.type === 'simple' ? voc.conf.urls.stats : voc.conf.urls.logoutStats;
    $.ajax({
      url:    statsurl,
      dataType: 'json', 
      success: function(data) {
       // if($('#monitorPage').is(':visible')){ 
          $('.flotChart').each(function(){
            $(this).flotChart('updateFromDatasource', data);
          });
       // }
        if(cb){ cb(data); } 
      }
    });
  }; 

  voc.controllers.monitor.init = function(){
    this.filterChartTypes(voc.conf.charts.chartTypes); 
    $('#monitorPage').show();
 
    // TODO: this needs a better abstraction
    // there may be more than one url to recieve stats-data from
    var statsurl = voc.type === 'simple' ? voc.conf.urls.stats : voc.conf.urls.logoutStats;
    $.ajax({
      url:    statsurl,
      dataType: 'json', 
      success: function(data) {
        for(var i=0; i<chartTypes.length; i++){
          // add a chart of every type
          $('<li />', {id:'flotChart_'+i}).appendTo($('#monitorCharts')); 
          $('#flotChart_'+i).flotChart({ 
            chartType : chartTypes[i], 
            chartTypes: chartTypes, 
            datasource: data,  
            width: voc.conf.charts.small.width, 
            height: voc.conf.charts.small.height 
          }).find('.chartContainer').click(function(e){
            var opts = $(this).parents('.flotChart').flotChart('getOptions');
            voc.stack.add(voc.controllers.monitorDetail, [opts] );  
          });
        } 
        $('<img />', { 
          src: 'images/add-btn.gif',
          alt:'add', 
          className:"monitorAddChart", 
          click: function(){ 
            try { 
              var i =  parseInt($('#monitorPage').find('.flotChart').last().attr('id').split('_')[1],10) + 1; 
            } catch (err){ 
              if(err.name === 'TypeError'){ 
                var i = 0; 
              } else { 
                throw err;
              }  
            } 
            i = i || 0; 
            $('<li />', { id:'flotChart_'+i }).appendTo($('#monitorCharts')); 
            $('#flotChart_'+i).flotChart({ 
              chartType : chartTypes[0], 
              chartTypes: chartTypes, 
              datasource: data,  
              width: voc.conf.charts.small.width, 
              height: voc.conf.charts.small.height 
            }).find('.chartContainer').click(function(e){
              var opts=$(this).parents('.flotChart').flotChart('getOptions');
              voc.stack.add(voc.controllers.monitorDetail, [opts]);  
            });
          }
        }).appendTo($('#monitorPage')); 
      }
    });


    // this get called in an interval, gets the data from the server
    // and updates the charts
    /*
    var updateAllCharts = function (){
      $.ajax({
        url:    statsurl,
        dataType: 'json', 
        success: function(data) {
          $('.flotChart').each(function(){
            $(this).flotChart('updateFromDatasource', data);
          });
        }
      });
    }; 
    */ 

    this.intervals = [];
    this.intervals.push(window.setInterval(voc.controllers.monitor.updateAllCharts, voc.conf.charts.updateInterval));
    /* 
    // a random number chart data test. displays continiously updating rnd-nums. 
    var mockdata = [[new Date().getTime(), Math.floor(Math.random()*10+1)]];
    var appendtomd = function(){ 
      mockdata.push([new Date().getTime(), Math.floor(Math.random()*10+1)]) 
      if(mockdata.length > 15){mockdata = mockdata.slice(mockdata.length-15);} 
      $('#flotChart_100').flotChart('update', [mockdata]);    
    }; 
    $('<li />', { id: 'flotChart_100',  }).appendTo($('#monitorCharts'))
    .flotChart({ chartType : {
      name: 'random', label: 'Random Numbers', notInDatasource: true
    }, rawDatasets: [mockdata], values: ['RND']  });
    this.intervals.push(window.setInterval(appendtomd, 1000));
    */ 
    this.inited = true; 
    return true;
  }; 
  voc.controllers.monitorDetail = function(options){ 
    $('body').append('<div id="overlay"></div>'); 
    var ol = $('#overlay').empty().attr('class',''); 
    var statsurl = voc.type === 'simple' ? voc.conf.urls.stats : voc.conf.urls.logoutStats;
    $.ajax({
      url: statsurl,
      dataType: 'json', 
      success: function(data) {
        var div = $('<div />').appendTo(ol); 
        options.width = voc.conf.charts.big.width; 
        options.height = voc.conf.charts.big.height; 
        options.datasource = data; 
        // big chart need no removebutton in options
        options.removable = false; 
        var fc = $(div).flotChart(options);
        $('#overlay').overlay({
          load:true,
          top: 150,
          left: 350,
          mask: {
            color: '#000',
            loadSpeed: 200,
            opacity: 0.8
          }, 
          onClose: function(){
            fc.flotChart('destroy');
            $('#overlay').remove(); 
            voc.stack.back();         
          } 
        }); 
      }
    });
  }; 
  voc.controllers.monitorDetail.inited = true; 

  // controller for logviewer -----------------------------------
  voc.controllers.logviewer = function(){ 
    $('.page').hide(); 
    $('#pageLog').show();
  }; 

  voc.controllers.logviewer.init = function(){ 
    $('#logTableContainer').logTable({
      url: voc.conf.urls.logs
    });
    this.inited = true; 
    return this.inited;
  };


  // controller for config editor  ------------------------------
  voc.controllers.configeditor = function(){ 
    $('.page').hide(); 
    $('#pageConf').show();
  };

  voc.controllers.configeditor.init = function(){ 
    var confedOptions = { 
      url: voc.conf.urls.backendconf 
    }; 
    // optionally pass options for fields. 
    if(voc.conf.configeditor.fields){ 
      confedOptions['fields'] = voc.conf.configeditor.fields; 
    } 
    $('#configEditorContainer').configEditor(confedOptions);
    return this.inited = true;
  };

  // controller for rest interface
  voc.controllers.restEditor = function(){ 
    $('.page').hide(); 
    $('#pageRest').show();
  }; 
  voc.controllers.restEditor.init = function(){ 
    var keylist = $('#pageRest .keys').keyList({
      pagination: voc.conf.datainterface.keylistpagination,
      keysearchurl:  voc.conf.urls.keysearch, 
      keyclick: function(e,key){
        $('#pageRest .restForm').find('.keyInput').val(key);
        $('#pageRest .restForm').restForm('httpGet'); 
      }
    }); 

    var xkv = $('#pageRest .extendedKeyValDiv').extendedKeyVals({
      //types: voc.conf.datainterface.extendedvals.types
      typesUrl: voc.conf.urls.ekvTypes
    });

    $('#pageRest .restFormDiv').restForm({
      url: voc.conf.urls.getvalue,
      keylist: keylist, 
      // TODO doc this widget-obj
      extendedvals: xkv.data('extendedKeyVals'),
      result: function(e, args){ 
        var data =args[0],
            code = args[1], 
            xhr = args[2], 
            method = args[3],
            url = args[4]; 
        if(typeof data === 'object'){
          data = JSON.stringify(data);
        }
        $('.restDisplay').httpRequestList('addCycle', method, url, xhr, data); 
        xkv.data('extendedKeyVals').loadMap(xhr);      
      }, 
      onKey: function(e, args){
        var key = args[0];
        $('.keys').keyList('add', key); 
      } 
    });

    $('.restDisplay').httpRequestList({}); 
    return this.inited = true;
  };

  // -------------------------------------------

  voc.controllers.login = function(){ 
    $('.page').hide(); 
    $('#pageLogin').show();
  }; 

  voc.controllers.login.init = function(){ 
    $('#loginContainer').loginForm({
      url: voc.conf.urls.login.replace('{sid}', voc.session.sid), 
      loggedin: function(e, session){ 
        voc.session.rights.push.apply(voc.session.rights, session.rights);
        voc.session.sid = session.sid;
        voc.session.user = session.user;
        voc.session.init();
        voc.stack.back(); 
        $('#loginlink').text('Logout').unbind().click(function(){ 
          voc.stack.clear();
          voc.session.get(function(){voc.buildNavi();});
          voc.stack.add(voc.controllers.login); 
          $(this).text('Login');
        });
      },
      cancel: function(){ 
        voc.stack.back(); 
      } 
    });
    this.inited = true; 
    return this.inited;
  };

  // -------------------------------------------

  voc.controllers.userManager = function(){ 
    $('.page').hide(); 
    $('#pageUserManager').show();
  }; 

  voc.controllers.userManager.init = function(){ 
    function addBtn(){ // local fun
      $('<img />', { 
        src: 'images/add-btn.gif',
        alt:'add', 
        click: function(){ 
          $(this).replaceWith(
            $('#userDetailContainer').userDetail('newUserForm')
          ); 
        }
      }).appendTo($('#newUserContainer').empty()); 
    }; 

    $('#userListContainer').userList({
      url: voc.conf.urls.allUsers,
      userclick : function(e, args){ 
        var username = args[0];
        $('#userDetailContainer').userDetail('getUserData', username);
      } 
    });

    $('#userDetailContainer').userDetail({
      url: voc.conf.urls.userInfo.replace('{sid}', voc.session.sid),
      rights: voc.session.rightNames, 
      newUserUrl: voc.conf.urls.newUser.replace('{sid}', voc.session.sid), 
      newuser: function(e, data){ 
        $('#userListContainer').userList('addItem', data);
        addBtn();
        $('#userDetailContainer').userDetail('getUserData', data.name);
      },  
      userdeleted: function(e, uname, data){ 
        $('#userListContainer').userList('removeItem', uname);
        $('#userDetailContainer').userDetail('empty');
        
        
      } 
    });

    addBtn();



    this.inited = true; 
    return this.inited;
  };

  // -------------------------------------------
/* 
  voc.controllers.test.init = function(){ 
    $('body').append('<div id="overlay"></div>'); 
    var ol = $('#overlay').empty().attr('class',''); 
    // a random number chart data test. displays continiously updating rnd-nums. 
    var mockdata = [[new Date().getTime(), Math.floor(Math.random()*10+1)]];
    var appendtomd = function(){ 
      mockdata.push([new Date().getTime(), Math.floor(Math.random()*10+1)]) 
      if(mockdata.length > 15){mockdata = mockdata.slice(mockdata.length-15);} 
      $('#flotChart_100').flotChart('update', [mockdata]);    
    }; 
    $('<li />', { id: 'flotChart_100',  }).appendTo($('#monitorCharts'))
    .flotChart({ chartType : {
      name: 'random', label: 'Random Numbers', notInDatasource: true
    }, rawDatasets: [mockdata], values: ['RND']  });
    this.intervals.push(window.setInterval(appendtomd, 1000));
    this.inited=true; 
  }; 
  //*/ 
 
  // -------------------------------------------
  voc.highlightNavi = function(id){ 
    // sets the proper css-class for the navigation
    // TODO this should probably not be a global func
    $('#subnavi li').removeClass('current_page_item'); 
    $('#subnavi #' + id).parents('li').addClass('current_page_item'); 
  } 
})(voc); 
