////////////////////////////////////////////////////////////////////////////////
/// @brief SimpleVOC server
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Dr. Oreste Costa-Panaia
/// @author Copyright 2010, triagens GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include <Basics/Common.h>

#include <boost/program_options.hpp>

#include <Admin/ApplicationAdminServer.h>
#include <Basics/Logger.h>
#include <Rest/AnyServer.h>
#include <Rest/ApplicationHttpServer.h>
#include <Rest/ApplicationLineServer.h>
#include <Rest/ApplicationServer.h>
#include <Rest/HttpHandlerFactory.h>

#include "HttpServer/PathHandler.h"
#include "MCHandler/MCHandlerFactory.h"
#include "PostfixHandler/PostfixHandlerFactory.h"
#include "PostfixHandler/Transformation.h"
#include "RestHandler/RestAdminBeConfigurationHandler.h"
#include "RestHandler/RestAdminExtendedKeyValues.h"
#include "RestHandler/RestAdminFeConfigurationHandler.h"
#include "RestHandler/RestAdminSearchHandler.h"
#include "RestHandler/RestFlushHandler.h"
#include "RestHandler/RestPrefixHandler.h"
#include "RestHandler/RestStatisticsOverviewHandler.h"
#include "RestHandler/RestValueHandler.h"
#include "RestHandler/RestVersionHandler.h"

using namespace triagens;
using namespace triagens::basics;
using namespace triagens::rest;
using namespace triagens::admin;
using namespace triagens::simple;

namespace bpo = boost::program_options;

////////////////////////////////////////////////////////////////////////////////
// @brief simple voc server
////////////////////////////////////////////////////////////////////////////////

namespace {
  class SimpleVoc : public AnyServer {
    public:
      SimpleVoc (int argc, char** argv)
        : argc(argc),
          argv(argv) {
        workingDirectory = "/var/tmp";
      }


    public:

      ////////////////////////////////////////////////////////////////////////////////
      /// {@inheritDoc}
      ////////////////////////////////////////////////////////////////////////////////

      void buildApplicationServer () {
        applicationServer = ApplicationServer::create("[<options>] - starts the triAGENS SimpleVOC", basics::version());
        applicationServer->setSystemConfigFile("simplevoc.conf");
        applicationServer->setUserConfigFile(".simplevoc/simplevoc.conf");

        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        // allow multi-threading scheduler
        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

        applicationServer->allowMultiScheduler(true);

        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        // configure the application as a line server
        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

        applicationLineServer = ApplicationLineServer::create(applicationServer);
        applicationServer->addFeature(applicationLineServer);

        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        // and as a http server
        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

        applicationHttpServer = ApplicationHttpServer::create(applicationServer);
        applicationServer->addFeature(applicationHttpServer);

        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        // and start a simple user manager
        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

        applicationAdminServer = ApplicationAdminServer::create(applicationServer);
        applicationServer->addFeature(applicationAdminServer);

        applicationAdminServer->allowAdminDirectory();
        applicationAdminServer->allowLogViewer();
        applicationAdminServer->allowSessionManagement();


        vector<right_t> rightsAnonymous;

        applicationAdminServer->setAnonymousRights(rightsAnonymous);


        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        // create an instance of the simpleVoc model
        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

        model = new SimpleModel();

        map<ApplicationServer::section_e, ProgramOptionsDescription> additional;

        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        // daemon and supervisor mode
        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

        additional[ApplicationServer::OPTIONS_CMDLINE]
          ("daemon", "run as daemon")
          ("supervisor", "starts a supervisor and runs as daemon")
          ("change-directory", &workingDirectory, "change into working directory")
          ("pid-file", &pidFile, "pid-file in daemon mode")
        ;

        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        // for this server we display our own options such as port to use
        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

        applicationLineServer->showPortOptions(false);
        applicationHttpServer->showPortOptions(false);

        ProgramOptionsDescription portsCmd("Connection Options");

        portsCmd
          ("server.http-port", &httpPorts, "ports for HTTP access")
          ("server.memcached-port", &memcachedPorts, "ports for MEMCACHED access")
          ("server.postfix-port", &postfixPorts, "ports for POSTFIX hash table access")
          ("server.admin-port", &adminPorts, "ports for admin access")
          ("server.fe-configuration",&feConfiguration, "file to store the front-end preferences")
          ("server.extended-key-values", &extendedKeyValues, "extended key values in the form <name>:<type>")
        ;

        additional[ApplicationServer::OPTIONS_SERVER](portsCmd);

        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        // display the command line options used for size restrictions
        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

        limitValues.maxNumKeys = 0;
        limitValues.maxKeySize = 0;
        limitValues.maxValueSize = 0;
        limitValues.maxTotalKeySize = 0;
        limitValues.maxTotalSize = 0;
        limitValues.maxTotalValueSize = 0;
        limitValues.limitsInUse = false;

        additional[ApplicationServer::OPTIONS_LIMITS]
          ("limit.key-size", &limitValues.maxKeySize, "maximum size of a key")
          ("limit.value-size", &limitValues.maxValueSize, "maximum size of a value")
          ("limit.num-keys", &limitValues.maxNumKeys, "maximum number of keys to be stored")
          ("limit.total-key-size", &limitValues.maxTotalKeySize, "maximum size allocated for key storage")
          ("limit.total-value-size", &limitValues.maxTotalValueSize, "maximum size allocated for value storage")
          ("limit.total-size", &limitValues.maxTotalSize, "maximum size allocated for key and value storage")
        ;

        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        // display the command line options used for postfix
        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

        ProgramOptionsDescription postfixCmd("Postfix Options");

        postfixCmd
          ("postfix.transformation", &postfixTransformations, "transformation of postfix key get request")
        ;

        additional[ApplicationServer::OPTIONS_SERVER](postfixCmd);

        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        // parse the command line options - exit if there is a parse error
        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

        if (! applicationServer->parse(argc, argv, additional)) {
          exit(EXIT_FAILURE);
        }

        if (applicationServer->programOptions().has("daemon")) {
          daemonMode = true;
        }

        if (applicationServer->programOptions().has("supervisor")) {
          supervisorMode = true;
        }

        if (daemonMode) {
          if (pidFile.empty()) {
            LOGGER_FATAL << "no pid-file defined, but daemon mode requested";
            exit(EXIT_FAILURE);
          }
        }
      }

      ////////////////////////////////////////////////////////////////////////////////
      /// {@inheritDoc}
      ////////////////////////////////////////////////////////////////////////////////

      int startupServer () {

        // add the http ports
        vector<AddressPort> httpAP;

        for (vector<string>::iterator i = httpPorts.begin();  i != httpPorts.end();  ++i) {
          AddressPort ap(*i);
          httpAP.push_back(ap);
        }

        // add the admin ports
        vector<AddressPort> adminAP;

        for (vector<string>::iterator i = adminPorts.begin();  i != adminPorts.end();  ++i) {
          AddressPort ap(*i);
          adminAP.push_back(ap);
        }

        // add the memcache ports
        vector<AddressPort> memcachedAP;

        for (vector<string>::iterator i = memcachedPorts.begin();  i != memcachedPorts.end();  ++i) {
          AddressPort ap(*i);
          memcachedAP.push_back(ap);
        }

        // add the postfix ports
        vector<AddressPort> postfixAP;

        for (vector<string>::iterator i = postfixPorts.begin();  i != postfixPorts.end();  ++i) {
          AddressPort ap(*i);
          postfixAP.push_back(ap);
        }

        // if user has set one or more limit values, then pass these onto the model
        if ( (limitValues.maxNumKeys != 0)
          || (limitValues.maxKeySize != 0)
          || (limitValues.maxValueSize != 0)
          || (limitValues.maxTotalKeySize != 0)
          || (limitValues.maxTotalSize != 0)
          || (limitValues.maxTotalValueSize != 0) ) {
          limitValues.limitsInUse = true;
          model->setLimitValues(limitValues);
        }

        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        // convert transformations
        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

        vector<Transformation> transformations;

        for (vector<string>::const_iterator i = postfixTransformations.begin();  i != postfixTransformations.end();  ++i) {
          Transformation transformation(*i);

          transformations.push_back(transformation);
        }

        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        // add extended key values
        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

        for (vector<string>::const_iterator i = extendedKeyValues.begin();  i != extendedKeyValues.end();  ++i) {
          bool ok = model->addExtendedKeyValueDescription(*i);

          if (! ok) {
            LOGGER_FATAL << "cannot parse extended key-value definition '" << *i << "'";
            exit(EXIT_FAILURE);
          }
        }

        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        // and finalize the module
        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

        model->finalise();

        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        // create the various parts of the simpleVoc server
        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

        applicationServer->buildScheduler();
        applicationServer->buildControlCHandler();

        model->setScheduler(applicationServer->scheduler());

        // memcache
        MCHandlerFactory* mcHandlerFactory = new MCHandlerFactory(model);
        applicationLineServer->buildServer(mcHandlerFactory, memcachedAP);

        // postfix
        postfixHandlerFactory = new PostfixHandlerFactory(model, transformations);
        applicationLineServer->buildServer(postfixHandlerFactory, postfixAP);

        // http
        HttpHandlerFactory* httpHandlerFactory = new HttpHandlerFactory();

        httpHandlerFactory->addHandler("/flush", RestFlushHandler::create, (void*) model);
        httpHandlerFactory->addHandler("/version", RestVersionHandler::create, (void*) model);
        httpHandlerFactory->addPrefixHandler("/keys", RestPrefixHandler::create, (void*) model);
        httpHandlerFactory->addPrefixHandler("/value", RestValueHandler::create, (void*) model);

        applicationHttpServer->buildServer(httpHandlerFactory, httpAP);

        // admin
        HttpHandlerFactory* adminHandlerFactory = new HttpHandlerFactory();

        adminHandlerFactory->addHandler("/flush", RestFlushHandler::create, (void*) model);
        adminHandlerFactory->addPrefixHandler("/keys", RestPrefixHandler::create, (void*) model);
        adminHandlerFactory->addPrefixHandler("/value", RestValueHandler::create, (void*) model);

        adminHandlerFactory->addHandler("/admin/be-configuration", RestAdminBeConfigurationHandler::create, (void*) model);
        adminHandlerFactory->addHandler("/admin/be-configuration.html", RestAdminBeConfigurationHandler::createHtml, (void*) model);
        adminHandlerFactory->addHandler("/admin/fe-configuration", RestAdminFeConfigurationHandler::create, (void*) feConfiguration.c_str());
        adminHandlerFactory->addHandler("/admin/extended-key-values", RestAdminExtendedKeyValues::create, (void*) model);
        adminHandlerFactory->addHandler("/admin/statistics/overview", RestStatisticsOverviewHandler::create, (void*) model);
        adminHandlerFactory->addHandler("/admin/statistics/overview.html", RestStatisticsOverviewHandler::createHtml, (void*) model);
        adminHandlerFactory->addPrefixHandler("/admin/search", RestAdminSearchHandler::create, (void*) model);

        adminHandlerFactory->addHandler("/version", RestVersionHandler::create, 0);

        // user and session management
        applicationAdminServer->addHandlers(adminHandlerFactory, "/admin");

        applicationHttpServer->buildServer(adminHandlerFactory, adminAP);

        // done with setup
        LOGGER_INFO << "SimpleVOC is ready for business";

        applicationServer->start();
        applicationServer->wait();

        LOGGER_INFO << "SimpleVOC has been shut down";

        delete model;

        return 0;
      }

    private:
      int argc;
      char** argv;

      ApplicationLineServer* applicationLineServer;
      ApplicationHttpServer* applicationHttpServer;
      ApplicationAdminServer* applicationAdminServer;

      vector<string> httpPorts;
      vector<string> memcachedPorts;
      vector<string> postfixPorts;
      vector<string> adminPorts;

      vector<string> extendedKeyValues;

      LimitValues limitValues;

      vector<string> postfixTransformations;

      PostfixHandlerFactory* postfixHandlerFactory;

      SimpleModel* model;

      string feConfiguration;
  };
}

////////////////////////////////////////////////////////////////////////////////
// @brief creates an application server
////////////////////////////////////////////////////////////////////////////////

int main (int argc, char* argv[]) {
  TRIAGENS_HPDF_INITIALISE;

  // create SimpleVoc server
  SimpleVoc server(argc, argv);

  return server.start();
}
