////////////////////////////////////////////////////////////////////////////////
/// @brief session handler
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @auther Martin Schoenert
/// @author Copyright 2011-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "SessionHandler.h"

#include <Admin/ApplicationAdminServer.h>
#include <Basics/MutexLocker.h>
#include <Basics/VariantArray.h>
#include <Basics/VariantBoolean.h>
#include <Basics/VariantInt32.h>
#include <Basics/VariantVector.h>
#include <Rest/HttpRequest.h>
#include <Rest/HttpResponse.h>

#include "SessionManager/Session.h"


using namespace triagens::basics;
using namespace triagens::rest;

namespace triagens {
  namespace admin {

    // -----------------------------------------------------------------------------
    // constructors and destructors
    // -----------------------------------------------------------------------------

    SessionHandler::SessionHandler (HttpRequest* request, ApplicationAdminServer* server)
      : RestBaseHandler(request), _server(server) {
    }

    // -----------------------------------------------------------------------------
    // HttpHandler methods
    // -----------------------------------------------------------------------------

    HttpHandler::status_e SessionHandler::execute () {

      // execute the request
      switch (request->requestType()) {
        case HttpRequest::HTTP_REQUEST_POST: return executePost();
        case HttpRequest::HTTP_REQUEST_GET: return executeGet();
        case HttpRequest::HTTP_REQUEST_PUT: return executePut();
        case HttpRequest::HTTP_REQUEST_DELETE: return executeDelete();

        default:
          generateError(ERROR_ILLEGAL_METHOD, true, "method not supported");
          return HANDLER_DONE;
      }
    }

    // -----------------------------------------------------------------------------
    // private methods
    // -----------------------------------------------------------------------------

    HttpHandler::status_e SessionHandler::executePost () {
      if (request->suffix().size() != 0) {
        generateError(ERROR_ILLEGAL_PARAMETER, true, "expecting POST <prefix>/session");
        return HANDLER_DONE;
      }

      // create a new session
      MUTEX_LOCKER(guard, Session::lock);

      Session* session = Session::create();

      generateSession(session);
      return HANDLER_DONE;
    }



    HttpHandler::status_e SessionHandler::executeGet () {
      vector<string> const& suffix = request->suffix();

      if (suffix.size() != 1) {
        generateError(ERROR_ILLEGAL_PARAMETER, true, "expecting GET <prefix>/session/<sid>");
        return HANDLER_DONE;
      }

      string const& sid = suffix[0];

      // lookup an existing session
      MUTEX_LOCKER(guard, Session::lock);

      Session* session = Session::lookup(sid);

      if (session != 0) {
        generateSession(session);
      }
      else {
        generateError(ERROR_ILLEGAL_PARAMETER, true, "unknown session");
      }

      return HANDLER_DONE;
    }



    HttpHandler::status_e SessionHandler::executePut () {
      vector<string> const& suffix = request->suffix();

      if (suffix.size() != 2) {
        generateError(ERROR_ILLEGAL_PARAMETER, true, "expecting PUT <prefix>/session/<sid>/<method>");
        return HANDLER_DONE;
      }

      string const& sid = suffix[0];
      string const& method = suffix[1];

      // lookup an existing session
      MUTEX_LOCKER(guard, Session::lock);

      Session* session = Session::lookup(sid);

      if (session != 0) {

        generateError(ERROR_ILLEGAL_PARAMETER, true, "unknown method '" + method + "'");
        return HANDLER_DONE;
      }
      else {
        generateError(ERROR_ILLEGAL_PARAMETER, true, "unknown session");
      }

      return HANDLER_DONE;
    }



    HttpHandler::status_e SessionHandler::executeDelete () {
      vector<string> const& suffix = request->suffix();

      if (suffix.size() != 1) {
        generateError(ERROR_ILLEGAL_PARAMETER, true, "expecting DELETE <prefix>/session/<sid>");
        return HANDLER_DONE;
      }

      string const& sid = suffix[0];

      // lookup an existing session
      MUTEX_LOCKER(guard, Session::lock);

      Session* session = Session::lookup(sid);
      bool removed = false;

      if (session != 0) {
        removed = Session::remove(session);
      }

      VariantArray* result = new VariantArray();
      result->add("removed", new VariantBoolean(removed));

      generateResult(result);
      return HANDLER_DONE;
    }




    void SessionHandler::generateSession (Session* session) {
      VariantArray* result = new VariantArray();
      result->add("sid", session->getSid());

      VariantVector* r = new VariantVector();

      set<right_t> const& rights = Session::anonymousRights();

      for (set<right_t>::const_iterator i = rights.begin();  i != rights.end();  ++i) {
        r->add(new VariantInt32(*i));
      }

      result->add("rights", r);

      generateResult(result);
    }

  }
}
