////////////////////////////////////////////////////////////////////////////////
/// @brief abstract base class for result generators
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Achim Brandt
/// @author Copyright 2008-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#ifndef FYN_RESULTGENERATOR_RESULT_GENERATOR_H
#define FYN_RESULTGENERATOR_RESULT_GENERATOR_H 1

#include <Basics/VariantObject.h>
#include <Basics/Exceptions.h>

namespace triagens {
  namespace basics {
    struct StringBuffer;
  }

  namespace rest {

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief abstract base class for result generators
    ////////////////////////////////////////////////////////////////////////////////

    class ResultGenerator : boost::noncopyable {
      public:
        typedef void (*generate_fptr)(ResultGenerator const* generator, basics::StringBuffer&, basics::VariantObject*);

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief list of generators
        ////////////////////////////////////////////////////////////////////////////////

        enum ResultGeneratorType {
          RESULT_GENERATOR_HTML,
          RESULT_GENERATOR_JSON,
          RESULT_GENERATOR_JSONX,
          RESULT_GENERATOR_PHP,
          RESULT_GENERATOR_XML
        };

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief registers an output for a variant
        ////////////////////////////////////////////////////////////////////////////////

        static void registerObject (basics::VariantObject::ObjectType, ResultGeneratorType, generate_fptr);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief registers an output for a variant
        ////////////////////////////////////////////////////////////////////////////////

        static void registerObject (basics::VariantObject::ObjectType, ResultGeneratorType, basics::VariantObject::ObjectType);

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief destructs a result generator
        ////////////////////////////////////////////////////////////////////////////////

        virtual ~ResultGenerator () {
        }

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief generator type
        ////////////////////////////////////////////////////////////////////////////////

        virtual ResultGeneratorType type () const = 0;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief content type
        ////////////////////////////////////////////////////////////////////////////////

        virtual string contentType () const {
          return "text/plain; charset=utf-8";
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief generate object
        ////////////////////////////////////////////////////////////////////////////////

        void generate (basics::StringBuffer&, basics::VariantObject*) const;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief generate a variant object
        ////////////////////////////////////////////////////////////////////////////////

        virtual void generateVariant (basics::StringBuffer&, basics::VariantObject*) const;

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief generate a string
        ////////////////////////////////////////////////////////////////////////////////

        virtual void generateAtom (basics::StringBuffer&, char const*) const = 0;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief generate a string
        ////////////////////////////////////////////////////////////////////////////////

        virtual void generateAtom (basics::StringBuffer&, char const*, size_t, bool quote) const = 0;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief generate a boolean
        ////////////////////////////////////////////////////////////////////////////////

        virtual void generateAtom (basics::StringBuffer&, bool) const = 0;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief generate a double
        ////////////////////////////////////////////////////////////////////////////////

        virtual void generateAtom (basics::StringBuffer&, double) const = 0;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief generate a float
        ////////////////////////////////////////////////////////////////////////////////

        virtual void generateAtom (basics::StringBuffer&, float) const = 0;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief generate a int16
        ////////////////////////////////////////////////////////////////////////////////

        virtual void generateAtom (basics::StringBuffer&, int16_t) const = 0;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief generate a int32
        ////////////////////////////////////////////////////////////////////////////////

        virtual void generateAtom (basics::StringBuffer&, int32_t) const = 0;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief generate a int64
        ////////////////////////////////////////////////////////////////////////////////

        virtual void generateAtom (basics::StringBuffer&, int64_t) const = 0;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief generate a string
        ////////////////////////////////////////////////////////////////////////////////

        virtual void generateAtom (basics::StringBuffer&, string const&) const = 0;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief generate a uint16
        ////////////////////////////////////////////////////////////////////////////////

        virtual void generateAtom (basics::StringBuffer&, uint16_t) const = 0;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief generate a uint32
        ////////////////////////////////////////////////////////////////////////////////

        virtual void generateAtom (basics::StringBuffer&, uint32_t) const = 0;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief generate a uint64
        ////////////////////////////////////////////////////////////////////////////////

        virtual void generateAtom (basics::StringBuffer&, uint64_t) const = 0;

      protected:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief generate result begin
        ////////////////////////////////////////////////////////////////////////////////

        virtual void generateResultBegin (basics::StringBuffer&, basics::VariantObject*) const {}

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief generate result end
        ////////////////////////////////////////////////////////////////////////////////

        virtual void generateResultEnd (basics::StringBuffer&, basics::VariantObject*) const {}
    };
  }
}

#endif
