////////////////////////////////////////////////////////////////////////////////
/// @brief statistics overview request handler
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Achim Brandt
/// @author Copyright 2010, triagens GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "RestStatisticsOverviewHandler.h"

#include <Basics/VariantVector.h>

#include "GeneralServer/GeneralFigures.h"

using namespace triagens::basics;
using namespace triagens::rest;

namespace triagens {
  namespace simple {

    // -----------------------------------------------------------------------------
    // Handler methods
    // -----------------------------------------------------------------------------

    HttpHandler::status_e RestStatisticsOverviewHandler::execute () {
      StatsValues stats;
      ResponseType result = model->statsValue(stats);

      if (result == OK) {
        LimitValues limit = model->getLimitValues();
        VariantArray* result = new VariantArray();

        VariantArray* memory = new VariantArray();
        result->add("memory", memory);

        memory->add("numberKeys", new VariantInt32(stats.numKeyValue));
        memory->add("limitKeys", new VariantInt32(limit.maxNumKeys));

        memory->add("totalKeySize", new VariantDouble(stats.totalKeySize));
        memory->add("limitKeySize", new VariantDouble(limit.maxKeySize));
        memory->add("limitTotalKeySize", new VariantDouble(limit.maxTotalKeySize));

        memory->add("totalDataSize", new VariantDouble(stats.totalDataSize));
        memory->add("limitDataSize", new VariantDouble(limit.maxValueSize));
        memory->add("limitTotalDataSize", new VariantDouble(limit.maxTotalValueSize));

        memory->add("limitTotalSize", new VariantDouble(limit.maxTotalSize));

        VariantArray* runtime = new VariantArray();
        result->add("runtime", runtime);

        VariantArray* connection = new VariantArray();
        result->add("connection", connection);

        for (size_t i = 0;  i < 4;  ++i) {
          vector<SimpleVocStatistics> v;
          vector<time_t> tv;

          vector<rest::GeneralFigures::GeneralServerStatistics> w;
          vector<time_t> tw;

          string name = "unknown";

          switch (i) {
            case 0:
              name = "second";
              v = FiguresSecond.values(tv);
              w = rest::GeneralFigures::FiguresSecond.values(tw);
              break;

            case 1:
              name = "minute";
              v = FiguresMinute.values(tv);
              w = rest::GeneralFigures::FiguresMinute.values(tw);
              break;

            case 2:
              name = "hour";
              v = FiguresHour.values(tv);
              w = rest::GeneralFigures::FiguresHour.values(tw);
              break;

            case 3:
              name = "day";
              v = FiguresDay.values(tv);
              w = rest::GeneralFigures::FiguresDay.values(tw);
              break;
          }

          if (v.empty()) {
            continue;
          }

          // -----------------------------------------------------------------------------
          // runtime
          // -----------------------------------------------------------------------------

          VariantArray* values = new VariantArray();
          runtime->add(name, values);

          VariantVector* start = new VariantVector();
          values->add("start", start);

          size_t nstarts = 0;

          for (vector<time_t>::const_iterator k = tv.begin();  k != tv.end();  ++k, ++nstarts) {
            start->add(new VariantUInt32(*k));
          }

          if (0 < nstarts) {
            --nstarts;
          }

          size_t const n = size_t(STAT_END);
          size_t const m = size_t(GROUP_END);

          VariantArray* stats[n + m];
          VariantVector* vecs[n + m][2];

          static int const COUNT = 0;
          static int const MEAN = 1;

          for (size_t k = 0;  k < n;  ++k) {
            stats[k] = new VariantArray();
            values->add(StatisticsName(stat_e(k)), stats[k]);

            vecs[k][COUNT] = new VariantVector();
            stats[k]->add("count", vecs[k][COUNT]);

            vecs[k][MEAN] = new VariantVector();
            stats[k]->add("mean", vecs[k][MEAN]);
          }

          for (size_t k = n;  k < n + m;  ++k) {
            stats[k] = new VariantArray();
            values->add(GroupName(group_e(k - n)), stats[k]);

            vecs[k][COUNT] = new VariantVector();
            stats[k]->add("count", vecs[k][COUNT]);

            vecs[k][MEAN] = new VariantVector();
            stats[k]->add("mean", vecs[k][MEAN]);
          }

          size_t pos = 0;

          for (vector<SimpleVocStatistics>::iterator j = v.begin();  j != v.end() && pos < nstarts;  ++j, ++pos) {
            uint32_t groupCount[m] = {0, 0, 0, 0};
            double groupSum[m] = {0.0, 0.0, 0.0, 0.0};

            for (size_t k = 0;  k < n;  ++k) {
              uint32_t count = j->runtimes.count[k];
              double sum = j->runtimes.sum[k];

              vecs[k][COUNT]->add(new VariantUInt32(count));
              vecs[k][MEAN]->add(new VariantDouble(count == 0 ? 0.0 : (sum / count)));

              group_e group = StatisticsGroup(stat_e(k));

              if (group != GROUP_ALL) {
                groupCount[size_t(group)] += count;
                groupSum[size_t(group)] += sum;
              }

              groupCount[size_t(GROUP_ALL)] += count;
              groupSum[size_t(GROUP_ALL)] += sum;

            }

            for (size_t p = 0;  p < size_t(GROUP_END);  ++p) {
              vecs[n + p][COUNT]->add(new VariantUInt32(groupCount[p]));
              vecs[n + p][MEAN]->add(new VariantDouble(groupCount[p] == 0 ? 0.0 : (groupSum[p] / groupCount[p])));
            }
          }

          // -----------------------------------------------------------------------------
          // connections
          // -----------------------------------------------------------------------------

          values = new VariantArray();
          connection->add(name, values);

          start = new VariantVector();
          values->add("start", start);

          nstarts = 0;

          for (vector<time_t>::const_iterator k = tw.begin();  k != tw.end();  ++k, ++nstarts) {
            start->add(new VariantUInt32(*k));
          }

          if (0 < nstarts) {
            --nstarts;
          }

          // http connections
          VariantArray* httpConnections = new VariantArray();
          values->add("httpConnections", httpConnections);

          VariantVector* http = new VariantVector();
          httpConnections->add("count", http);

          // line connections
          VariantArray* lineConnections = new VariantArray();
          values->add("lineConnections", lineConnections);

          VariantVector* line = new VariantVector();
          lineConnections->add("count", line);

          // sum
          VariantArray* totalConnections = new VariantArray();
          values->add("connections", totalConnections);

          VariantVector* total = new VariantVector();
          totalConnections->add("count", total);

          pos = 0;

          for (vector<rest::GeneralFigures::GeneralServerStatistics>::iterator j = w.begin();  j != w.end() && pos < nstarts;  ++j, ++pos) {
            http->add(new VariantInt32(j->http.count));
            line->add(new VariantInt32(j->line.count));
            total->add(new VariantInt32(j->http.count + j->line.count));
          }
        }

        generateResult(result, htmlContent);
      }
      else {
        generateError("cannot extract statistics");
      }

      return HANDLER_DONE;
    }
  }
}
