////////////////////////////////////////////////////////////////////////////////
/// @brief prefix request handler
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Achim Brandt
/// @author Copyright 2010, triagens GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "RestPrefixHandler.h"

#include <Basics/StringBuffer.h>
#include <Hpdf/HpdfFilterTree.h>

using namespace triagens::basics;
using namespace triagens::rest;
using namespace triagens::hpdf;

namespace triagens {
  namespace simple {

    // -----------------------------------------------------------------------------
    // Handler methods
    // -----------------------------------------------------------------------------

    HttpHandler::status_e RestPrefixHandler::execute() {

      // check for illegal request
      if (request->requestType() == rest::HttpRequest::HTTP_REQUEST_ILLEGAL) {
        generateError("method not supported");
        return HANDLER_DONE;
      }

      computePrefix();

      if (! allowEmptyPrefix && prefix.empty()) {
        generateError("empty prefix");
        return HANDLER_DONE;
      }

      bool found;
      string filterStr = request->value("filter", found);

      HpdfFilterTree* filter = 0;

      if (found) {
        filter = model->dynamicFilterTree(filterStr);

        if (filter == 0) {
          generateError("cannot parse filter definition");
          return HANDLER_DONE;
        }
      }

      DynamicSort::DynamicSortDescription sorting;

      string sortingStr = request->value("sort", found);

      if (found) {
        sorting = model->dynamicSortDescription(sortingStr);

        if (sortingStr.empty()) {
          generateError("cannot parse sort definition");
          return HANDLER_DONE;
        }
      }

      switch (request->requestType()) {
        case rest::HttpRequest::HTTP_REQUEST_GET:
          return executeGet(filter, sorting);

        default:
          generateError("method not supported");
          break;
      }

      return HANDLER_DONE;
    }

    // -----------------------------------------------------------------------------
    // protected methods
    // -----------------------------------------------------------------------------

    HttpHandler::status_e RestPrefixHandler::executeGet (HpdfFilterTree* tree, DynamicSort::DynamicSortDescription const& sorting) {
      vector<Value> values;

      Value value;
      value.identifier = prefix;

      ResponseType result = model->prefixGetValue(value, tree, sorting, values);

      if (tree != 0) {
        delete tree;
      }

      if (result == OK) {
        response = new rest::HttpResponse(rest::HttpResponse::OK);
        response->body().appendChar('[');

        string sep = "\"";

        for (vector<Value>::iterator i = values.begin();  i != values.end();  ++i) {
          Value& v = *i;

          response->body().appendText(sep);
          response->body().appendText(StringUtils::escapeUnicode(v.identifier));
          response->body().appendText("\"");

          sep = ",\"";
        }

        response->body().appendChar(']');
        response->setContentType("application/json");
      }
      else {
        generateError("prefix not found");
      }

      return HANDLER_DONE;
    }

    // -----------------------------------------------------------------------------
    // private methods
    // -----------------------------------------------------------------------------

    void RestPrefixHandler::computePrefix () {
      basics::StringBuffer sb;
      sb.initialise();

      bool first = true;

      for (vector<string>::const_iterator i = request->suffix().begin();
           i != request->suffix().end();
           ++i) {

        if (first) {
          first = false;
        }
        else {
          sb.appendChar('/');
        }
        sb.appendText(*i);
      }

      if (sb.length() == 0) {
        return;
      }

      prefix = string(sb.c_str(), sb.length());

      sb.free();
    }
  }
}


