////////////////////////////////////////////////////////////////////////////////
/// @brief tasks used to handle timer events
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Achim Brandt
/// @author Copyright 2008-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "TimerTask.h"

#include <ev.h>

#include <Basics/Logger.h>
#include <Rest/Scheduler.h>

#include "Scheduler/EventLoop.h"

using namespace triagens::basics;
using namespace triagens::rest;

namespace triagens {
  namespace rest {

    // -----------------------------------------------------------------------------
    // classes
    // -----------------------------------------------------------------------------

    struct TimerTask::watcher_t {
      ev_timer timer;
      TimerTask* task;
    };

    // -----------------------------------------------------------------------------
    // static protected methods
    // -----------------------------------------------------------------------------

    void TimerTask::callback (event_loop_t*, watcher_t* w, int revents) {
      TimerTask* task = w->task;

      if (task != 0 && task->isActive()) {
        task->handleEvent(reinterpret_cast<void*>(w), revents);
      }
    }

    // -----------------------------------------------------------------------------
    // constructors and destructors
    // -----------------------------------------------------------------------------

    TimerTask::TimerTask (double seconds)
      : Task("TimerTask"),
        seconds(seconds) {
      watcher = new watcher_t;
      watcher->task = this;
    }



    TimerTask::~TimerTask () {
      delete watcher;
    }

    // -----------------------------------------------------------------------------
    // Task methods
    // -----------------------------------------------------------------------------

    void TimerTask::setup (Scheduler* scheduler, event_loop_t* loop) {
      this->scheduler = scheduler;
      this->loop = loop;

      ev_timer* et = (ev_timer*) watcher;
      ev_timer_init(et,
                    (void (*)(struct ev_loop*, ev_timer*, int)) callback,
                    seconds,
                    0.0);

      if (0.0 < seconds) {
        ev_timer_start((struct ev_loop*) loop, et);

        LOGGER_TRACE << "armed TimerTask with " << seconds << " seconds";
      }
    }



    void TimerTask::cleanup () {
      ev_timer_stop((struct ev_loop*) loop, (ev_timer*) watcher);
    }



    bool TimerTask::handleEvent (void* token, int revents) {
      bool result = true;

      if (token == (void*) watcher) {
        if (revents & EV_TIMER) {
          ev_timer_stop((struct ev_loop*) loop, (ev_timer*) watcher);

          result = handleTimeout();
        }
      }

      return result;
    }
  }
}
