////////////////////////////////////////////////////////////////////////////////
/// @brief base class for input-output tasks from sockets
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Achim Brandt
/// @author Copyright 2009-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "SocketTask.h"

#include <errno.h>

#include <Basics/Logger.h>
#include <Basics/MutexLocker.h>
#include <Basics/StringBuffer.h>
#include <Rest/Scheduler.h>

#include "Scheduler/EventLoop.h"

using namespace triagens::basics;

namespace triagens {
  namespace rest {

    static int _numWrites = 0;
    static Mutex lockNumWrites;
    // -----------------------------------------------------------------------------
    // classes
    // -----------------------------------------------------------------------------

    struct SocketTask::watcher_t {
      ev_async async;
      SocketTask* task;
    };

    // -----------------------------------------------------------------------------
    // static protected methods
    // -----------------------------------------------------------------------------

    void SocketTask::callback (event_loop_t*, watcher_t* w, int revents) {
      SocketTask* task = w->task;

      if (task != 0 && task->isActive()) {
        task->handleEvent(reinterpret_cast<void*>(w), revents);
      }
    }

    // -----------------------------------------------------------------------------
    // constructors and destructors
    // -----------------------------------------------------------------------------

    SocketTask::SocketTask (socket_t fd)
      : Task("SocketTask"),
        commSocket(fd),
        writeBuffer(0),
        ownBuffer(true),
        writeLength(0) {
      watcher = new watcher_t;
      watcher->task = this;

      readBuffer = new StringBuffer();
      tmpReadBuffer = new char[READ_BLOCK_SIZE];
    }



    SocketTask::~SocketTask () {
      if (commSocket != -1) {
        close(commSocket);
      }

      if (writeBuffer != 0) {
        if (ownBuffer) {
          writeBuffer->free();
          delete writeBuffer;
        }
      }

      delete watcher;

      readBuffer->free();
      delete readBuffer;

      delete[] tmpReadBuffer;
    }

    // -----------------------------------------------------------------------------
    // Task methods
    // -----------------------------------------------------------------------------

    void SocketTask::setup (Scheduler* scheduler, event_loop_t* loop) {
      this->scheduler = scheduler;
      this->loop = loop;

      ev_async* aw = (ev_async*) watcher;
      ev_async_init(aw, (void (*)(struct ev_loop*, ev_async*, int)) callback);
      ev_async_start((struct ev_loop*) loop, aw);

      ev_io* rw = (ev_io*) readWatcher;
      ev_io_init(rw,
                 (void (*)(struct ev_loop*, ev_io*, int)) IoTask::callback,
                 commSocket,
                 EV_READ);
      ev_io_start((struct ev_loop*) loop, rw);

      ev_io* ww = (ev_io*) writeWatcher;
      ev_io_init(ww,
                 (void (*)(struct ev_loop*, ev_io*, int)) IoTask::callback,
                 commSocket,
                 EV_WRITE);
      ev_io_start((struct ev_loop*) loop, ww);

      IoTask::setup(scheduler, loop);
    }



    void SocketTask::cleanup () {
      ev_async_stop((struct ev_loop*) loop, (ev_async*) watcher);
      ev_io_stop((struct ev_loop*) loop, (ev_io*) readWatcher);
      ev_io_stop((struct ev_loop*) loop, (ev_io*) writeWatcher);
    }



    bool SocketTask::handleEvent (void* token, int revents) {
      bool result = true;
      bool closed = false;

      if (token == (void*) readWatcher) {
        if (revents & EV_READ) {
          result = handleRead(closed);
        }
      }

      if (result && ! closed && token == (void*) writeWatcher) {
        bool noWrite = false;

        {
          MUTEX_LOCKER(guard, writeBufferLock);
          noWrite = (writeBuffer == 0);
        }

        if (revents & EV_WRITE) {
          result = handleWrite(closed, noWrite);
        }
      }

      if (result) {
        MUTEX_LOCKER(guard, writeBufferLock);
        ev_io* w = (ev_io*) writeWatcher;

        if (writeBuffer == 0) {
          ev_io_stop((struct ev_loop*) loop, w);
        }
        else if (! ev_is_active(w)) {
          ev_io_start((struct ev_loop*) loop, w);
        }
      }

      return result;
    }

    // -----------------------------------------------------------------------------
    // protected methods
    // -----------------------------------------------------------------------------

    bool SocketTask::handleWrite (bool& closed, bool noWrite) {
      closed = false;

      if (noWrite) {
        return true;
      }

      bool callCompletedWriteBuffer = false;

      {
        MUTEX_LOCKER(guard, writeBufferLock);

        size_t len = writeBuffer->length() - writeLength;
        /*
        {
          //MUTEX_LOCKER(guard2, lockNumWrites);
          ++_numWrites;
          if (_numWrites % 1000 == 0) {
            cout << __FILE__ << ":" << __LINE__ << ":" << "numwrites = " << _numWrites << ":" << writeBuffer->c_str() << ":" << pthread_self() << endl;
          }
        }
        */
        int nr = 0;

        if (0 < len) {
          nr = write(commSocket, writeBuffer->begin() + writeLength, (int) len);
          //cout << __FILE__ << ":" << __LINE__ << ":" << "numwrites = " << _numWrites << ":" << nr << ":" << len << endl;

          if (nr < 0) {
            if (errno == EINTR) {
              return handleWrite(closed, noWrite);
            }
            else if (errno != EWOULDBLOCK) {
              LOGGER_DEBUG << "write failed with " << errno << " (" << strerror(errno) << ")";

              return false;
            }
            else {
              nr = 0;
            }
          }

          len -= nr;
        }

        if (len == 0) {
          if (ownBuffer) {
            writeBuffer->free();
            delete writeBuffer;
          }

          writeBuffer = 0;
          callCompletedWriteBuffer = true;
        }
        else {
          writeLength += nr;
        }
      }

      // we have to release the lock, before calling completedWriteBuffer
      if (callCompletedWriteBuffer) {
        completedWriteBuffer(closed);

        if (closed) {
          return false;
        }
      }

      // we might have a new write buffer or none at all
      ev_async_send((struct ev_loop*) loop, (ev_async*) watcher);

      return true;
    }



    bool SocketTask::hasWriteBuffer () const {
      MUTEX_LOCKER(guard, writeBufferLock);

      return writeBuffer != 0;
    }



    void SocketTask::setWriteBuffer (StringBuffer* buffer, bool ownBuffer) {
      bool callCompletedWriteBuffer = false;

      {
        MUTEX_LOCKER(guard, writeBufferLock);

        writeLength = 0;

        if (buffer->empty()) {
          if (ownBuffer) {
            buffer->free();
            delete buffer;
          }

          writeBuffer = 0;
          callCompletedWriteBuffer = true;
        }
        else {
          if (writeBuffer != 0) {
            if (this->ownBuffer) {
              writeBuffer->free();
              delete writeBuffer;
            }
          }

          writeBuffer = buffer;
          this->ownBuffer = ownBuffer;
        }
      }

      // we have to release the lock, before calling completedWriteBuffer
      if (callCompletedWriteBuffer) {
        bool closed;
        completedWriteBuffer(closed);
      }

      // we might have a new write buffer or none at all
      ev_async_send((struct ev_loop*) loop, (ev_async*) watcher);
    }



    bool SocketTask::fillReadBuffer (bool& closed) {
      closed = false;

      int nr = read(commSocket, tmpReadBuffer, READ_BLOCK_SIZE);

      if (nr > 0) {
        readBuffer->appendText(tmpReadBuffer, nr);

        return true;
      }
      else if (nr == 0) {
        closed = true;

        LOGGER_TRACE << "read return 0 with " << errno << " (" << strerror(errno) << ")";

        return false;
      }
      else {
        if (errno == EINTR) {
          return fillReadBuffer(closed);
        }
        else if (errno != EWOULDBLOCK) {
          LOGGER_TRACE << "read failed with " << errno << " (" << strerror(errno) << ")";

          return false;
        }
        else {
          LOGGER_TRACE << "read would block with " << errno << " (" << strerror(errno) << ")";

          return true;
        }
      }
    }
  }
}
