////////////////////////////////////////////////////////////////////////////////
/// @brief tasks used to handle signals
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Achim Brandt
/// @author Copyright 2008-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "SignalTask.h"

#include <ev.h>

#include <Basics/Logger.h>
#include <Basics/MutexLocker.h>
#include <Rest/Scheduler.h>

using namespace triagens::basics;

namespace triagens {
  namespace rest {

    // -----------------------------------------------------------------------------
    // classes
    // -----------------------------------------------------------------------------

    struct SignalTask::watcher_t {
      ev_signal signal;
      SignalTask* task;
    };

    // -----------------------------------------------------------------------------
    // static protected methods
    // -----------------------------------------------------------------------------

    void SignalTask::callback (event_loop_t*, watcher_t* w, int revents) {
      SignalTask* task = w->task;

      if (task != 0 && task->isActive()) {
        task->handleEvent(reinterpret_cast<void*>(w), revents);
      }
    }

    // -----------------------------------------------------------------------------
    // constructors and destructors
    // -----------------------------------------------------------------------------

    SignalTask::SignalTask ()
      : Task("SignalTask") {
      watcher = new watcher_t[MAX_SIGNALS];

      for (size_t i = 0;  i < MAX_SIGNALS;  ++i) {
        watcher[i].task = this;
      }
    }



    SignalTask::SignalTask (int signal)
      : Task("SignalTask") {
      signals.insert(signal);

      for (size_t i = 0;  i < MAX_SIGNALS;  ++i) {
        watcher[i].task = this;
      }

      delete[] watcher;
    }



    SignalTask::~SignalTask () {
    }

    // -----------------------------------------------------------------------------
    // public methods
    // -----------------------------------------------------------------------------

    bool SignalTask::addSignal (int signal) {
      MUTEX_LOCKER(guard, changeLock);

      if (signals.size() >= MAX_SIGNALS) {
        LOGGER_ERROR << "maximal number of signals reached";
        return false;
      }
      else {
        if (scheduler != 0) {
          scheduler->unregisterTask(this);
        }

        signals.insert(signal);

        if (scheduler != 0) {
          scheduler->registerTask(this);
        }

        return true;
      }
    }

    // -----------------------------------------------------------------------------
    // Task methods
    // -----------------------------------------------------------------------------

    void SignalTask::setup (Scheduler* scheduler, event_loop_t* loop) {
      this->scheduler = scheduler;
      this->loop = loop;

      size_t pos = 0;

      for (set<int>::iterator i = signals.begin();  i != signals.end() && pos < MAX_SIGNALS;  ++i, ++pos) {
        ev_signal* w = (ev_signal*) &watcher[pos];
        ev_signal_init(w,
                       (void (*)(struct ev_loop*, ev_signal*, int)) callback,
                       *i);
        ev_signal_start((struct ev_loop*) loop, w);
      }
    }



    void SignalTask::cleanup () {
      size_t pos = 0;

      for (set<int>::iterator i = signals.begin();  i != signals.end() && pos < MAX_SIGNALS;  ++i, ++pos) {
        ev_signal_stop((struct ev_loop*) loop, (ev_signal*) &watcher[pos]);
      }
    }



    bool SignalTask::handleEvent (void* token, int revents) {
      bool result = true;

      if (revents & EV_SIGNAL) {
        size_t pos = 0;

        for (set<int>::iterator i = signals.begin();  i != signals.end();  ++i, ++pos) {
          if (token == &watcher[pos].signal) {
            result = handleSignal();
            break;
          }
        }
      }

      return result;
    }



    bool SignalTask::needsMainEventLoop () const {
      return true;
    }
  }
}
