////////////////////////////////////////////////////////////////////////////////
/// @brief input-output scheduler
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Achim Brandt
/// @author Copyright 2008-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#ifndef TRIAGENS_REST_SCHEDULER_H
#define TRIAGENS_REST_SCHEDULER_H 1

#include <Basics/Common.h>

////////////////////////////////////////////////////////////////////////////////
/// @defgroup Scheduler I/O Scheduler
////////////////////////////////////////////////////////////////////////////////

namespace triagens {
  namespace basics {
    class ConditionVariable;
  }

  namespace rest {
    class Task;

    ////////////////////////////////////////////////////////////////////////////////
    /// @ingroup Scheduler
    /// @brief interface of a input-output scheduler
    ////////////////////////////////////////////////////////////////////////////////

    class Scheduler : boost::noncopyable {
      public:

        /////////////////////////////////////////////////////////////////////////
        /// @brief creates a single-threaded scheduler
        /////////////////////////////////////////////////////////////////////////

        static Scheduler* create ();

      public:

        /////////////////////////////////////////////////////////////////////////
        /// @brief destructor
        /////////////////////////////////////////////////////////////////////////

        virtual ~Scheduler () {
        }

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief checks if scheduler is shuting down
        ////////////////////////////////////////////////////////////////////////////////

        virtual bool isShutdownInProgress () = 0;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief checks if scheduler is still running
        ////////////////////////////////////////////////////////////////////////////////

        virtual bool isRunning () = 0;

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief registers a new task
        ////////////////////////////////////////////////////////////////////////////////

        virtual void registerTask (Task*) = 0;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief unregisters a task
        ///
        /// Note that this method is called by the task itself when cleanupTask is
        /// executed. If a Task failed in setupTask, it must not call unregisterTask.
        ////////////////////////////////////////////////////////////////////////////////

        virtual void unregisterTask (Task*) = 0;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief destroys task
        ///
        /// Even if a Task failed in setupTask, it can still call destroyTask.
        ////////////////////////////////////////////////////////////////////////////////

        virtual void destroyTask (Task*) = 0;

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief starts scheduler, keeps running
        ////////////////////////////////////////////////////////////////////////////////

        virtual bool start (basics::ConditionVariable*) = 0;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief starts shutdown sequence
        ////////////////////////////////////////////////////////////////////////////////

        virtual void beginShutdown () = 0;

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief called to display current status
        ////////////////////////////////////////////////////////////////////////////////

        virtual void reportStatus () = 0;
    };
  }
}

#endif
