////////////////////////////////////////////////////////////////////////////////
/// @brief tasks used to handle periodic events
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Achim Brandt
/// @author Copyright 2008-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "PeriodicTask.h"

#include <ev.h>

#include <Rest/Scheduler.h>

namespace triagens {
  namespace rest {

    // -----------------------------------------------------------------------------
    // classes
    // -----------------------------------------------------------------------------

    struct PeriodicTask::watcher_t {
      ev_periodic periodic;
      PeriodicTask* task;
    };

    // -----------------------------------------------------------------------------
    // static protected methods
    // -----------------------------------------------------------------------------

    void PeriodicTask::callback (event_loop_t*, watcher_t* w, int revents) {
      PeriodicTask* task = w->task;

      if (task != 0 && task->isActive()) {
        task->handleEvent(reinterpret_cast<void*>(w), revents);
      }
    }

    // -----------------------------------------------------------------------------
    // constructors and destructors
    // -----------------------------------------------------------------------------

    PeriodicTask::PeriodicTask (double offset, double intervall)
      : Task("PeriodicTask"), offset(offset), intervall(intervall) {
      watcher = new watcher_t;
      watcher->task = this;
    }



    PeriodicTask::~PeriodicTask () {
      delete watcher;
    }

    // -----------------------------------------------------------------------------
    // Task methods
    // -----------------------------------------------------------------------------

    void PeriodicTask::resetTimer (double offset, double intervall) {
      ev_periodic* w = (ev_periodic*) watcher;
      ev_periodic_set(w, offset, intervall, 0);
      ev_periodic_again((struct ev_loop*) loop, w);
    }

    // -----------------------------------------------------------------------------
    // Task methods
    // -----------------------------------------------------------------------------

    void PeriodicTask::setup (Scheduler* scheduler, event_loop_t* loop) {
      this->scheduler = scheduler;
      this->loop = loop;

      ev_periodic* w = (ev_periodic*) watcher;
      ev_periodic_init(w,
                       (void (*)(struct ev_loop*, ev_periodic*, int)) callback,
                       offset,
                       intervall,
                       0);
      ev_periodic_start((struct ev_loop*) loop, w);
    }



    void PeriodicTask::cleanup () {
      ev_periodic_stop((struct ev_loop*) loop, (ev_periodic*) watcher);
    }



    bool PeriodicTask::handleEvent (void* token, int revents) {
      bool result = true;

      if (token == (void*) watcher) {
        if (revents & EV_PERIODIC) {
          result = handlePeriod();
        }
      }

      return result;
    }
  }
}
