////////////////////////////////////////////////////////////////////////////////
/// @brief tasks used to establish connections
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Achim Brandt
/// @author Copyright 2009-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#ifndef TRIAGENS_REST_LISTEN_TASK_H
#define TRIAGENS_REST_LISTEN_TASK_H 1

#include <Rest/IoTask.h>

#include <Basics/Mutex.h>
#include <Rest/ConnectionInfo.h>

namespace triagens {
  namespace rest {

    ////////////////////////////////////////////////////////////////////////////////
    /// @ingroup Scheduler
    /// @brief task used to establish connections
    ////////////////////////////////////////////////////////////////////////////////

    class ListenTask : public IoTask {
      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief maximal number of failed connects
        ////////////////////////////////////////////////////////////////////////////////

        static size_t const MAX_ACCEPT_ERRORS = 1000;

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief listen to given address and port
        ////////////////////////////////////////////////////////////////////////////////

        ListenTask (string const& address, int port, bool reuseAddress);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief listen to given port
        ////////////////////////////////////////////////////////////////////////////////

        ListenTask (int port, bool reuseAddress);

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief checks if listen socket is bound
        ////////////////////////////////////////////////////////////////////////////////

        bool isBound () const;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief try to rebind to port
        ///
        /// Note that this method can only be called before the task has been
        /// registered.
        ////////////////////////////////////////////////////////////////////////////////

        bool rebind ();

      protected:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief called by the task to indicate connection success
        ////////////////////////////////////////////////////////////////////////////////

        virtual bool handleConnected (socket_t fd, ConnectionInfo const& info) = 0;

      protected:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief destructs a listen task
        ///
        /// This method will close the underlying socket.
        ////////////////////////////////////////////////////////////////////////////////

        ~ListenTask ();

      protected:

        ////////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        ///
        /// Note that registerTask must only be called when the socket is bound.
        ////////////////////////////////////////////////////////////////////////////////

        void setup (Scheduler*, event_loop_t*);

        ////////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        ////////////////////////////////////////////////////////////////////////////////

        void cleanup ();

        ////////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        ////////////////////////////////////////////////////////////////////////////////

        bool handleEvent (void* token, int event);

      private:
        bool bindSocket ();

      private:
        bool reuseAddress;
        string const address;
        int const port;
        socket_t listenSocket;
        bool bound;

        size_t acceptFailures;

        mutable basics::Mutex changeLock;
    };
  }
}

#endif
