////////////////////////////////////////////////////////////////////////////////
/// @brief handler factory
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Copyright 2009-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "HttpHandlerFactory.h"

#include <Basics/Logger.h>
#include <Rest/HttpRequest.h>

using namespace triagens::basics;

namespace triagens {
  namespace rest {

    // -----------------------------------------------------------------------------
    // constructors and destructors
    // -----------------------------------------------------------------------------

    HttpHandlerFactory::HttpHandlerFactory ()
      : _notFound(0) {
    }



    HttpHandlerFactory::HttpHandlerFactory (HttpHandlerFactory const& that)
      : _constructors(that._constructors),
        _datas(that._datas),
        _prefixes(that._prefixes),
        _notFound(that._notFound) {
    }



    HttpHandlerFactory& HttpHandlerFactory::operator= (HttpHandlerFactory const& that) {
      if (this != &that) {
        _constructors = that._constructors;
        _datas = that._datas;
        _prefixes = that._prefixes;
        _notFound = that._notFound;
      }

      return *this;
    }

    // -----------------------------------------------------------------------------
    // public methods
    // -----------------------------------------------------------------------------

    void HttpHandlerFactory::addHandler (string const& path, create_fptr func, void* data) {
      _constructors[path] = func;
      _datas[path] = data;
    }



    void HttpHandlerFactory::addPrefixHandler (string const& path, create_fptr func, void* data) {
      _constructors[path] = func;
      _datas[path] = data;
      _prefixes.push_back(path);
    }



    void HttpHandlerFactory::addNotFoundHandler (create_fptr func) {
      _notFound = func;
    }



    pair<size_t, size_t> HttpHandlerFactory::sizeRestrictions () {
      static size_t m = (size_t) -1;

      return make_pair(m, m);
    }



    HttpRequest* HttpHandlerFactory::createRequest (char const* ptr, size_t length) const {
      return new HttpRequest(ptr, length);
    }



    HttpHandler* HttpHandlerFactory::createHandler (HttpRequest* request) const {
      map<string, create_fptr> const& ii = _constructors;
      string const& path = request->requestPath();
      map<string, create_fptr>::const_iterator i = ii.find(path);
      void* data = 0;

      // no direct match, check prefix matches
      if (i == ii.end()) {
        LOGGER_TRACE << "no direct handler found, trying prefixes";

        // find longest match
        string prefix;
        vector<string> const& jj = _prefixes;

        for (vector<string>::const_iterator j = jj.begin();  j != jj.end();  ++j) {
          string const& p = *j;

          if (path.compare(0, p.size(), p) == 0) {
            if (p.size() < path.size() && path[p.size()] == '/') {
              if (prefix.size() < p.size()) {
                prefix = p;
              }
            }
          }
        }

        if (! prefix.empty()) {
          LOGGER_TRACE << "found prefix match '" << prefix << "'";

          size_t l = prefix.size() + 1;
          size_t n = path.find_first_of('/', l);

          while (n != string::npos) {
            request->addSuffix(path.substr(l, n - l));
            l = n + 1;
            n = path.find_first_of('/', l);
          }

          if (l < path.size()) {
            request->addSuffix(path.substr(l));
          }

          request->setRequestPath(prefix);
          i = ii.find(prefix);
        }
      }

      // no match
      if (i == ii.end()) {
        if (_notFound != 0) {
          return _notFound(request, data);
        }
        else {
          LOGGER_TRACE << "no not-found handler, giving up";
          return 0;
        }
      }

      // look up data
      map<string, void*> const& jj = _datas;
      map<string, void*>::const_iterator j = jj.find(path);

      if (j != jj.end()) {
        data = j->second;
      }

      LOGGER_TRACE << "found handler for path '" << path << "'";

      return i->second(request, data);
    }
  }
}
