////////////////////////////////////////////////////////////////////////////////
/// @brief get request handler
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Oreste Costa-Panaia
/// @author Copyright 2010, triagens GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#ifndef SIMPLEVOC_POSTFIXHANDLER_POSTFIX_GET_HANDLER_H
#define SIMPLEVOC_POSTFIXHANDLER_POSTFIX_GET_HANDLER_H 1

#include "PostfixHandler/PostfixBaseHandler.h"

#include <Rest/ListenTask.h>

namespace triagens {
  namespace simple {

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief get request handler
    ////////////////////////////////////////////////////////////////////////////////

    class PostfixGetHandler : public PostfixBaseHandler {

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief constructor
        ////////////////////////////////////////////////////////////////////////////////

        PostfixGetHandler (rest::LineRequest* request, SimpleModel* model, vector<Transformation> const& transformations)
          : PostfixBaseHandler(request, model, STAT_POSTFIX_GET), transformations(transformations) {
        }

      protected:

        ////////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        ////////////////////////////////////////////////////////////////////////////////

        void work (PostfixRequest* request, PostfixResponse* response) {
          char const* key = request->getKey();

          if (key == 0) {
            response->body().appendText("500 no key\n");
          }
          else {
            Value value;
            ResponseType result;

            if (transformations.empty()) {
              result = withoutTransformations(key, value);
            }
            else {
              result = withTransformations(key, value);
            }

            if (result == OK) {
              response->body().appendText("200 ", 4);
              response->body().appendText(basics::StringUtils::escapeHex(value.data, '%'));
              response->body().appendText("\n", 1);
            }
            else {
              response->body().appendText("500 key not found\n");
            }
          }
        }

      private:
        ResponseType withoutTransformations (char const* key, Value& value) {
          value.identifier = key;

          return model->getValue(value);
        }



        ResponseType withTransformations (char const* key, Value& value) {
          map<string, string> dictionary;
          rest::ConnectionInfo const& connectionInfo = request->connectionInfo();

          dictionary["0"] = key;
          dictionary["SERVER-IP"] = connectionInfo.serverAddress;
          dictionary["SERVER-PORT"] = basics::StringUtils::itoa(connectionInfo.serverPort);
          dictionary["CLIENT-IP"] = connectionInfo.clientAddress;
          dictionary["CLIENT-PORT"] = basics::StringUtils::itoa(connectionInfo.clientPort);

          size_t pos = 0;

          for (vector<Transformation>::const_iterator i = transformations.begin();  i != transformations.end();  ++i) {
            Transformation const& transformation = *i;

            string n = transformation.evaluate(dictionary);

            if (n.empty()) {
              return ERROR;
            }

            value.identifier = n;
            ResponseType result = model->getValue(value);

            if (result != OK) {
              return result;
            }

            ++pos;

            dictionary[basics::StringUtils::itoa(pos)] = value.data;

            LOGGER_TRACE << "postfix key mapped to: '" << value.data << "'";
          }

          return OK;
        }

      private:
        vector<Transformation> const& transformations;
    };
  }
}

#endif
