////////////////////////////////////////////////////////////////////////////////
/// @brief memcached request
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Oreste Costa-Panaia
/// @author Copyright 2010, triagens GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#ifndef SIMPLEVOC_MCHANDLER_MCREQUEST_H
#define SIMPLEVOC_MCHANDLER_MCREQUEST_H 1

#include <Rest/LineRequest.h>

#include <Basics/StringBuffer.h>
#include <Basics/Exceptions.h>

namespace triagens {
  namespace simple {

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief line request
    ///
    /// The line server reads a line of text and sends onto here to determine what
    /// type of request we have.
    ////////////////////////////////////////////////////////////////////////////////

    class  MCRequest : public rest::LineRequest {
      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief command structure
        ////////////////////////////////////////////////////////////////////////////////

        struct CommandStructure {

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief constructor
          ////////////////////////////////////////////////////////////////////////////////

          CommandStructure ()
            : bytes(-1),
              bytesRemaining(0),
              deletionTime(0),
              expireTime(-1),
              flags(0),
              flushAllDelay(0),
              increment(0),
              noreply(false) {
            body.initialise();
            keys.push_back( pair<char*,size_t>(0,0) );
          }

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief destructor
          ////////////////////////////////////////////////////////////////////////////////

          ~CommandStructure () {
            body.free();

            for (vector< pair<char*,size_t> >::iterator j = keys.begin();  j != keys.end();  ++j) {
              if ( (*j).first != 0) {
                delete[] (*j).first;
              }
            }
          }

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief the key or keys which to store or retrieve the data
          ////////////////////////////////////////////////////////////////////////////////

          vector< pair<char*,size_t> > keys;

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief storage for the blob of data which is sent
          ////////////////////////////////////////////////////////////////////////////////

          basics::StringBuffer body;

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief the number of bytes in the blob;
          ////////////////////////////////////////////////////////////////////////////////

          size_t bytes;

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief bytes still needed
          ////////////////////////////////////////////////////////////////////////////////

          size_t bytesRemaining;

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief amount of time to wait before deleting
          ////////////////////////////////////////////////////////////////////////////////

          int64_t deletionTime;

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief if true deletion is pending
          ////////////////////////////////////////////////////////////////////////////////

          bool deletionPending;

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief number of seconds before the item is automatically deleted
          ////////////////////////////////////////////////////////////////////////////////

          int64_t expireTime;

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief 64 bits of various flags which the client can use
          ////////////////////////////////////////////////////////////////////////////////

          uint64_t flags;

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief amount of time to wait before we indicate that the item as deleted
          ////////////////////////////////////////////////////////////////////////////////

          uint64_t flushAllDelay;

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief amount to increment/decrement
          ////////////////////////////////////////////////////////////////////////////////

          uint64_t increment;

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief if true no reply is sent back to the client
          ////////////////////////////////////////////////////////////////////////////////

          bool noreply;

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief check and store if no update.
          ///
          /// Everytime I update the blob generate a unique 64 bit number get returns
          /// cas_unique.
          ////////////////////////////////////////////////////////////////////////////////

          uint64_t casUnique;
        };

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief various requests
        ////////////////////////////////////////////////////////////////////////////////

        enum MCRequestType {
          MC_REQUEST_ADD,
          MC_REQUEST_APPEND,
          MC_REQUEST_CAS,
          MC_REQUEST_DATABASE, // sets the database to use - not yet implemented
          MC_REQUEST_DECR,
          MC_REQUEST_DELETE,
          MC_REQUEST_FLUSH_ALL,
          MC_REQUEST_GET,
          MC_REQUEST_GETS,
          MC_REQUEST_INCR,
          MC_REQUEST_INVALID,
          MC_REQUEST_PGET,     // partial get - extension to memcache interface
          MC_REQUEST_PGETS,    // partial gets - extension to memcache interface
          MC_REQUEST_PREPEND,
          MC_REQUEST_QUERY,    // sets the query to use - not yet implemented
          MC_REQUEST_QUIT,
          MC_REQUEST_REPLACE,
          MC_REQUEST_SET,
          MC_REQUEST_STATS,
          MC_REQUEST_VERSION
        };

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief constructor
        ////////////////////////////////////////////////////////////////////////////////

        MCRequest ();

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief constructor
        ////////////////////////////////////////////////////////////////////////////////

        explicit MCRequest (string const& commandLine);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief constructor
        ////////////////////////////////////////////////////////////////////////////////

        MCRequest (char const* commandLine, size_t length);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief destructor
        ////////////////////////////////////////////////////////////////////////////////

        virtual ~MCRequest ();

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        ////////////////////////////////////////////////////////////////////////////////

        void setLineRequestInvalid ()  {
          type = MC_REQUEST_INVALID;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        ////////////////////////////////////////////////////////////////////////////////

        void addBodyLine (char const*, size_t) {
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        ////////////////////////////////////////////////////////////////////////////////

        void addBody (char const* ptr, size_t length) {
          cmdStructure.body.appendText(ptr, length);
          cmdStructure.bytesRemaining -= length;
        }

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief returns what action the command line specified
        ////////////////////////////////////////////////////////////////////////////////

        MCRequestType requestType () const {
          return type;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief returns what action the command line specified as string
        ////////////////////////////////////////////////////////////////////////////////

        string requestTypeString () const {
          switch (type) {
            case MC_REQUEST_ADD:       return "MC_REQUEST_ADD";
            case MC_REQUEST_APPEND:    return "MC_REQUEST_APPEND";
            case MC_REQUEST_CAS:       return "MC_REQUEST_CAS";
            case MC_REQUEST_DATABASE:  return "MC_REQUEST_DATABASE";
            case MC_REQUEST_DECR:      return "MC_REQUEST_DECR";
            case MC_REQUEST_DELETE:    return "MC_REQUEST_DELETE";
            case MC_REQUEST_FLUSH_ALL: return "MC_REQUEST_FLUSH_ALL";
            case MC_REQUEST_GET:       return "MC_REQUEST_GET";
            case MC_REQUEST_GETS:      return "MC_REQUEST_GETS";
            case MC_REQUEST_INCR:      return "MC_REQUEST_INCR";
            case MC_REQUEST_INVALID:   return "MC_REQUEST_INVALID";
            case MC_REQUEST_PGET:      return "MC_REQUEST_PGET";
            case MC_REQUEST_PGETS:     return "MC_REQUEST_PGETS";
            case MC_REQUEST_PREPEND:   return "MC_REQUEST_PREPEND";
            case MC_REQUEST_QUERY:     return "MC_REQUEST_QUERY";
            case MC_REQUEST_QUIT:      return "MC_REQUEST_QUIT";
            case MC_REQUEST_REPLACE:   return "MC_REQUEST_REPLACE";
            case MC_REQUEST_SET:       return "MC_REQUEST_SET";
            case MC_REQUEST_STATS:     return "MC_REQUEST_STATS";
            case MC_REQUEST_VERSION:   return "MC_REQUEST_VERSION";
          }

          THROW_INTERNAL_ERROR("unknown request type");
        }

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief command line structure of the request
        ////////////////////////////////////////////////////////////////////////////////

        CommandStructure cmdStructure;

      private:
        void parseHeader (size_t length);

        bool readInteger (char*& start, char const* end, int64_t& value);
        bool readInteger (char*& start, char const* end, uint32_t& value);
        bool readInteger (char*& start, char const* end, uint64_t& value);

#ifdef TC_OVERLOAD_FUNCS_SIZE_T

        bool readInteger (char*& start, char const* end, size_t& value) {
          sizetint_t v = 0;
          bool ok = readInteger(start, end, v);
          value = v;
          return ok;
        }
#endif

        bool readString (char*& start, char const* end, pair<char*,size_t>& value);
        bool readStringLower (char*& start, char const* end, pair<char*,size_t>& value);
        bool readStringNoReply (char*& start, char const* end, bool& value);
        bool readStringTest (char* start, char const* end);

      private:
        MCRequestType type;
        char* cmdLine;
    };
  }
}

#endif
