////////////////////////////////////////////////////////////////////////////////
/// @brief class for result rows
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Achim Brandt
/// @author Copyright 2008-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "VariantRow.h"

#include <Basics/Logger.h>
#include <Basics/StringBuffer.h>
#include <Basics/VariantBoolean.h>
#include <Basics/VariantInt64.h>
#include <Basics/VariantInt32.h>
#include <Basics/VariantInt16.h>
#include <Basics/VariantUInt64.h>
#include <Basics/VariantUInt32.h>
#include <Basics/VariantUInt16.h>
#include <Basics/VariantDouble.h>
#include <Basics/VariantFloat.h>
#include <Basics/VariantDatetime.h>
#include <Basics/VariantDate.h>
#include <Basics/VariantString.h>
#include <Basics/VariantNull.h>
#include <Basics/VariantBlob.h>
#include <Hpdf/AttributeDescriptor.h>
#include <ResultGenerator/ResultGenerator.h>

using namespace triagens;
using namespace triagens::basics;
using namespace triagens::rest;
using namespace triagens::hpdf;

namespace triagens {
  namespace hpdf {

    // -----------------------------------------------------------------------------
    // static public methods
    // -----------------------------------------------------------------------------

    void VariantRow::initialise () {
      ResultGenerator::registerObject(VARIANT_ROW, ResultGenerator::RESULT_GENERATOR_HTML,  VARIANT_ARRAY);
      ResultGenerator::registerObject(VARIANT_ROW, ResultGenerator::RESULT_GENERATOR_JSON,  VARIANT_ARRAY);
      ResultGenerator::registerObject(VARIANT_ROW, ResultGenerator::RESULT_GENERATOR_JSONX, VARIANT_ARRAY);
      ResultGenerator::registerObject(VARIANT_ROW, ResultGenerator::RESULT_GENERATOR_PHP,   VARIANT_ARRAY);
      ResultGenerator::registerObject(VARIANT_ROW, ResultGenerator::RESULT_GENERATOR_XML,   VARIANT_ARRAY);
    }

    // -----------------------------------------------------------------------------
    // constructors and destructors
    // -----------------------------------------------------------------------------

    VariantRow::VariantRow () {
    }



    VariantRow::VariantRow (Row const* row, AttributeDescriptor const* desc) {
      static set<string> empty;
      generate(reinterpret_cast<void const*>(row), desc, empty);
    }



    VariantRow::VariantRow (void const* row, AttributeDescriptor const* desc) {
      static set<string> empty;
      generate(row, desc, empty);
    }



    VariantRow::VariantRow (Row const* row, AttributeDescriptor const* desc, set<string> const& hidden) {
      generate(reinterpret_cast<void const*>(row), desc, hidden);
    }



    VariantRow::VariantRow (void const* row, AttributeDescriptor const* desc, set<string> const& hidden) {
      generate(row, desc, hidden);
    }

    // -----------------------------------------------------------------------------
    // VariantObject methods
    // -----------------------------------------------------------------------------

    VariantObject* VariantRow::clone () const {
      VariantRow* copy = new VariantRow();

      for (map<string, VariantObject*>::const_iterator i = mapping.begin();  i != mapping.end();  ++i) {
        copy->add(i->first, i->second->clone());
      }

      return copy;
    }

    // -----------------------------------------------------------------------------
    // private methods
    // -----------------------------------------------------------------------------

    void VariantRow::generate (void const* row, AttributeDescriptor const* desc, set<string> const& hidden) {
      for (AttributeDescriptor const* d = desc;  d->name != 0;  ++d) {
        if (hidden.find(d->name) == hidden.end()) {
          generate(row, d);
        }
        else {
          LOGGER_TRACE << "field '" << d->name << "' is hidden";
        }
      }
    }



    void VariantRow::generate (void const* row, AttributeDescriptor const* descriptor) {
      size_t off = descriptor->offset;

      switch (descriptor->type) {
        case AT_BOOLEAN:
          add(descriptor->name, new VariantBoolean(0 != (* (uint16_t*) (((char*) row) + off))));
          break;

        case AT_BIG_INTEGER:
          add(descriptor->name, new VariantInt64(* (int64_t*) (((char*) row) + off)));
          break;

        case AT_INTEGER:
          add(descriptor->name, new VariantInt32(* (int32_t*) (((char*) row) + off)));
          break;

        case AT_SMALL_INTEGER:
          add(descriptor->name, new VariantInt16(* (int16_t*) (((char*) row) + off)));
          break;

        case AT_UNSIGNED_BIG_INTEGER:
          add(descriptor->name, new VariantUInt64(* (uint64_t*) (((char*) row) + off)));
          break;

        case AT_UNSIGNED_INTEGER:
          add(descriptor->name, new VariantUInt32(* (uint32_t*) (((char*) row) + off)));
          break;

        case AT_UNSIGNED_SMALL_INTEGER:
          add(descriptor->name, new VariantUInt16(* (uint16_t*) (((char*) row) + off)));
          break;

        case AT_DOUBLE:
          add(descriptor->name, new VariantDouble(* (double*) (((char*) row) + off)));
          break;

        case AT_FLOAT:
          add(descriptor->name, new VariantFloat(* (float*) (((char*) row) + off)));
          break;

        case AT_DATETIME:
          add(descriptor->name, new VariantDatetime(* (datetime_t*) (((char*) row) + off)));
          break;

        case AT_DATE:
          add(descriptor->name, new VariantDate(* (date_t*) (((char*) row) + off)));
          break;

        case AT_STRING:
          {
            char const* ptr = * (char const**) (((char*) row) + off);

            if (ptr == 0) {
              add(descriptor->name, new VariantNull());
            }
            else {
              add(descriptor->name, new VariantString(ptr));
            }
          }

          break;

        case AT_BLOB:
          {
            blob_t * b = (blob_t*) (((char*) row) + off);
            add(descriptor->name, new VariantBlob(b->data, b->length));
          }
          break;

        default:
          THROW_INTERNAL_ERROR("missing type");
      }
    }
  }
}

