////////////////////////////////////////////////////////////////////////////////
/// @brief High-Performance Database Framework made by triagens
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Copyright 2008-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#ifndef TRIAGENS_HPDF_TABLE_H
#define TRIAGENS_HPDF_TABLE_H 1

#include <iostream>
#include <fstream>

#include <Basics/Mutex.h>
#include <Basics/MutexLocker.h>
#include <Basics/ReadWriteLock.h>

#include <Hpdf/Common.h>
#include <Basics/VariantVector.h>


namespace triagens {
  namespace hpdf {
    class TableVisitor;
    struct Row;

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief abstract base class of tables
    ///
    /// A table is either a noble or a vassal. It is either direct or
    /// indirect. A noble table owns its row or index space. A vassal table uses
    /// the row or index space of another table. A direct table stores the rows
    /// in blocks of memory.  An indirect table stores a consecutive index. An
    /// index entry at least points to a row. It might contains copies of the
    /// rows values.
    ///
    /// - a table defines the type: row_t
    /// - a vassal tables defines the type: operand_row_t
    /// - a direct table defines the type: iterator
    /// - an indirect table defines the types: iterator, index_t
    ///
    /// Note that in case of an indirect table the iterator is always a pointer to
    /// index_t.
    ////////////////////////////////////////////////////////////////////////////////

    class Table {
      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief list of various transactions which can occur to a row
        ////////////////////////////////////////////////////////////////////////////////

        enum TransactionType {
          TRANSACTION_CREATE     = 1,
          TRANSACTION_MODIFY     = 2,
          TRANSACTION_DELETE     = 3,
          TRANSACTION_UNDEFINED  = 99
        };


        ////////////////////////////////////////////////////////////////////////////////
        /// @brief A respository which is filled in by the server if recovery is activated
        ////////////////////////////////////////////////////////////////////////////////

        struct RecoveryStructure {
          fstream* logFile;      // log file used for logging transactions and for play back purposes
          string*   logFileName;  // the name of the log file we are dealing with

          // mutex for locking the transaction file so only one transaction can be made at a time
          triagens::basics::ReadWriteLock* logFileLock;

          // indicate whether the transaction file is used for playback
          bool playBack;

          // a number which is 1 less than the last snap shot taken
          size_t snapShotNumber;

          fstream* snapShot;      // log file used for logging transactions and for play back purposes
          string*   snapShotName;  // the name of the log file we are dealing with
          // mutex for locking the all tables
          triagens::basics::ReadWriteLock* snapShotLock;

          double snapShotDateTime;

          bool inSnapShot;
        };

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief constructor
        ////////////////////////////////////////////////////////////////////////////////


        Table () : _allowSnapShot(false), _recoveryRowId(0) {
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief destructor
        ////////////////////////////////////////////////////////////////////////////////

        virtual ~Table () {
        }

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief size of table
        ////////////////////////////////////////////////////////////////////////////////

        virtual size_t size () const = 0;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief tries to reserve space
        ////////////////////////////////////////////////////////////////////////////////

        virtual void reserve (size_t) = 0;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief adds a row
        ////////////////////////////////////////////////////////////////////////////////

        virtual Row const* add (Row const *) = 0;

        virtual bool addRecovery (const char* data, const size_t dataLength);
        virtual bool modRecovery (const char* data, const size_t dataLength);
        virtual bool remRecovery (const char* data, const size_t dataLength);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief visits all rows
        ////////////////////////////////////////////////////////////////////////////////

        virtual void forall (TableVisitor *) const = 0;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief visits a row
        ////////////////////////////////////////////////////////////////////////////////

        virtual void forany (TableVisitor *) const = 0;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief returns a row or zero
        ////////////////////////////////////////////////////////////////////////////////

        virtual Row const* any () const;


        ////////////////////////////////////////////////////////////////////////////////
        /// @brief set whether the table takes part in a snapshot
        ////////////////////////////////////////////////////////////////////////////////

        bool getAllowSnapShot () const {
          return _allowSnapShot;
        }

        void setAllowSnapShot(bool allow) {
          _allowSnapShot = allow;
        }

        int getRecoveryFileType () const {
          return _recoveryFileType;
        }

        void setRecoveryFileType(int rfType) {
          _recoveryFileType = rfType; // < 1: no recovery, 1: text, 2: binary
        }



      protected:


        /////////////////////////////////////////////////////////////////////////////
        /// @brief Flag which tells us whether or not the table is too take part
        /// in a snapshot.
        /////////////////////////////////////////////////////////////////////////////

        bool _allowSnapShot;  // defaults to false -- so tables are NOT automatically snapshoted

        int _recoveryFileType; // <= 0: none, 1: text, 2: binary

        uint64_t _recoveryRowId;  // the value of the hidden attribute used to recovery -- incremented when rows are added to the table


      public:

        uint32_t _recoveryTableId; // a CRC value used to uniquely identify the table

        RecoveryStructure* _recovery; // shared recovery structure

      public:


        virtual size_t getRowSize () const ;

        virtual bool tableSnapShotBegin (const size_t snapShotID);
        virtual bool tableSnapShotDo ();
        virtual bool tableSnapShotEnd ();

        virtual bool tableRestoreSnapShot (ifstream& inputFile, const size_t numRows);

        virtual void dumpTableRow (void const* row, char*& data, size_t& dataLength);
        virtual void dumpTableRow (const void* startRow, const void* endRow);

        virtual bool restoreTableRow (void* row, const char* data, size_t dataLength);

        virtual bool setRecoveryStructure (RecoveryStructure* recover);

        virtual bool setRecoveryTableId (uint32_t id);


    };
  }
}

#endif
