////////////////////////////////////////////////////////////////////////////////
/// @brief sorted array
///
/// @file
/// Implementation of sorted arrays. The description structure
/// must define the following methods:
///
///   isLessFind(R* left, R* right) - used to perform a comparison on a find
///   isLessAdd(R* left, R* right) - used to perform a comparison when we add elements
///   Comparisons for a find and add are fundamentally different, a find can use partial
///   keys while an add uses all keys.
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Oreste Costa-Panaia
/// @author Copyright 2009-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#ifndef TRIAGENS_HPDF_SORTED_ARRAY_H
#define TRIAGENS_HPDF_SORTED_ARRAY_H 1

#include <Hpdf/Common.h>

namespace triagens {
  namespace hpdf {

    // ///////////////////////////////////////////////////////////////////////////////////
    // TODO: add fruit sort
    //       add support for locale sorting of strings
    // ///////////////////////////////////////////////////////////////////////////////////

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief sorted array
    ////////////////////////////////////////////////////////////////////////////////

    template <typename R, typename DESC> // storage of the rows R and the call back functions
    class SortedArray : boost::noncopyable {
      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief constructs a new sorted array
        ////////////////////////////////////////////////////////////////////////////////

        explicit  SortedArray (uint64_t mSize, uint64_t sSize, bool strictRemove = true, string sortAlg = "fruit") : desc() {
          initialise(mSize, sSize, strictRemove, sortAlg);
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief deletes a sorted array
        ////////////////////////////////////////////////////////////////////////////////

        ~SortedArray () {
          delete[] main_Array;
          delete[] side_Array;
        }

      private:

        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        // Sets some options which the SortArray can operate with
        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

        void initialise (uint64_t mSize, uint64_t sSize, bool strictRemove, string sortAlg) {
          main_Capacity = mSize;
          side_Capacity = sSize;
          stable_Array    = strictRemove; // if true, slower since 0 are removed during copy
          sort_Algorithm  = sortAlg; // sorting algorithm to use, quicksort, fruit sort etc.
          numSorts = 0;
          numExpansions = 0;
          this->initialise();
        }

        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        // main array initialisation
        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

        void initialise () {
          main_Array    = new R* [main_Capacity];
          memset (main_Array,0,sizeof(R*) * main_Capacity);
          main_Used       = 0;
          main_Active     = 0;
          main_ArrayEnd   = main_Array;

          // side array initialisation
          side_Array    = new R* [side_Capacity];
          memset(side_Array,0,sizeof(R*) * side_Capacity);
          side_Used       = 0;
          side_Active     = 0;
          side_ArrayEnd   = side_Array;

          // Invalidation of the arrays causes to the status of the sort to be reset
          sort_Status = -1;
        }

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the current number of non-zero records stored in the main array.
        ////////////////////////////////////////////////////////////////////////////////

        uint64_t mainActive () const {
          return main_Active;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the current size (memory allocated) of the main sorted array
        ////////////////////////////////////////////////////////////////////////////////

        uint64_t mainCapacity () const {
          return main_Capacity;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the number of elements which are currently stored in the main
        /// sorted array. This includes rows which are no longer valid - 0.
        ////////////////////////////////////////////////////////////////////////////////

        uint64_t mainUsed () const {
          return main_Used;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the current number of non-zero records stored in the side array.
        ////////////////////////////////////////////////////////////////////////////////

        uint64_t sideActive () const {
          return side_Active;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the current size (memory allocated) of the side sorted array
        ////////////////////////////////////////////////////////////////////////////////

        uint64_t sideCapacity () const {
          return side_Capacity;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the number of elements which are currently stored in the side
        /// sorted array. This includes rows which are no longer valid - 0.
        ////////////////////////////////////////////////////////////////////////////////

        uint64_t sideUsed () const {
          return side_Used;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Clears the main array and the unsorted side array.
        ////////////////////////////////////////////////////////////////////////////////

        void clearSortArray () {
          delete[] main_Array;
          delete[] side_Array;
          initialise();
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief finds a given element
        ////////////////////////////////////////////////////////////////////////////////

        void findElement (const R* element, vector<R*>* storage) {

          // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Keys have been sent, now find the elements which match the keys.
          // If the sort status is invalid, we must sort here.
          // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          if (sort_Status < 0) {
            numSorts++;
            std::sort(main_Array, main_ArrayEnd, DESC::isLessAdd);
            sort_Status = 1;
          }

          // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Go through the side array and see if we have any matches
          // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

          for (R** i = side_Array; (side_Used !=0) && (i < side_ArrayEnd); i++) {
            if (*i != 0 && DESC::isEqualFind(*i,element)) {
              storage->push_back(*i);
            }
          }

          // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Go through the main array and see if we have any matches
          // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

          pair<R**,R**> result = std::equal_range(main_Array,main_ArrayEnd,element,DESC::isLessFind);

          for (R** i = result.first; i < result.second; i++) {
            if (*i != 0) {
              storage->push_back(*i);
            }
          }

        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief finds a first given element (so called unique)
        ////////////////////////////////////////////////////////////////////////////////

        R* findUniqueElement (const R* element) {

          // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Keys have been sent, now find the elements which match the keys.
          // If the sort status is invalid, we must sort here.
          // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          if (sort_Status < 0) {
            numSorts++;
            std::sort(main_Array, main_ArrayEnd, DESC::isLessAdd);
            sort_Status = 1;
          }

          // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Go through the side array and see if we have any matches
          // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

          for (R** i = side_Array; (side_Used !=0) && (i < side_ArrayEnd); i++) {
            if (*i != 0 && DESC::isEqualFind(*i,element)) {
              return *i;
            }
          }

          // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Go through the main array and see if we have any matches
          // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

          pair<R**,R**> result = std::equal_range(main_Array,main_ArrayEnd,element,DESC::isLessFind);

          for (R** i = result.first; i < result.second; i++) {
            if (*i != 0) {
              return *i;
            }
          }

          return (0);
        }


        ////////////////////////////////////////////////////////////////////////////////
        /// @brief adds a new element
        ////////////////////////////////////////////////////////////////////////////////

        void addElement (R* element) {

          *side_ArrayEnd = element;
          side_ArrayEnd++;
          side_Used++;
          side_Active++;

          if (side_ArrayEnd - side_Capacity != side_Array) { return; }

          // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // We have reached capacity - so we now need to move the side array into the main
          // array. Invalid the sort status, since elements are being added to the main array.
          // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          sort_Status = -1;

          START_LOOP:

          // Is there enough room in the existing main array to hold the side Array?
          if (main_Capacity - main_Used > side_Capacity) {
            // Depending if we have stable arrays in use
            if (stable_Array) {
              // check whether or not the sidearray element is empty, if not then copy it
              uint64_t newElements = 0;
              for (R** i = side_Array; i != side_ArrayEnd; i++) {
                if (*i == 0) { continue; }
                *main_ArrayEnd = *i;
                main_ArrayEnd++;
                newElements++;
                *i = 0;
              }
              main_Used += newElements;
            }
            else {
              // copy all the elements from the side array into the main array
              memcpy(main_ArrayEnd, side_Array, side_Used * sizeof(R*)  );
              main_ArrayEnd += side_Used;
              main_Used     += side_Used;
              memset(side_Array, 0, sizeof(R*) * side_Capacity);
            }
            main_Active    += side_Active;
            side_Active     = 0;
            side_Used       = 0;
            side_ArrayEnd   = side_Array;
          }

          // Main array too small to hold the side array, extend the main array by 20%
          else {
            numExpansions++;
            main_Capacity += main_Capacity / 5;
            R** newMainArray = new R* [main_Capacity];
            memset(newMainArray, 0, sizeof(R*) * main_Capacity);

            // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            // If the number of active and used elements falls below a certain
            // percentage, force stable to be true
            // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

            bool tempStableArray = stable_Array;
            if ( (double(main_Active) / double(main_Used) ) < 0.5 ) {
              tempStableArray = true;
            }

            // Depending if we have stable arrays in use
            if (tempStableArray) {
              uint64_t newElements = 0;
              R** newMainArrayEnd   = newMainArray;
              for (R** i = main_Array; i != main_ArrayEnd; i++) {
                if (*i == 0) { continue; }
                *newMainArrayEnd = *i;
                newMainArrayEnd++;
                newElements++;
              }
              delete [] main_Array;
              main_Array      = newMainArray;
              main_ArrayEnd   = newMainArrayEnd;
              main_Used       = newElements;
              main_Active     = newElements;

            }
            else {
              // copy all the elements from the old main array into the new main array
              memcpy(newMainArray, main_Array, sizeof(R*) * main_Used);
              delete [] main_Array;
              main_Array      = newMainArray;
              main_ArrayEnd   = main_Array + main_Used;
            }
            goto START_LOOP;
          }
        }


        ////////////////////////////////////////////////////////////////////////////////
        /// @brief removes an element
        ////////////////////////////////////////////////////////////////////////////////

        void removeElement (const R* element) {

          // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Go through the side array and see if we have any matches
          // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          for (R** i = side_Array; i < side_ArrayEnd; i++) {
            if (*i != element) { continue; }
            *i = 0;
            side_Active--;
            return; // there can only be one match for R* element
          }

          // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Go through the main array and see if we have any matches
          // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          for (R** i = main_Array; i < main_ArrayEnd; i++) {
            if (*i != element) { continue; }
            *i = 0;
            main_Active--;
            return; // there can only be one match for R* element
          }

        }

      // -----------------------------------------------------------------------------
      // COMPARE FUNCTIONS
      // -----------------------------------------------------------------------------

      inline int CMP (const char* left, const char* right) {
        return strcmp(left, right);
      }

      inline int NCMP (const char* left, const char* right, const size_t lastKeyLength) {
        return strncmp(left, right, lastKeyLength);
      }



      void countDebug() {
        uint64_t temp = 0;
        for (R** i = main_Array; i != main_ArrayEnd; i++) {
          if (*i != 0) { temp++;}
        }
        cout << "main array = " << temp << endl;
        temp = 0;
        for (R** i = side_Array; i != side_ArrayEnd; i++) {
          if (*i != 0) { temp++;}
        }
        cout << "side array = " << temp << endl;
      }

      // -----------------------------------------------------------------------------
      // private variables
      // -----------------------------------------------------------------------------

      public:
        DESC desc; // stores a structure which contains the call functions used for comparison etc
      private:
        R** main_Array;
        uint64_t main_Capacity;
        uint64_t main_Used;
        uint64_t main_Active;

        R** side_Array;
        uint64_t side_Capacity;
        uint64_t side_Used;
        uint64_t side_Active;

        R** main_ArrayEnd;
        R** side_ArrayEnd;
        bool stable_Array;     // whether or not we remove entries no longer required when we create new arrays
        string sort_Algorithm; // the sort algorithm to use, e.g. fruit or std sort todo

        // book keeping stuff
        uint64_t numSorts;
        uint64_t numExpansions;

        // status of the sort
        int sort_Status;
    };
  }
}

#endif
