////////////////////////////////////////////////////////////////////////////////
/// @brief High-Performance Database Framework made by triagens
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. O
/// @author Copyright 2009-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#ifndef TRIAGENS_HPDF_HPDF_PAGE_H
#define TRIAGENS_HPDF_HPDF_PAGE_H 1

#include <Hpdf/Common.h>
#include <Basics/Logger.h>
#include <Basics/ReadWriteLock.h>
#include <Basics/VariantVector.h>
#include <Basics/VariantInt16.h>
#include <Basics/VariantInt32.h>
#include <Basics/VariantInt64.h>
#include <Basics/VariantBlob.h>
#include <Basics/VariantDate.h>
#include <Basics/VariantDatetime.h>
#include <Basics/VariantDouble.h>
#include <Basics/VariantFloat.h>
#include <Basics/VariantString.h>
#include <Basics/VariantUInt16.h>
#include <Basics/VariantUInt32.h>
#include <Basics/VariantUInt64.h>

#include <assert.h>

namespace triagens {
  namespace hpdf {

    // forward declare the iterator class, required for the Page class
    template<typename R, typename IX>
    class PageIterator;

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief Page used to store data rows. A linked list.
    ////////////////////////////////////////////////////////////////////////////////

    template<typename R, typename IX>
    class Page {
      friend class PageIterator<R, IX>;

      public:

        typedef PageIterator<R, IX> iterator;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Constructs a new page.
        ////////////////////////////////////////////////////////////////////////////////

        Page (size_t pageSize, Page<R, IX>* pPage, double fill, RowDeletionType rowDelMethod)
          : pageSize(pageSize), length(0), capacity(0), initialFill(fill), rowDeleteMethod(rowDelMethod) {

          // determine the page size in rows
          maxRows = (pageSize  + sizeof(R) - 1) / sizeof(R);

          // we cannot allocate space here, as we are using the Page also for vassals
          dataBegin = 0;
          dataEnd   = dataBegin;

          // create the link
          nextPage = 0;
          prevPage = pPage;

          number = (pPage == 0) ? 0 : (pPage->getNumber() + 1);

          // snap shot stuff
          _tableSnapShotID    = -1;
          _pageSnapShotStatus = -1;
          _useCopy            = false;
          _copyBegin          = 0;
          _copyEnd            = 0;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Destructor. Destroys a page.
        ////////////////////////////////////////////////////////////////////////////////

        ~Page () {

          // we cannot delete the memory allocated from within the destructor since
          // some pages are used to refer to the data -- that is they are temporary pages
          // and we only delete the pages but NOT the actual rows
        }

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the next page in the linked list.
        ////////////////////////////////////////////////////////////////////////////////

        inline Page<R, IX>* operator++ () const {
          return nextPage;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the previous page in the linked list.
        ////////////////////////////////////////////////////////////////////////////////

        inline Page<R, IX>* operator-- () const {
          return prevPage;
        }

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the page size.
        ////////////////////////////////////////////////////////////////////////////////

        inline size_t getPageSize () const {
           return pageSize;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the maximum number of rows which can be allocated on a page.
        ////////////////////////////////////////////////////////////////////////////////

        inline size_t getMaxRows () const {
           return maxRows;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the actual number of rows on the page.
        ////////////////////////////////////////////////////////////////////////////////

        inline size_t getLength () const {
           return length;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Sets the number of rows on the page.
        ////////////////////////////////////////////////////////////////////////////////

        inline void setLength (size_t value) {
          length = value;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Recalculates the number of rows on a page using the first and last row.
        ////////////////////////////////////////////////////////////////////////////////

        void resetLength () {
          length = dataEnd - dataBegin;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the number of the page. Each page has an internal counter.
        ////////////////////////////////////////////////////////////////////////////////

        inline size_t getNumber () const {
           return number;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief returns true if the page is full.
        ////////////////////////////////////////////////////////////////////////////////

        inline bool isFull () const {
           return length >= maxRows;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the address of the first row allocated on the page.
        ////////////////////////////////////////////////////////////////////////////////

        inline R* begin () const {
           return dataBegin;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Sets the pointer value of the first data row on the page.
        ////////////////////////////////////////////////////////////////////////////////

        inline void setDataStart (R* dataPtr) {
          dataBegin = dataPtr;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the address of the last row allocated on the page.
        ////////////////////////////////////////////////////////////////////////////////

        inline R* end () const {
           return dataEnd;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Sets the pointer value of the last data row on the page.
        ////////////////////////////////////////////////////////////////////////////////

        inline void setDataEnd (R* dataPtr) {
          dataEnd = dataPtr;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns a pointer to the next page. Same as the operator++ ().
        ////////////////////////////////////////////////////////////////////////////////

        inline Page<R, IX>* next () const {
          return nextPage;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Sets the nextPage pointer associated with this page.
        ////////////////////////////////////////////////////////////////////////////////

        inline void setNextPage (Page<R, IX>* value) {
          nextPage = value;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns a pointer to the previous page. Same as operator-- ().
        ////////////////////////////////////////////////////////////////////////////////

        inline Page<R, IX>* prev () const {
          return prevPage;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Sets the prevPage pointer associated with this page.
        ////////////////////////////////////////////////////////////////////////////////

        inline void setPrevPage (Page<R, IX>* value) {
          prevPage = value;
        }

      private:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Adds a new page to the linked list and marks it as the nextPage.
        ////////////////////////////////////////////////////////////////////////////////

        inline Page<R, IX>* addPage () {
          if (nextPage != 0) {
            throw "internal error";
          }

          nextPage = new Page<R, IX>(pageSize, this, initialFill, rowDeleteMethod);

          return nextPage;
        }

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Adds one row of data to the page.
        ///
        /// It is implicit that there is enough room on the page for this
        /// happen, further an undefined row is created as the last row on the
        /// page
        ////////////////////////////////////////////////////////////////////////////////

        inline R* addRow (R const * row, IX& tableIndexes) {

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Warning: currently coded so that only one thread can add a row to the table
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Are we currently in the process of taking a table snap shot?
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          if (_tableSnapShotID != -1) {

            bool pageLock = false;

            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            // Place a read lock so we can ensure that _pageSnapShotStatus is correct
            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            if ( !_pageSnapShotStatusLock.readLock() ) {
              return 0;
            }

            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            // There are now three possiblities
            // _pageSnapShotStatus == -1 : page snap shot not yet done
            // _pageSnapShotStatus ==  0 : in the process of doing it
            // _pageSnapShotStatus ==  1 : page snap shot completed
            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

            if (_pageSnapShotStatus == -1) {
              // we have to create a copy of the page, this copy is used by the snap shot
              // of course we only create a copy of the page, if we do not already have!
              this->createSnapShotCopy();
            }

            else if (_pageSnapShotStatus == 0) {
              // block until the snap shot has finished -- only if we do not have a copy
              if (!_useCopy) {
                if (!_pageLock.writeLock()) {
                  if (_pageSnapShotStatusLock.unlock()) {
                  }
                  return 0;
                }
                pageLock = true;
              }
            }

            else if (_pageSnapShotStatus == 1) {
              // since we have finished with the page, nothing to do.
            }

            memcpy(dataEnd, row, sizeof(R));
            ++length;
            // remove status lock
            if (_pageSnapShotStatusLock.unlock()) {
            }

            if (pageLock) {
              if (_pageLock.unlock()) {
              }
            }

            // add any indexes required for this table
            tableIndexes.addIndexes(dataEnd);
            return ++dataEnd;
          }


          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // We are not currently in a snap shot process - so proceed normally
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          else {
            memcpy(dataEnd, row, sizeof(R));
            ++length;
            // add any indexes required for this table

           tableIndexes.addIndexes(dataEnd);
            return ++dataEnd;
          }

        } // end of function addRow

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Attempts to modify a given oldRow with a newRow
        ///
        /// Indices are handled automatically
        ////////////////////////////////////////////////////////////////////////////////

        inline R* modifyRow (const R* oldRow, const R* newRow, IX& tableIndexes, iterator& beginIt, iterator& endIt) {

          R* oRow = const_cast<R*>(oldRow);
          R* nRow = const_cast<R*>(newRow);

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Are we currently in the process of taking a table snap shot?
          // A tableSnapShotID of -1 indicates that no TABLE snap shot is taking place
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Snap shot IS taking place
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          if (_tableSnapShotID != -1) {

            bool pageLock = false;

            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            // Place a read lock so we can ensure that _pageSnapShotStatus is correct
            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            if ( !_pageSnapShotStatusLock.readLock() ) {
              // modification could not take place
              return 0;
            }

            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            // There are now three possiblities
            // _pageSnapShotStatus == -1 : page snap shot not yet done
            // _pageSnapShotStatus ==  0 : in the process of doing it
            // _pageSnapShotStatus ==  1 : page snap shot completed
            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            // The snap shot is taking place, however this page has not been completed
            // make a copy if we have not made one before
            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            if (_pageSnapShotStatus == -1) {
              this->createSnapShotCopy();
            }

            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            // The snap shot is taking place, and this page IS currently being dumped to file
            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            else if (_pageSnapShotStatus == 0) {
              // block until the snap shot has finished -- only if we do not have a copy
              // if the snap shot is using a copy, then no need to worry here
              if (!_useCopy) {
                if (!_pageLock.writeLock()) {
                  if (_pageSnapShotStatusLock.unlock()) {
                  }
                  return 0;
                }
                pageLock = true;
              }
            }

            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            // The snap shot is taking place, this page has already been completed
            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            else if (_pageSnapShotStatus == 1) {
              // nothing really to do here
            }

            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            // first remove any indexes which are associated with the oldRow
            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            tableIndexes.preModifyIndexes(oRow, nRow);

            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            // next copy the newRow into the memory where the oldRow was
            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            memcpy(oRow, nRow, sizeof(R));

            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            // Add any indexes which are associated with the newRow
            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            tableIndexes.postModifyIndexes(oRow, nRow);

            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            // Remove the page status lock
            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            if (_pageSnapShotStatusLock.unlock()) {
            }

            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            // If we added a page lock, because the snap shot for this page was taking place
            // then we remove it.
            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            if (pageLock) {
              if (_pageLock.unlock()) {
              }
            }

            return nRow;

          }

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Snap shot NOT taking place
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          else {
            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            // first remove any indexes which are associated with the oldRow
            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            tableIndexes.preModifyIndexes(oRow, nRow);

            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            // next copy the newRow into the memory where the oldRow was
            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            memcpy(oRow, nRow, sizeof(R));

            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            // finally add any indexes which are associated with the newRow
            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            tableIndexes.postModifyIndexes(oRow, nRow);

            return oRow;
          }

          return 0;

        }


        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Removes one data row from the page.
        /// If the table has associated indexes, then these indices are also removed.
        ////////////////////////////////////////////////////////////////////////////////


        inline bool removeRow (R* row, IX& tableIndexes, iterator& beginIt, iterator& endIt) {

          // /////////////////////////////////////////////////////////////////////////////////
          // WARNING: This causes fragment of the page. Require some sort of defragmentation.
          //          In the worst case you could end up with one row per page if you use the
          //          stable row deletion method.
          //
          // WARNING: Not thread safe. If two requests to remove row occur here, then you have lots of trouble.
          //
          // WARNING: If you are in the middle of iterating your rows, and BEGIN, END iterators change,
          //          then trouble again.
          //
          // TODO: Consider locking the current page, prev page and next page, and the iterators.
          //
          // TODO: Filling in an empty row (as in the stable deletion method below), modifies the pointers on all
          //       subsequent rows. This implies that all indexes (which would store these pointers)
          //       need to be modified for these rows. This can be slow if the Page Size is large.
          //       Consider adding syntax to the table definition which indicates how rows are to
          //       be removed. E.g. just leave an empty row and let the iterator skip the row.
          // /////////////////////////////////////////////////////////////////////////////////

          bool result = false;

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Different removal methods depending on whether or not snap shot taking
          // place.
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Snap shot IS taking place
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          if (_tableSnapShotID != -1) {

            // here we switch to various removal method
            if (rowDeleteMethod == RD_UNSTABLE ) {
              result = this->removeRowUnstableSnapShot(row, tableIndexes, beginIt, endIt);
            }
            else {
              result = this->removeRowStableSnapShot(row, tableIndexes, beginIt, endIt);
            }

          }

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Snap shot IS NOT taking place
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          else {
            // here we switch to various removal method
            if (rowDeleteMethod == RD_UNSTABLE ) {
              result = this->removeRowUnstable(row, tableIndexes, beginIt, endIt);
            }
            else {
              result = this->removeRowStable(row, tableIndexes, beginIt, endIt);
            }
          }

          return result;

        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Deletes the data stored in the page.
        ////////////////////////////////////////////////////////////////////////////////

        void deleteData () {
          delete[] dataBegin;

          dataBegin = 0;
          dataEnd   = 0;
          length    = 0;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Determines if there is enough room on the page for an additonal data row.
        /// If not, then we increase the capacity of the page by 20%, ensuring
        /// that the pagesize is not exceeded.
        ////////////////////////////////////////////////////////////////////////////////

        inline void checkCapacity (IX& tableIndexes) {
          if (length == capacity) {
            if (maxRows == 0) {
              throw "internal error";
            }

            // determine the capacity of newly filled tables
            if (capacity == 0) {
              capacity = max(int(maxRows * initialFill), 1);
            }
            else {
              capacity = size_t(1 + 1.2 * capacity);
            }

            // capacity cannot be more than pageSize
            if (capacity > maxRows) {
              capacity = maxRows;
            }

            R* copy = new R[capacity];

            if (dataBegin != 0) {
              memcpy(copy, dataBegin, length * sizeof(R));

              // replace what the indexes point to
              for (size_t j = 0;  j < length;  ++j) {
                tableIndexes.replaceIndexes(&dataBegin[j], &copy[j]);
              }

              delete[] dataBegin;
            }

            dataBegin = copy;
            dataEnd   = dataBegin + length;

          }
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the total number of pages in memory.
        /// Should be equivalent to the page number of the last page stored in the
        /// table object.
        ////////////////////////////////////////////////////////////////////////////////

        size_t numberPages () {
          if (nextPage != 0) {
            return 1 + nextPage->numberPages();
          }

          return 1;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Increments the data pointer on a page.
        /// If the end of the page is reached, moves onto the next page. If the
        /// end of the pages are reached, then the data pointer is moved to one
        /// position past the last data row on the last page - the last
        /// undefined row.
        /////////////////////////////////////////////////////////////////////////

        inline void incrementData (Page<R, IX>*& currentPage, R*& currentData) {
          currentData++;

          // we are at the undefined row so move onto the next page
          if (currentData == currentPage->end()) {
            Page<R, IX>* nPage = currentPage->next();

            if (nPage != 0) {
              currentPage = nPage;
              currentData = currentPage->begin();
            }
          }
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the total number rows stored in all the pages.
        ////////////////////////////////////////////////////////////////////////////////

        size_t size () {
          if (nextPage != 0) {
            return (dataEnd - dataBegin) + nextPage->size();
          }

          return dataEnd - dataBegin;
        }


      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Snapshot methods
        ////////////////////////////////////////////////////////////////////////////////

        inline void createSnapShotCopy () {
          if (_copyBegin != 0) {
            return;
          }
          _copyBegin = new R[length];
          _useCopy = true;
          memcpy(_copyBegin, dataBegin, length * sizeof(R));
        }

        inline const int& getPageSnapShotStatus () const {
          return _pageSnapShotStatus;
        }

        inline const bool& getUseSnapShotCopy () const {
          return _useCopy;
        }


        inline int64_t getTableSnapShotID () const {
          return _tableSnapShotID;
        }

        inline bool setAuxBlockingLock() {
          return _auxBlockingLock.writeLock();
        }

        inline bool setAuxBlockingUnlock() {
          return _auxBlockingLock.unlock();
        }

        inline bool setPageStatusReadLock () {
          return _pageSnapShotStatusLock.readLock();
        }

        inline bool setPageStatusUnlock () {
          return _pageSnapShotStatusLock.unlock();
        }

        inline bool setPageWriteLock () {
          return _pageLock.writeLock();
        }

        inline bool setPageUnlock () {
          return _pageLock.unlock();
        }

        //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        // Sets the status of every page -- indicating either we have started a snap shot
        // or we have finished.
        //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

        void setSnapShotStatus (Table* table, int64_t id, bool& ok) {
          if (! ok) {
            return;
          }
          _tableSnapShotID    = id; // non-zero integer indicates that snap shot has started
          _pageSnapShotStatus = -1; // indicates that we have not taken a snap shot of this page.
          _useCopy            = false;
          _removeInSnapShot   = false;

          if (id == -1) {
            if (_copyBegin != 0) {
              delete [] _copyBegin;
              _copyBegin = 0;
            }
          }
          else {
            _snapShotTable = table;
            if (_auxBlockingLock.readLock()) {
              ok = false;
              return;
            }
          }

          if (nextPage == 0) {
            return;
          }
          nextPage->setSnapShotStatus(table, id, ok); // iterate over all pages
        }

        void setSnapShotStatus (int64_t id) {
          _tableSnapShotID    = id; // non-zero integer indicates that snap shot has started
          _pageSnapShotStatus = -1; // indicates that we have not taken a snap shot of this page.
          _useCopy            = false;
          _removeInSnapShot   = false;
       }

        //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        // Actually dumps the contents of the page
        //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

        void pageSnapShotDump (R* startRow, R* endRow) {
          _snapShotTable->dumpTableRow(startRow, endRow);
        }

        void pageSnapShotDo (bool& snapShotOk) {

          if (! snapShotOk) {
            if (!_auxBlockingLock.unlock()) {}
            return;
          }

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // First we have to change the status of the page to 0: meaning that we have
          // started the snap shot of this page. We have to block until we are free
          // to change the status
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          if (!_pageSnapShotStatusLock.writeLock()) {
            snapShotOk = false;
            if (!_auxBlockingLock.unlock()) {}
            LOGGER_ERROR << "Internal error  snap shot: can not write lock page.";
            return;
          }

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // set the status of this page to indicate that snap shot has started
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          this->_pageSnapShotStatus = 0;

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Status of page has been changed, so we can unlock
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          if (!_pageSnapShotStatusLock.unlock()) {
            if (!_auxBlockingLock.unlock()) {}
            snapShotOk = false;
            LOGGER_ERROR << "Internal error  snap shot: can not unlock page.";
            return;
          }

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Here we dump the contents. Either we use actual contents of the page,
          // or we use a copy which has been previously made. If we use a copy we
          // do not need to lock the page. If we use the actual contents, then we
          // must lock the page so nothing else can write to it.
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Here we are not using the page copy of the rows
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

          if (!_useCopy) {
            // lock the page
            if (!_pageLock.writeLock()) {
              LOGGER_ERROR << "Internal error  snap shot: can not lock page.";
              snapShotOk = false;
              if (!_auxBlockingLock.unlock()) {}
              return;
            }

            cout << __FILE__ << ":" << __FUNCTION__ << ":" << __LINE__ << ":" << "started page dump write lock" << endl;
            this->pageSnapShotDump(dataBegin, dataEnd);

            cout << __FILE__ << ":" << __FUNCTION__ << ":" << __LINE__ << ":" << "finished page dump write lock" << endl;


            if (!_pageLock.unlock()) {
              LOGGER_ERROR << "Internal error  snap shot: can not lock page.";
              snapShotOk = false;
              if (!_auxBlockingLock.unlock()) {}
              return;
            }
            cout << __FILE__ << ":" << __FUNCTION__ << ":" << __LINE__ << ":" << "removed page dump write lock" << endl;

          }

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Here we ARE using the page copy of the rows. No requirement to lock page
          // since any modifications to page are not added to the copy
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

          else {

            cout << __FILE__ << ":" << __FUNCTION__ << ":" << __LINE__ << ":" << "started page dump write lock" << endl;
            this->pageSnapShotDump(_copyBegin, _copyEnd);

            // since we have finished with the copy of the page, we will delete it here
            delete [] _copyBegin;
            _copyBegin = 0;
            _useCopy = false;
            cout << __FILE__ << ":" << __FUNCTION__ << ":" << __LINE__ << ":" << "finished page dump write lock" << endl;
          }



          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // We must now change the status of the page to indicate that we have
          // finished with the page dump.
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

          if (!_pageSnapShotStatusLock.writeLock()) {
            LOGGER_ERROR << "Internal error snap shot : can not lock page.";
            snapShotOk = false;
            if (!_auxBlockingLock.unlock()) {}
            return;
          }


          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // change status of this page to indicate that we have finished with snap shot
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          this->_pageSnapShotStatus = 1;

          Page<R,IX>* oldNextPage = nextPage;

          if (!_pageSnapShotStatusLock.unlock() ) {
            LOGGER_ERROR << "Internal error snap shot: can not unlock page.";
            snapShotOk = false;
            if (!_auxBlockingLock.unlock()) {}
            return;
          }


          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // If this is not the last page of the linked list, continue down the list
          // While getting the next page, the next page value may change
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~




          if (!_auxBlockingLock.unlock()) {}

          if (oldNextPage != 0) {
            oldNextPage->pageSnapShotDo(snapShotOk);
          }

          return;
        }


      private:

        /////////////////////////////////////////////////////////////////////////////////////////////////////
        // Removes a row using unstable method (where the last row on the last page) is swapped with the
        // row to be removed. This function is only called when the table is NOT in the process of being
        // snap shot. See function removeRowUnstableSnapShot for the other case
        /////////////////////////////////////////////////////////////////////////////////////////////////////

        inline bool removeRowUnstable (R* row, IX& tableIndexes, iterator& beginIt, iterator& endIt) {

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // First remove the index for this row
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          tableIndexes.removeIndexes(row);

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Locate the LAST page and the LAST row
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          Page<R, IX>* lastPage = endIt.getPage();
          R* lastRow = lastPage->end();
          --lastRow;

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // The current row we are deleting is NOT the last row on the last page
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          if ( lastRow != row ) {
            //-----------------------------------------------------------------------------------------------
            // We need to remove the indexes of the last row since we are moving the last row
            //-----------------------------------------------------------------------------------------------
            tableIndexes.removeIndexes(lastRow);

            //-----------------------------------------------------------------------------------------------
            // Copy the memory allocation from the lastRow to the current row
            //-----------------------------------------------------------------------------------------------
            memcpy(row,lastRow,sizeof(R));

            //-----------------------------------------------------------------------------------------------
            // Add the index for the lastRow
            //-----------------------------------------------------------------------------------------------
            tableIndexes.addIndexes(row);
          }
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // The current row we are deleting IS the last row on the last page -- nothing to do
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          else {
          }

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // We have to fix the iterators and the last page
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          size_t lengthLastPage = lastPage->getLength();

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // There are at least two rows on the last page. Essentially reduce the page length by 1
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          if (lengthLastPage > 1) {
            lastPage->setDataEnd(lastRow);
            lastPage->resetLength();
            endIt.resetPage(lastPage);
            endIt.resetPointer(lastPage->getLength());
            return true;
          }

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // There is only ONE row left on the last page. We are required to remove the page from memory
          // and make the predecessor of the last page, then NEW last page.
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

          Page<R, IX>* predecessorLastPage = lastPage->prev();

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // There may be only ONE PAGE left in memory. In this case we keep this page
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          if (predecessorLastPage == 0) {
            lastPage->setDataEnd(lastRow);
            lastPage->resetLength();
            endIt.resetPointer(lastPage->getLength());
            return true;
          }

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Ok the last page is empty and it has a predecessor.
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

          // remove the memory allocated to this page.
          delete[] lastPage->begin();

          // remove this page from memory
          delete lastPage;

          // set the NEXT PAGE of the predecessor Page to NULL
          predecessorLastPage->setNextPage(0);

          // the previous page is now the end and affects the end iterator
          endIt.resetPage(predecessorLastPage);
          endIt.resetPointer(predecessorLastPage->getLength());

          return true;
        } // end of function removeRowUnstable


        inline bool removeRowUnstableSnapShot (R* row, IX& tableIndexes, iterator& beginIt, iterator& endIt) {

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Locate the LAST page and the LAST row
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

          Page<R, IX>* lastPage = endIt.getPage();
          R* lastRow = lastPage->end();
          --lastRow;

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Place a read lock so that the status of the page does not change
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          if ( !_pageSnapShotStatusLock.readLock() ) {
            // removal could not take place
            return false;
          }


          if (this == lastPage) {

            // snap shot for this page has not started
            if ( _pageSnapShotStatus == -1 ) {

              // there is only one row left of this page so this page may be removed if it is not the first page
              if (length == 1) {
                // block until page has completely finished
                //release the read lock so the page can be dumped
                if (!_pageSnapShotStatusLock.unlock() ){
                  return false;
                }
                if (!this->setAuxBlockingLock()) {
                  return false;
                }
                if (!this->setAuxBlockingUnlock()) {
                  return false;
                }
              }
              else {
                this->createSnapShotCopy();
              }
            }

            // snap shot dump of page taking place
            else if ( _pageSnapShotStatus == 0 ) {
              // block until dump of page finished
              if (!_useCopy) {
                if (!_pageLock.writeLock()) {
                  if (!_pageSnapShotStatusLock.unlock() ){ }
                  return false;
                }
              }
              // remove lock
              if (!_pageLock.unlock()) {
              }

            }

            // snap shot finished for this page
            else if ( _pageSnapShotStatus == 1 ) {
            }

            // normal removal - finished snap shot
            bool result = this->removeRowUnstable(row, tableIndexes, beginIt, endIt);
            if (!_pageSnapShotStatusLock.unlock() ) {
              return false;
            }
            return result;

          }


          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Snap shot has not got past this page so could not have got past last page.
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          if (_pageSnapShotStatus == -1) {
            this->createSnapShotCopy();
            bool result = this->removeRowUnstable(row, tableIndexes, beginIt, endIt);
            if (!_pageSnapShotStatusLock.unlock() ) {
              return false;
            }
            return result;
          }


          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Snap shot is currently taking place for this page
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          if (_pageSnapShotStatus == 0) {
            // block until dump of page finished
            if (!_pageLock.writeLock()) {
              if (!_pageSnapShotStatusLock.unlock() ) { }
              return false;
            }
            if (_pageLock.unlock() ) {
            }
            bool result = this->removeRowUnstable(row, tableIndexes, beginIt, endIt);
            if (!_pageSnapShotStatusLock.unlock() ) {
              return false;
            }
            return result;
          }

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Snap shot has finished with this page. Since this page not last page block until snap shot has
          // finished with last page.
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          if ( ! lastPage->setAuxBlockingLock() ) {
            // removal could not take place
            if (!_pageSnapShotStatusLock.unlock() ) { }
            return false;
          }
          if ( ! lastPage->setAuxBlockingUnlock() ) {
            // removal could not take place
            if (!_pageSnapShotStatusLock.unlock() ) { }
            return false;
          }

          bool result = this->removeRowUnstable(row, tableIndexes, beginIt, endIt);
          if (!_pageSnapShotStatusLock.unlock() ) {
            return false;
          }
          return result;

        } // end of function removeRowUnstableSnapShot


        /////////////////////////////////////////////////////////////////////////////////////////////////////
        // Removes a row using stable method. Essentially the order of the rows are kept by moving the block
        // of rows on the page being removed to "plug" the hole left over.
        // If the page contains only one row, then we remove the page and fix the prev and next page pointers.
        /////////////////////////////////////////////////////////////////////////////////////////////////////

        inline bool removeRowStable (R* row, IX& tableIndexes, iterator& beginIt, iterator& endIt) {

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Remove all the indexes for the row being deleted and all rows greater than this one
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          for (R* i = row; i < dataEnd; ++i) {
            tableIndexes.removeIndexes(i);
          }

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // reduce the length of this page, and move the data end pointer up by one.
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          --length;
          --dataEnd;

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Move the block of memory to fill in the hole left -- but only do this if the row is not the last
          // row
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          if (row != dataEnd) {
            size_t numBytes = (dataEnd - row) * sizeof(R);
            memmove(row, (row + 1), numBytes);
          }

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // row now points to the filled in hole, not the original row which was to be removed. Add the
          // indexes
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          for (R* i = row; i < dataEnd; ++i) {
            tableIndexes.addIndexes(i);
          }

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // There are some rows left on this page --essentially modification of this page
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          if (length != 0) {
            // if this page is the last page, then reset the end iterator
            if (nextPage == 0) {
              endIt.resetPointer(length);
            }
            return true;
          }


          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // This page is empty. There any no rows left on the page. This is the only page left in memory,
          // we have to leave the page in memory.
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          if (nextPage == 0 && prevPage == 0) {
            endIt.resetPointer(0);
            return true;
          }

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // This is the last page -- end iterator CHANGES!
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          else if (nextPage == 0 && prevPage != 0) {
            prevPage->setNextPage(0);
            endIt.resetPage(prevPage);
            endIt.resetPointer(prevPage->getLength());
          }

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // The page is empty and this is not the last page nor the first page.
          // begin and end iterators do not change in this case
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          else if (nextPage != 0 && prevPage != 0) {
            prevPage->setNextPage(nextPage);
            nextPage->setPrevPage(prevPage);
          }

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // This is the first page -- begin iterator CHANGES!
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          else if (nextPage != 0 && prevPage == 0) {
            nextPage->setPrevPage(0);
            beginIt.resetPage(nextPage);
            beginIt.resetPointer(0);
          }

          delete[] this->dataBegin;
          delete this;
          return true;
        } // end of function removeRowStable


        inline bool removeRowStableSnapShot (R* row, IX& tableIndexes, iterator& beginIt, iterator& endIt) {
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Lock status can not change
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          if (!_pageSnapShotStatusLock.readLock() ) {
            return false;
          }

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // If there are two or more rows OR this is the only page in memory, essentially a modification
          // is taking place -- no pages are being removed.
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

          if ( (length > 1) || (prevPage == 0 && nextPage == 0) ) {

            // Snap shot of the page has NOT taken place yet.
            if (_pageSnapShotStatus == -1) {
              this->createSnapShotCopy();
            }

            // The snap shot is taking place, and this page IS currently being dumped to file
            else if (_pageSnapShotStatus == 0) {
              // block until the snap shot has finished -- only if we do not have a copy
              // if the snap shot is using a copy, then no need to worry here
              if (!_useCopy) {
                if (!_pageLock.writeLock()) {
                  if (!_pageSnapShotStatusLock.unlock() ) { }
                  return false;
                }
                if (!_pageLock.unlock()) {
                  if (!_pageSnapShotStatusLock.unlock() ) { }
                  return false;
                }
              }
            }

            // This page has already been completed
            else if (_pageSnapShotStatus == 1) {
              // nothing really to do here
            }

            bool result = removeRowStable (row, tableIndexes, beginIt, endIt);
            if (!_pageSnapShotStatusLock.unlock() ) {
              return false;
            }
            return result;
          }

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Ok this page will be removed from memory. Block until the last affected page has been dumped.
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          Page<R, IX>* affectedPage = 0;
          if (nextPage != 0 ) {
            if (nextPage->next() != 0) {
              affectedPage = nextPage->next();
            }
            else {
              affectedPage = nextPage;
            }
          }
          else {
            affectedPage = this;
          }

          if (!affectedPage->setAuxBlockingLock()) {
            if (!_pageSnapShotStatusLock.unlock() ) { }
            return false;
          }

          if (!affectedPage->setAuxBlockingUnlock()) {
            if (!_pageSnapShotStatusLock.unlock() ) { }
            return false;
          }

          bool result = removeRowStable (row, tableIndexes, beginIt, endIt);
          if (!_pageSnapShotStatusLock.unlock() ) {
            return false;
          }
          return result;

        } // end of function removeRowUnstableSnapShot


      private:
        size_t pageSize;
        size_t maxRows;
        size_t length;    // number of rows actually on the page
        size_t capacity;
        size_t number;    // the page number which we are looking at

        double initialFill;

        RowDeletionType rowDeleteMethod; // used to determine how a row is deleted on the page

        R* dataBegin;
        R* dataEnd;

        Page<R, IX>* nextPage;
        Page<R, IX>* prevPage;

        // snapshot variables required

        int64_t _tableSnapShotID;     // non-zero value here indicates that the table is in the process of being snap shot
        int     _pageSnapShotStatus;
        bool    _useCopy;
        R*      _copyBegin;
        R*      _copyEnd;

        triagens::basics::ReadWriteLock _pageSnapShotStatusLock;
        triagens::basics::ReadWriteLock _pageLock;
        triagens::basics::ReadWriteLock _auxBlockingLock;

        Table* _snapShotTable; // only valid while a snap shot is taking place
        bool   _removeInSnapShot; // used for removal of pages when we remove a row
    };


    ////////////////////////////////////////////////////////////////////////////////
    /// @brief Page iterator, used to iterate through a set of linked list pages.
    ////////////////////////////////////////////////////////////////////////////////

    template<typename R, typename IX>
    class PageIterator {

      public:
        typedef PageIterator<R, IX> iterator;
        typedef Page<R, IX> page_t;
        typedef R row_t;

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Attempts to locate a row within a page.
        ////////////////////////////////////////////////////////////////////////////////

        static page_t * findPageRow (iterator const & b, row_t const * mark) {
          page_t* pageBegin = b.getPage();
          page_t* i         = pageBegin;

          for (i = pageBegin; i != NULL; i = i->next()) {
            if (mark <= i->end() && mark >= i->begin()) {
                return i;
            }
          }

          return 0;
        }


        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Uses a binary seach to locate a particular row on a page.
        ////////////////////////////////////////////////////////////////////////////////

        template<typename P>
        static row_t * lookupSorted (iterator const & b,
                                     iterator const & e,
                                     row_t const * mark,
                                     P const & less) {
          page_t * pageBegin = b.getPage();
          page_t * pageLast  = e.getPage();

          for (page_t* i = pageBegin; i != pageLast; ++i) {

            // get the first and last pointer
            row_t* ptrBegin = (i == pageBegin) ? b.getPointer() : i->begin();
            row_t* ptrEnd   = (i == pageLast)  ? e.getPointer() : i->end();

            // find the mark within the range
            row_t* found = std::lower_bound(ptrBegin, ptrEnd, mark, P::isLess);

            if (found != ptrEnd) {
              return found;
            }
          }

          return 0;
        }

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Constructor. Creates empty iterator with no row or page pointer.
        ////////////////////////////////////////////////////////////////////////////////

        PageIterator ()
          : pointer(0), page(0) {
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Constructor. Creates an iterator given a row pointer and page pointer.
        ////////////////////////////////////////////////////////////////////////////////

        PageIterator (row_t * currentData, page_t * currentPage)
          : pointer(currentData), page(currentPage) {
        }

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Adds a data row.
        ////////////////////////////////////////////////////////////////////////////////

        inline row_t* add (R const * row, IX& tableIndexes) {

          if (page->getTableSnapShotID() != -1) {

            if (page->isFull()) {

              //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
              // Place a read lock so we can ensure that _pageSnapShotStatus is correct
              // and that we suspend processing until snap shot for page is done
              //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
              if (!page->setPageStatusReadLock()) {
                return 0;
              }

              page_t* oldPage = page;
              page = page->addPage();

              //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
              // copy the snap status to the new page
              //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
              page->setSnapShotStatus(oldPage->getTableSnapShotID());

              page->checkCapacity(tableIndexes);

              //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
              // unlock the page
              //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
              if (page->setPageStatusUnlock()) {
              }

            }


            pointer = page->addRow(row, tableIndexes);

            return pointer - 1;
          }

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // No snap shots taking place
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          else {
            if (page->isFull()) {
              page = page->addPage();
            }

            page->checkCapacity(tableIndexes);

            pointer = page->addRow(row, tableIndexes);

            return pointer - 1;
          }

        }

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the page associated with the iterator.
        ////////////////////////////////////////////////////////////////////////////////

        inline page_t* getPage () const {
           return page;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the row pointer associated with the iterator.
        ////////////////////////////////////////////////////////////////////////////////

        inline row_t* getPointer () const {
           return pointer;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the last row.
        ////////////////////////////////////////////////////////////////////////////////

        inline row_t* getLast () const {
           return (pointer - 1);
        }


      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Difference operator. Returns the number of rows between \em this
        /// and another.
        ////////////////////////////////////////////////////////////////////////////////

        inline size_t operator- (const PageIterator<R, IX>& right) const {
          size_t s = 0;

          row_t* b = right.getPointer();
          page_t* p = right.getPage();

          while (p != page) {
            s += p->end() - b;

            p = p->next();
            assert(p != 0);

            b = p->begin();
          }

          return s + (pointer - b);
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Increments the iterator by incrementing the row pointer.
        /// The next page (in the linked list) is called if the row pointer goes past
        /// the last row of the page associated with \em this iterator.
        ////////////////////////////////////////////////////////////////////////////////

        inline PageIterator<R, IX>& operator++ () {
          pointer++;

          if (pointer != page->end()) {
            return *this;
          }

          // we are at the undefined row so move onto the next page
          page_t* nPage = page->next();

          // we are at the last page
          if (nPage == 0) {
             return *this;
          }

          // incremented the page and set the pointer to the start of the page
          page    = nPage;
          pointer = page->begin();

          return *this;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns a reference to the current position.
        ////////////////////////////////////////////////////////////////////////////////

        inline row_t& operator* () const {
          return *pointer;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns true if the iterators are equal.
        ////////////////////////////////////////////////////////////////////////////////

        inline bool operator== (const PageIterator<R, IX>& rhs) const {
          // no need to worry about if the pointers are on different pages
          // since we assuming no memory overlap
          return (pointer == rhs.pointer);
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief returns true if the iterators are not equal.
        ////////////////////////////////////////////////////////////////////////////////

        inline bool operator!= (const PageIterator<R, IX>& rhs) const {
          // no need to worry about if the pointers are on different pages
          // since we assuming no memory overlap
          return pointer != rhs.pointer;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Advances the iterator by the integer \em rhs.
        ///
        /// rhs determines the number of rows to add to the iterator. Note: we
        /// assume all pages are of same size. A different + operator would be required
        /// for pages of differing sizes
        ////////////////////////////////////////////////////////////////////////////////

        PageIterator<R, IX> operator+ (size_t rhs) const {
          page_t* iPage = page;

          size_t pageSize  = iPage->getMaxRows();
          size_t remainder = rhs % pageSize;
          size_t pageSkip  = (rhs - remainder) / pageSize;

          // skip the pages in memory
          for (size_t i = 1;  i <= pageSkip;  ++i){

            // check and ensure that we do not go past the end of the pages
            // if we do simply return the last undefined row
            if (iPage->next() == 0) {
              return PageIterator<R, IX>(iPage->end(), iPage);
            }

            iPage = iPage->next();
          }

          // move to the correct position on the page
          size_t rowSkip = (pointer - page->begin()) + remainder;
          row_t* iPtr = iPage->begin() + rowSkip;

          return PageIterator<R, IX>(iPtr, iPage);
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief type conversion
        ////////////////////////////////////////////////////////////////////////////////

        operator row_t* () const {
          return pointer;
        }

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief resets the page associated with the iterator
        ////////////////////////////////////////////////////////////////////////////////

        inline void resetPage (page_t* newPage) {
          page = newPage;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief resets the pointer
        ////////////////////////////////////////////////////////////////////////////////

        inline void resetPointer (size_t off) {
          pointer = page->begin() + off;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief advance the iterator by one
        ////////////////////////////////////////////////////////////////////////////////

        inline bool advanceItr (PageIterator<R, IX> const & e) {
          ++pointer;

          if (pointer < page->end()) {
            if (page == e.getPage()) {
              return pointer < e.getPointer();
            }
            else {
              return true;
            }
          }

          // check and see if there is a next page
          if (page == e.getPage()) {
            return false;
          }

          // incremented the page and set the pointer to the start of the page
          page = page->next();
          pointer = page->begin();

          if (page == e.getPage()) {
            return pointer < e.getPointer();
          }
          else {
            return true;
          }
        }

      private:
        row_t* pointer;
        page_t* page;
    };

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief helper function for iterators
    ////////////////////////////////////////////////////////////////////////////////

    template <typename T, typename IX>
    inline bool advanceIterator (PageIterator<T, IX> & b, PageIterator<T, IX> const & e) {
      return b.advanceItr(e);
    }
  }
}

#endif
