////////////////////////////////////////////////////////////////////////////////
/// @brief High-Performance Database Framework made by triagens
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Copyright 2008-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#ifndef TRIAGENS_HPDF_HPDF_INDIRECT_TABLE_H
#define TRIAGENS_HPDF_HPDF_INDIRECT_TABLE_H 1

#include <Hpdf/Common.h>

#include <Hpdf/TableVisitor.h>
#include <Hpdf/Table.h>

namespace triagens {
  namespace hpdf {
    using namespace std;

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief helper function for iterators
    ////////////////////////////////////////////////////////////////////////////////

    template <typename T>
    inline bool advanceIterator (T* & b, T* const & e) {
      return ++b < e;
    }

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief helper class for indirect tables without additional information
    ////////////////////////////////////////////////////////////////////////////////

    template<typename R, typename F>
    struct NoIndex {
        typedef R row_t;
        typedef F source_row_t;

        NoIndex () : _row(0) {
        }

        inline void setRow (row_t const * row) {
          _row = row;
        }

        inline source_row_t const * operator-> () const {
          return reinterpret_cast<source_row_t const *>(_row);
        }

        row_t const * _row;
    };

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief indirect table
    ///
    /// Constructs a new IndirectTable given a row description R, an index
    /// description X, and an row iterator I. The index description must support
    /// the operator-> to access the elements.  The iterator iterates over
    /// elements of type X.
    ////////////////////////////////////////////////////////////////////////////////

    template<typename R, typename X>
    class IndirectTable : public Table {

      public:
        typedef X * iterator;
        typedef X index_t;
        typedef R row_t;

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief returns the current row from an iterator
        ////////////////////////////////////////////////////////////////////////////////

        static row_t const * row (index_t const * const & i) {
          typename X::row_t const * const * rptr = &((*i)._row);
          return *reinterpret_cast<row_t const * const *>(rptr);
        }

      // -----------------------------------------------------------------------------
      // constructors and destructors
      // -----------------------------------------------------------------------------

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Constructs an empty table.
        ////////////////////////////////////////////////////////////////////////////////

        IndirectTable ()
          : dataBegin(0), dataEnd(0) {
        }

      // -----------------------------------------------------------------------------
      // table methods
      // -----------------------------------------------------------------------------

      public:

        //////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        //////////////////////////////////////////////////////////////////////////////

        size_t size () const {
          return dataEnd - dataBegin;
        }

        //////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        //////////////////////////////////////////////////////////////////////////////

        void forall (TableVisitor * visitor) const {
          size_t pos = 0;

          for (iterator i = dataBegin;  i != dataEnd;  ++i, ++pos) {
            visitor->visit(reinterpret_cast<Row const *>(row(i)), pos);
          }
        }

        //////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        //////////////////////////////////////////////////////////////////////////////

        void forany (TableVisitor * visitor) const {
          size_t pos = 0;

          if (dataBegin != dataEnd) {
            visitor->visit(reinterpret_cast<Row const *>(row(dataBegin)), pos);
          }
        }

      // -----------------------------------------------------------------------------
      // iterators
      // -----------------------------------------------------------------------------

      public:

        //////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        //////////////////////////////////////////////////////////////////////////////

        inline iterator begin () const {
          return dataBegin;
        }

        //////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        //////////////////////////////////////////////////////////////////////////////

        inline iterator end () const {
          return dataEnd;
        }

      // -----------------------------------------------------------------------------
      // public methods
      // -----------------------------------------------------------------------------

      public:

        /////////////////////////////////////////////////////////////////////////////
        /// @brief finds a mark within the sorted table
        /////////////////////////////////////////////////////////////////////////////

        template<typename P>
        row_t const * lookupSorted (row_t const * mark, P const& compare) {
          index_t * found = lower_bound(dataBegin, dataEnd, mark, P::isLess);

          if (found == dataEnd) {
            return 0;
          }

          return row(found);
        }

      protected:
        iterator dataBegin;
        iterator dataEnd;
    };

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief indirect table owning its index space
    ////////////////////////////////////////////////////////////////////////////////

    template< typename R, typename X = NoIndex<R, R> >
    class IndirectNoble : public IndirectTable<R, X> {

      public:
        typedef X * iterator;
        typedef X index_t;
        typedef R row_t;

      ////////////////////////////////////////////////////////////////////////////////
      // constructors and destructors
      ////////////////////////////////////////////////////////////////////////////////

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Constructor. Constructs a new noble.
        ////////////////////////////////////////////////////////////////////////////////

        IndirectNoble (size_t capacity = 100)
          : capacity(capacity), length(0) {
          this->dataBegin = new index_t[capacity];
          this->dataEnd = this->dataBegin;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Destructor. Removes memory allocated for the noble.
        ////////////////////////////////////////////////////////////////////////////////

        ~IndirectNoble () {
          delete[] this->dataBegin;
        }

      // -----------------------------------------------------------------------------
      // table methods
      // -----------------------------------------------------------------------------

      public:

        //////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        //////////////////////////////////////////////////////////////////////////////

        Row const* add (Row const * row) {
          return reinterpret_cast<Row const*>(add(reinterpret_cast<row_t const *>(row)));
        }

        //////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        //////////////////////////////////////////////////////////////////////////////

        void reserve (size_t num) {
          if (length + num > capacity) {
            capacity = size_t(length + num);

            iterator copy = new index_t[capacity];
            memcpy(copy, this->dataBegin, length * sizeof(index_t));

            delete[] this->dataBegin;

            this->dataBegin = copy;
            this->dataEnd   = copy + length;
          }
        }

      // -----------------------------------------------------------------------------
      // public methods
      // -----------------------------------------------------------------------------

      public:

        /////////////////////////////////////////////////////////////////////////////
        /// @brief adds one row of the data type R
        /////////////////////////////////////////////////////////////////////////////

        inline index_t* add (row_t const * row) {
          if (length == capacity) {
            reserve(1 + capacity / 5);
          }

          this->dataEnd->setRow(row);

          index_t* result = this->dataEnd;

          ++this->dataEnd;
          length++;

          return result;
        }

      protected:
        size_t capacity;
        size_t length;
    };

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief indirect table not owning its index space
    ///
    /// The vassal has to use the iterator of its noble.
    ////////////////////////////////////////////////////////////////////////////////

    template<typename R, typename N>
    class IndirectVassal : public IndirectTable<R, typename N::index_t> {

      public:
        typedef typename IndirectTable<R, typename N::index_t>::iterator iterator;
        typedef typename IndirectTable<R, typename N::index_t>::index_t index_t;
        typedef typename IndirectTable<R, typename N::index_t>::row_t operand_row_t;
        typedef R row_t;

      // -----------------------------------------------------------------------------
      // constructors and destructors
      // -----------------------------------------------------------------------------

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief constructs an empty vassal
        ////////////////////////////////////////////////////////////////////////////////

        IndirectVassal () {
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief constructs a new vassal
        ////////////////////////////////////////////////////////////////////////////////

        template<typename S>
        explicit IndirectVassal (S& input) {
          this->dataBegin = input.begin();
          this->dataEnd   = input.end();
        }

      // -----------------------------------------------------------------------------
      // table methods
      // -----------------------------------------------------------------------------

      public:

        //////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        //////////////////////////////////////////////////////////////////////////////

        Row const* add (Row const *) {
          throw "internal error";
        }

        //////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        //////////////////////////////////////////////////////////////////////////////

        void reserve (size_t) {
          throw "internal error";
        }
    };

  }
}

#endif
