////////////////////////////////////////////////////////////////////////////////
/// @brief High-Performance Database Framework made by triagens
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Copyright 2008-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#ifndef TRIAGENS_HPDF_HPDF_DIRECT_TABLE_H
#define TRIAGENS_HPDF_HPDF_DIRECT_TABLE_H 1

#include <Basics/Common.h>
#include <Basics/Logger.h>
#include <Basics/Mutex.h>
#include <Basics/MutexLocker.h>
#include <Basics/ReadWriteLock.h>
#include <Basics/StringUtils.h>

#include <Hpdf/Common.h>
#include <Hpdf/TableVisitor.h>
#include <Hpdf/Table.h>
#include <Hpdf/HpdfPage.h>

namespace triagens {
  namespace hpdf {

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief HashIndex forward declarations
    /// Empty structure used for tables which do not have indices.
    ////////////////////////////////////////////////////////////////////////////////

    template<typename R>
    struct EmptyIndex {

      //////////////////////////////////////////////////////////////////////////////
      /// @brief add indices to indexes defined.
      //////////////////////////////////////////////////////////////////////////////

      inline static void addIndexes (R const*) { // row
      }

      //////////////////////////////////////////////////////////////////////////////
      /// @brief used for recovery purposes. Empty index does not have recovery
      //////////////////////////////////////////////////////////////////////////////

      inline static R* recoveryFindRow(const R*) {
        return 0;
      }


      //////////////////////////////////////////////////////////////////////////////
      /// @brief postModifyIndexes used when we modify a row and table has no indexes
      //////////////////////////////////////////////////////////////////////////////

      inline static void postModifyIndexes (R const*, R const*) { // oldRow, newRow
      }


      //////////////////////////////////////////////////////////////////////////////
      /// @brief preModifyIndexes used when we modify a row and table has no indexes
      //////////////////////////////////////////////////////////////////////////////

      inline static void preModifyIndexes (R const*, R const*) { // oldRow, newRow
      }

      //////////////////////////////////////////////////////////////////////////////
      /// @brief replaces indices.
      //////////////////////////////////////////////////////////////////////////////

      inline static void replaceIndexes (R const*, R const*) { // oldRow, newRow
      }

      //////////////////////////////////////////////////////////////////////////////
      /// @brief removes indices.
      //////////////////////////////////////////////////////////////////////////////

      inline static void removeIndexes (R const*) { // row
      }

    };

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief direct table
    /// Constructs a new DirectTable given a row description R and an row iterator I.
    ////////////////////////////////////////////////////////////////////////////////

    template<typename R, typename IX, typename I>
    class DirectTable : public Table {

      public:
        typedef I iterator;
        typedef R row_t;

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief gets the current row from a const iterator
        ////////////////////////////////////////////////////////////////////////////////

        static inline row_t * row (iterator & i) {
          return reinterpret_cast<row_t*>(&(*i));
        }

      // -----------------------------------------------------------------------------
      // constructors and destructors
      // -----------------------------------------------------------------------------

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Constructor. Constructs an empty table.
        ////////////////////////////////////////////////////////////////////////////////

        DirectTable () {
        }

      // -----------------------------------------------------------------------------
      // table methods
      // -----------------------------------------------------------------------------

      public:

        //////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        //////////////////////////////////////////////////////////////////////////////

        size_t size () const {
          return dataEnd - dataBegin;
        }

        //////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        /// Perform a visit for each data row which essentially outputs the
        /// contents of the row. Note that the default copy
        /// constructor is called here for the iterator.
        //////////////////////////////////////////////////////////////////////////////

        void forall (TableVisitor * visitor) const {
          size_t pos = 0;

          for (iterator i = dataBegin;  i != dataEnd;  ++i, ++pos) {
            visitor->visit(reinterpret_cast<Row*>(&(*i)), pos);
          }
        }

        //////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        /// Perform a visit for the first data row which essentially outputs the
        /// contents of the row.
        //////////////////////////////////////////////////////////////////////////////

        void forany (TableVisitor * visitor) const {
          size_t pos = 0;

          if (dataBegin != dataEnd) {
            visitor->visit(reinterpret_cast<Row*>(&(*dataBegin)), pos);
          }
        }

      // -----------------------------------------------------------------------------
      // iterators
      // -----------------------------------------------------------------------------

      public:

        //////////////////////////////////////////////////////////////////////////////
        /// @brief beginning of data
        ///
        /// Essentially returns the first page and the first data pointer on that page.
        //////////////////////////////////////////////////////////////////////////////

        inline iterator begin () const {
          return dataBegin;
        }

        /////////////////////////////////////////////////////////////////////////////
        /// @brief end of data
        ///
        /// Essentially returns the last page and the last data pointer on that page.
        /////////////////////////////////////////////////////////////////////////////

        inline iterator end () const {
          return dataEnd;
        }

      // -----------------------------------------------------------------------------
      // public methods
      // -----------------------------------------------------------------------------

      public:

        /////////////////////////////////////////////////////////////////////////////
        /// @brief finds a mark within the table
        ///
        /// Uses the lower_bound routine from the std library to perform
        /// a binary search. This methods assumes that the iterator is a PageIterator.
        /////////////////////////////////////////////////////////////////////////////

        template<typename P>
        inline row_t const * lookupSorted (row_t const * mark, P const & less) {
          return I::lookupSorted(dataBegin, dataEnd, mark, less);
        }

      protected:
        iterator dataBegin;
        iterator dataEnd;
    };

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief direct table owning its address space
    ///
    /// Creates a direct table with a PageIterator.
    //  See below (Primary Table) for a discussion of the template arguments
    ////////////////////////////////////////////////////////////////////////////////

    template<typename R, typename MODIFY_DELETE, typename IX = EmptyIndex<R> >
    class DirectNoble : public DirectTable< R, IX, PageIterator<R,IX> > {

      public:
        typedef PageIterator<R, IX> iterator;
        typedef R row_t;

      // -----------------------------------------------------------------------------
      // constructors and destructors
      // -----------------------------------------------------------------------------

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Constructor. Constructs a new noble table.
        ////////////////////////////////////////////////////////////////////////////////

        DirectNoble (size_t psz = 1024, double initialFill = 1.0,  RowDeletionType rowDelMethod = RD_STANDARD) : tableIndexes() {
          typedef typename iterator::page_t page_t;
          size_t const MINIMAL_ROWS_PER_PAGE = 100;

          // calculate pagesize
          size_t rpsz = realPageSize(MINIMAL_ROWS_PER_PAGE, psz);

          // create an empty page in anticpation of being fillled
          page_t * page = new page_t(rpsz, 0, initialFill, rowDelMethod);

          // and check the capacity of the page and allocates memory for the page
          page->checkCapacity(tableIndexes);

          // update the iterators
          this->dataBegin = iterator(page->begin(), page);
          this->dataEnd   = iterator(page->begin(), page);
        }


        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Destructor. Destroys a noble table and any allocated pages and rows
        /// within the pages.
        ////////////////////////////////////////////////////////////////////////////////

        ~DirectNoble () {
          typedef typename iterator::page_t page_t;

          page_t* ptr = this->dataEnd.getPage();
          page_t* prv = 0;

          // go through all of your pages and delete them AND delete the data which is pointed to
          while (ptr != 0) {
            prv = ptr->prev();

            ptr->deleteData();
            delete ptr;

            ptr = prv;
          }
        }

      // -----------------------------------------------------------------------------
      // table methods
      // -----------------------------------------------------------------------------

      public:

        //////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        //////////////////////////////////////////////////////////////////////////////

        Row const* add (Row const * row) {
          return reinterpret_cast<Row const*>(add(reinterpret_cast<row_t const *>(row)));
        }

        //////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        //////////////////////////////////////////////////////////////////////////////

        void reserve (size_t) {
          // TODO don't worry about this now -- used for bulk loading
        }

      // -----------------------------------------------------------------------------
      // public methods
      // -----------------------------------------------------------------------------

      public:

        /////////////////////////////////////////////////////////////////////////////
        /// @brief adds one row of the data type R
        /////////////////////////////////////////////////////////////////////////////

        inline row_t* add (row_t const* row) {

          row_t* result = 0;

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // If this table is part of a snap shot, then we need to lock the add
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          if (this->_allowSnapShot) {

            // Add a read lock - but only if this table is to take part in a snap shot
            // Essentially all transactions are stopped until we have finished with
            // setting each page read to take a snap shot.
            if (! this->_recovery->snapShotLock->readLock() ) {
              return 0;
            }

            //---------------------------------------------------------------------------------
            // Since recovery is defined for this table we must set a row id for playback purposes
            //---------------------------------------------------------------------------------
            this->_recoveryRowId++;
            MODIFY_DELETE::setRecoveryRowId(const_cast<row_t*>(row),this->_recoveryRowId);
            result = this->dataEnd.add(row, tableIndexes);

            // remove a read lock - but only if this table is to take part in a snap shot
            if (! this->_recovery->snapShotLock->unlock() ) {
            }

            //---------------------------------------------------------------------------------
            // Since recovery is defined for this table we must log the transaction
            //---------------------------------------------------------------------------------
            if (! tableLogTransaction (row, Table::TRANSACTION_CREATE) ) {
              return 0;
            }
          }

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // This table is not part of a snap shot, don't waste time with locks
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          else {
            result = this->dataEnd.add(row, tableIndexes);
          }

          if (this->dataBegin.getPage() == this->dataEnd.getPage()) {
            this->dataBegin.resetPointer(0);
          }

          return result;
        }

        /////////////////////////////////////////////////////////////////////////////
        /// @brief adds one row of the data stored as text in a character array
        // USE only recovery purposes (transaciton playback)
        /////////////////////////////////////////////////////////////////////////////

        inline bool addRecovery (const char* data, const size_t dataLength) {
          row_t row;

          if (! MODIFY_DELETE::restoreTableRow(&row, data, dataLength)) {
            return false;
          }

          this->dataEnd.add(&row, tableIndexes);

          if (this->dataBegin.getPage() == this->dataEnd.getPage()) {
            this->dataBegin.resetPointer(0);
          }

          return true;
        }



        /////////////////////////////////////////////////////////////////////////////
        /// @brief replaces oldRow with newRow. Success returns the newRow which
        /// should be the same memory allocated to oldRow. Failure returns null.
        /////////////////////////////////////////////////////////////////////////////

        inline row_t* modify (row_t const* oldRow, row_t const* newRow) {
          typedef typename iterator::page_t page_t;
          row_t* oRow = const_cast<row_t*>(oldRow);
          row_t* nRow = const_cast<row_t*>(newRow);
          row_t* result = 0;

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Attempt to locate the page which the oldRow belongs to. The starting
          // iterator and the oldRow is used to locate the page.
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          page_t* page = PageIterator<R,IX>::findPageRow(this->dataBegin, oldRow);

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // For some reason the oldRow could not be located on a page. Return NULL
          // to indicate to the calling function that modification failed.
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          if (page == NULL) {
            return 0;
          }

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // If this table is part of a snap shot, then we need to do some locking
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

          if (this->_allowSnapShot) {

            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            // Add a read lock - but only if this table is to take part in a snap shot
            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            if (! this->_recovery->snapShotLock->readLock() ) {
              return 0;
            }

            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            // The page object requires oldRow, newRow, any table indexes defined and
            // the start/end page iterators. It will attempt to replace the oldRow
            // with the newRow. If changes to indexes are required, these will be
            // automatically made.
            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

            // first apply any methods which are required prior to being modified -- currently does nothing
            MODIFY_DELETE::preModify(oRow,nRow);

            // apply the modification to the page
            result = page->modifyRow(oRow, nRow, tableIndexes, this->dataBegin, this->dataEnd);

            // finally apply any methods which are required post modification - removes any allocated
            // memory for strings and blobs -- BUT only if these have been modified.
            MODIFY_DELETE::postModify(oRow,nRow);


            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            // remove a read lock - but only if this table is to take part in a snap shot
            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            if (! this->_recovery->snapShotLock->unlock() ) {
            }

            //---------------------------------------------------------------------------------
            // Since recovery is defined for this table we must log the transaction
            //---------------------------------------------------------------------------------
            if (! tableLogTransaction (nRow, Table::TRANSACTION_MODIFY) ) {
              return 0;
            }
          }

          else {
            // first apply any methods which are required prior to being modified -- currently does nothing
            MODIFY_DELETE::preModify(oRow,nRow);

            // apply the modification to the page
            result = page->modifyRow(oRow, nRow, tableIndexes, this->dataBegin, this->dataEnd);

            // finally apply any methods which are required post modification - removes any allocated
            // memory for strings and blobs -- BUT only if these have been modified.
            MODIFY_DELETE::postModify(oRow,nRow);
          }

          return result;
        }


        /////////////////////////////////////////////////////////////////////////////
        /// @brief modifies a row of the data stored as text in a character array
        // USE only recovery purposes (transaciton playback)
        /////////////////////////////////////////////////////////////////////////////

        inline bool modRecovery (const char* data, const size_t dataLength) {
          typedef typename iterator::page_t page_t;

          row_t newRow;

          if (! MODIFY_DELETE::restoreTableRow(&newRow, data, dataLength)) {
            return false;
          }

          // we have a row -- now attempt to locate the row using the hidden hash index
          row_t* oldRow = tableIndexes.recoveryFindRow(&newRow);
          if (oldRow == 0) { // indicates that the row could not be located using a hash index
            return false;
          }

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // now modify the old row
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

          // Locate the page where the existing row resides on.
          page_t* page = PageIterator<R,IX>::findPageRow(this->dataBegin, oldRow);

          // If we can not locate the page, then some sort of internal error occurred,
          // report this back to the calling function.
          if (page == NULL) {
            return false;
          }

          // first apply any methods which are required prior to being modified -- currently does nothing
          MODIFY_DELETE::preModify(oldRow,&newRow);

          // apply the modification to the page
          row_t* result = page->modifyRow(oldRow, &newRow, tableIndexes, this->dataBegin, this->dataEnd);

          // finally apply any methods which are required post modification - removes any allocated
          // memory for strings and blobs -- BUT only if these have been modified.
          MODIFY_DELETE::postModify(oldRow,&newRow);

          return (result == oldRow);
        }


        /////////////////////////////////////////////////////////////////////////////
        /// @brief removes one row (with delete class D)
        /////////////////////////////////////////////////////////////////////////////

        inline bool remove (row_t* row, bool deleteFlag = false) {
          typedef typename iterator::page_t page_t;

          bool result = false;
          row_t* row_copy = 0;

          // The function findPageRow() locates the page using the begin/end iterators
          // where the row resides.
          page_t* page = PageIterator<R,IX>::findPageRow(this->dataBegin, row);

          // check and see that we located the row on some page.
          if (page == NULL) {
            return false;
          }

          if (deleteFlag) {
            // save row to row_copy for freeing string values later
            row_copy = reinterpret_cast<row_t*>(new char[sizeof(row_t)]);
            memcpy(row_copy, row, sizeof(row_t));
          }

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // If this table is part of snap shot, then we need some locks
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

          if (this->_allowSnapShot) {

            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            // Add a read lock - but only if this table is to take part in a snap shot
            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            if (! this->_recovery->snapShotLock->readLock() ) {
              // lock failed
              if (row_copy != 0) {
                delete[] (char*) row_copy;
              }
              return 0;
            }

            // row is located on the page "page", remove it from the page (including any indexes the table uses)
            result = page->removeRow(row, tableIndexes, this->dataBegin, this->dataEnd);

            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            // remove a read lock - but only if this table is to take part in a snap shot
            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            if (! this->_recovery->snapShotLock->unlock() ) {
            }

            //---------------------------------------------------------------------------------
            // Since recovery is defined for this table we must log the transaction
            //---------------------------------------------------------------------------------
            if (! tableLogTransaction (row_copy, Table::TRANSACTION_DELETE) ) {
              if (row_copy != 0) {
                delete[] (char*) row_copy;
              }
              return 0;
            }

          }

          else {
            // row is located on the page "page", remove it from the page (including any indexes the table uses)
            result = page->removeRow(row, tableIndexes, this->dataBegin, this->dataEnd);
          }

          if (deleteFlag) {
            // free string values of removed row if the row was successfully removed
            if (result) {
              MODIFY_DELETE::destroy(row_copy);
            }
            delete[] (char*) row_copy;
          }

          return (result);
        }


        /////////////////////////////////////////////////////////////////////////////
        /// @brief removes a row of the data stored as text in a character array
        // USE only recovery purposes (transaciton playback)
        /////////////////////////////////////////////////////////////////////////////

        inline bool remRecovery (const char* data, const size_t dataLength) {
          typedef typename iterator::page_t page_t;

          row_t newRow;

          if (! MODIFY_DELETE::restoreTableRow(&newRow, data, dataLength)) {
            return false;
          }

          // we have a row -- now attempt to locate the row using the hidden hash index
          row_t* oldRow = tableIndexes.recoveryFindRow(&newRow);
          if (oldRow == 0) { // indicates that the row could not be located using a hash index
            return false;
          }

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // now remove the old row
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

          // Locate the page where the existing row resides on.
          page_t* page = PageIterator<R,IX>::findPageRow(this->dataBegin, oldRow);

          // If we can not locate the page, then some sort of internal error occurred,
          // report this back to the calling function.
          if (page == NULL) {
            return false;
          }

          // We need to remove any old memory associated with blobs, strings etc. Save the pointers of these
          // data structures here and then remove them.
          row_t* row_copy = new row_t;
          memcpy(row_copy, oldRow, sizeof(row_t));

          // Use the page method to remove it from the page (including any indexes the table uses)
          // begin and end page iterators required since the structure of the page linked list
          // may change
          bool result = page->removeRow(oldRow, tableIndexes, this->dataBegin, this->dataEnd);

          // free memory associated with blobs, strings  etc
          MODIFY_DELETE::destroy(row_copy);

          // free memory associated with the copy of the row to be deleted
          delete row_copy;

          return result;
        }


        /////////////////////////////////////////////////////////////////////////////
        /// @brief Makes the table permanent, where the table has been allocated on the stack.
        /////////////////////////////////////////////////////////////////////////////

        DirectNoble<R, MODIFY_DELETE, IX>* swap () {

          // here the table row deletion method will always default to standard
          DirectNoble<R, MODIFY_DELETE, IX>* dst = new DirectNoble(this->dataBegin.getPage()->getPageSize(), 0.0);

          // save the new iterators (we need to delete them eventually!)
          iterator tempBegin = dst->dataBegin;
          iterator tempEnd   = dst->dataEnd;

          // iterators and data now have to point to the new table
          dst->dataBegin = this->dataBegin;
          dst->dataEnd   = this->dataEnd;


          // clear the old table
          this->dataBegin = tempBegin;
          this->dataEnd   = tempEnd;


          return dst;
        }

      private:

        // TODO how should MINIMAL_ROWS_PER_PAGE, initialFill and pageSize work?

        /////////////////////////////////////////////////////////////////////////////
        /// @brief Calculates the size of page which stored the data rows.
        /////////////////////////////////////////////////////////////////////////////

        size_t realPageSize (size_t rows, size_t psz) {

          // get the pagesize of the machine
          size_t psz3 = ::getpagesize();

          // if R is too big for one page, adjust the page size
          if (psz3 < sizeof(R)) {
            psz3 = ((sizeof(R) + psz3 - 1) / psz3) * psz3;
          }

          size_t psz2 = ((psz + psz3 - 1) / psz3) * psz3;
          size_t mpsz = ((rows * sizeof(R) + psz3 - 1) / psz3) * psz3;

          return max(max(mpsz, psz2), psz3);
        }

      public:

        size_t getRowSize () const {
          return sizeof(row_t);
        }

        void dumpTableRow(void const* row, char*& data, size_t& dataLength) {
          MODIFY_DELETE::dumpTableRow(reinterpret_cast<const row_t*>(row), data, dataLength);
        }

        void dumpTableRow(const void* sRow, const void* eRow) {

          const R* startRow = reinterpret_cast<const R*>(sRow);
          const R* endRow   = reinterpret_cast<const R*>(eRow);

          for (const R* j = startRow; j < endRow; ++j) {
            char* data = 0;
            size_t dataLength = 0;
            this->dumpTableRow(j, data, dataLength);

            struct timeval tv;
            gettimeofday(&tv, 0);
            datetime_t transTime = tv.tv_sec + (tv.tv_usec / 1000000.0);

            string dataStr =
              basics::StringUtils::itoa(this->_recovery->snapShotNumber)    + ";"  +  // the last snap shot taken before this transaction was logged
              basics::StringUtils::ftoa(this->_recovery->snapShotDateTime)  + ";"  +  // the actual start date time of the snapshot
              basics::StringUtils::itoa(this->_recoveryTableId)             + ";"  +  // unique id of this table (crc32 hash)
              basics::StringUtils::ftoa(transTime)                           + ";";    // actual time for this row

            this->_recovery->snapShot->write(dataStr.c_str(),dataStr.size());
            this->_recovery->snapShot->write(data,dataLength);
            if (data != 0) {
              delete [] data;
              data = 0;
            }
          }
        }

        bool restoreTableRow (void* row, const char* data, const size_t dataLength) {
          return MODIFY_DELETE::restoreTableRow(reinterpret_cast<row_t*>(row), data, dataLength);
        }


        bool tableSnapShotBegin (const size_t snapShotNumber) {
          typedef typename iterator::page_t page_t;

          bool ok = true;
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Define the start page
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          page_t* startPage = this->dataBegin.getPage();

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Go through all the pages and mark them as being in a snap shot. Perform
          // any initialisations as required for the page.
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          startPage->setSnapShotStatus(this,snapShotNumber,ok);

          return ok;
        }


        bool tableSnapShotDo () {
          typedef typename iterator::page_t page_t;

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Define the start page
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          page_t* startPage = this->dataBegin.getPage();

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Iterate of the pages. We do not know the end page yet since this may
          // change while we are snapshooting.
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          bool ok = true;
          startPage->pageSnapShotDo(ok);
          return ok;
        }

        bool tableSnapShotEnd () {
          typedef typename iterator::page_t page_t;

          bool ok = true;
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Define the start page
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          page_t* startPage = this->dataBegin.getPage();

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // go through all the pages and mark them as NOT being in a snap shot.
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          startPage->setSnapShotStatus(this,-1, ok);

          return ok;
        }

      private:

        bool tableLogTransaction (const void* row, Table::TransactionType transType) {

          char* data = 0;
          size_t dataLength = 0;
          this->dumpTableRow(row, data, dataLength);

          if (data == 0 || dataLength == 0) {
            return false;
          }

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // attempt to Lock the transaction file
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          if (! this->_recovery->logFileLock->writeLock()) {
            delete data;
            return false;
          }

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // get the current time -- which will form part of the transaction log
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          struct timeval tv;
          gettimeofday(&tv, 0);
          datetime_t transTime = tv.tv_sec + (tv.tv_usec / 1000000.0);


          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // Add a few fields in front of the row which will be required for playback
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          string dataStr = basics::StringUtils::itoa(this->_recovery->snapShotNumber)    + ";"  +  // the last snap shot taken before this transaction was logged
                           basics::StringUtils::itoa(this->_recoveryTableId) + ";"  +  // unique id of this table (crc32 hash)
                           basics::StringUtils::itoa(transType)              + ";"  +  // type of transaction -- deletion
                           basics::StringUtils::ftoa(transTime)              + ";";    // date, time that this transaction was logged

          this->_recovery->logFile->write(dataStr.c_str(),dataStr.size());
          this->_recovery->logFile->write(data,dataLength);
          this->_recovery->logFile->flush();
          delete data;

          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          // attempt to unlock the transaction file
          //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          if (! this->_recovery->logFileLock->unlock()) {
            return false;
          }

          return true;
        }

      private:

      public:

        /////////////////////////////////////////////////////////////////////////////
        /// @brief Stores any associated indexes for the table.
        /////////////////////////////////////////////////////////////////////////////

        IX tableIndexes;

    };

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief PrimaryTable
    //  template arguments: R = row type, MODIFY_DELETE = structure which contains methods to
    //  handle modifications and removal of rows, IX a structure which contains methods to handle
    //  addition, removal, modifications of indexes for the table. When a table has
    //  NO indexes, then it defaults to the EmptyIndex<R> structure found at the top
    //  of this file.
    ////////////////////////////////////////////////////////////////////////////////

    template<typename R, typename MODIFY_DELETE, typename IX = EmptyIndex<R> >
    class PrimaryTable : public DirectNoble<R, MODIFY_DELETE, IX> {

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Constructor. Constructs a primary table
        ////////////////////////////////////////////////////////////////////////////////

        PrimaryTable (size_t pagesize = 1024, double initialFill = 1.0, RowDeletionType rowDeleteMethod = RD_STANDARD)
          : DirectNoble<R, MODIFY_DELETE, IX>(pagesize, initialFill, rowDeleteMethod) {

          // the HpdfCompiler determines whether or not this table is part of data recovery -set it here
          setAllowSnapShot(_modifyDelete.allowSnapShot);

          // the HpdfCompiler determines whether or not we the recovery is stored in text or binary
          setRecoveryFileType(_modifyDelete.recoveryType);

        }

        bool setRecoveryStructure(Table::RecoveryStructure* trans) {
          this->_recovery = trans;
          return true;
        }

        bool setRecoveryTableId(uint32_t id) {
          this->_recoveryTableId = id;
          return true;
        }

      private:

        ////////////////////////////////////////////////////////////////////////////////
        // For a primary table we create a class MODIFY_DELETE which gives us access
        // to various flags and methods which are produced by the compiler. Could make
        // these static, but not.
        ////////////////////////////////////////////////////////////////////////////////

        MODIFY_DELETE _modifyDelete;
    };



    ////////////////////////////////////////////////////////////////////////////////
    /// @brief direct table not owning its address space
    ///
    /// The vassal has to use the iterator of its noble.
    ////////////////////////////////////////////////////////////////////////////////

    template<typename R, typename N, typename IX = EmptyIndex<R> >
    class DirectVassal : public DirectTable<R, IX, typename N::iterator> {

      public:
        typedef typename DirectTable<R, IX, typename N::iterator>::iterator iterator;
        typedef typename DirectTable<R, IX, typename N::iterator>::row_t operand_row_t;
        typedef R row_t;

      // -----------------------------------------------------------------------------
      // constructors and destructors
      // -----------------------------------------------------------------------------

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Constructor. Constructs an empty vassal direct table.
        ////////////////////////////////////////////////////////////////////////////////

        DirectVassal () {
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief constructs a new vassal with given iterators.
        ////////////////////////////////////////////////////////////////////////////////

        template<typename S>
        explicit DirectVassal (S& input) {
          this->dataBegin = input.begin();
          this->dataEnd   = input.end();
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Destrcutor. Destroys a vassal \b without deallocating memory.
        ////////////////////////////////////////////////////////////////////////////////

        ~DirectVassal() {
          // the pages do not belong to us, so delete nothing
        }

      // -----------------------------------------------------------------------------
      // table methods
      // -----------------------------------------------------------------------------

      public:

        //////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        //////////////////////////////////////////////////////////////////////////////

        Row const* add (Row const *) {
          throw "internal error";
        }

        //////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        //////////////////////////////////////////////////////////////////////////////

        void reserve (size_t) {
          throw "internal error";
        }
    };



    ////////////////////////////////////////////////////////////////////////////////
    /// @brief Structure used to store non-unique results obtained from a hashindex search.
    ////////////////////////////////////////////////////////////////////////////////

    template < typename R>
    struct IX_Non_Unique_HashElement {
      vector< R*>* data;  // can not make this data element const since we add elements to it
    };

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief Structure used to store a unique result obtained from a hashindex search.
    ////////////////////////////////////////////////////////////////////////////////

    template < typename R>
    struct IX_Unique_HashElement {
      R* data;
    };


  }
}

#endif
