////////////////////////////////////////////////////////////////////////////////
/// @brief High-Performance Database Framework made by triagens
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Dr. Oreste Costa-Panaia
/// @author Copyright 2008-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#ifndef TRIAGENS_HPDF_HPDF_BINARY_TREE_NODE_H
#define TRIAGENS_HPDF_HPDF_BINARY_TREE_NODE_H 1

#include <Hpdf/Common.h>

namespace triagens {
  namespace hpdf {

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief logical type
    ///
    /// Note that the numbers must match the values FilterParserGrammar.
    ////////////////////////////////////////////////////////////////////////////////

    enum LogicalType {
      unknownID             = 0,
      andKeywordID          = 10,
      equalKeywordID        = 20,
      greaterKeywordID      = 30,
      greaterEqualKeywordID = 40,
      lessKeywordID         = 50,
      lessEqualKeywordID    = 60,
      notEqualKeywordID     = 70,
      notKeywordID          = 80,
      orKeywordID           = 90,
      logicalOperatorID     = 100,
      integerConstantID     = 1010,
      doubleConstantID      = 1020,
      stringConstantID      = 1040,
      identifierID          = 2010
    };

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief HpdfBinaryTreeNodeObject
    ///
    /// The actual object which represents a node for our internal tree. The internal
    /// is used for dynamic filtering.
    ////////////////////////////////////////////////////////////////////////////////

    class HpdfBinaryTreeNodeObject : boost::noncopyable {
      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief constructor
        ////////////////////////////////////////////////////////////////////////////////

        HpdfBinaryTreeNodeObject ()
          : dataOffset(0),
            dataSize(0),
            dataType(AT_UNKNOWN),
            dataConstants(0),
            logicalValue(false),
            logicalOperator(unknownID)  {
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief destructor
        ///
        /// The only memory we could have possibly allocated is for the data constants
        /// entered by the user. Clear these.
        ////////////////////////////////////////////////////////////////////////////////

        ~HpdfBinaryTreeNodeObject () {
          if (dataConstants != 0) {
            delete[] dataConstants;
          }
        }

      // -----------------------------------------------------------------------------
      // make the data within this object public so we can get to it quickly!
      // -----------------------------------------------------------------------------

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief stores the offset
        ///
        /// dataOffset stores the relative address of the data element within the row
        /// A value of 0 indicates that the data element is a constant entered by the
        /// user.
        ////////////////////////////////////////////////////////////////////////////////

        size_t dataOffset;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief stores the size
        ///
        /// dataSize stores the size of the data element.
        ////////////////////////////////////////////////////////////////////////////////

        size_t dataSize;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief stores the type
        ///
        /// The type of the data required for comparisons. valid types are: int,
        /// double, float, long, char (used for string storage) & their associated
        /// pointer types e.g. a string type stored within the row is referred to as
        /// attrib type = AT_STRING, a string type stored as a pointer is referred to
        /// as attrib type = AT_STRING_POINTER;
        ////////////////////////////////////////////////////////////////////////////////

        TypeAttribute dataType;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief stores the constants
        ///
        /// dataConstants allows the user to use constants in the filter expression
        /// e.g. a == 10, a is stored as an integer type with 4 bytes etc, 10 is
        /// stored here in dataConstants as an integer. When dataOffset == 0 then we
        /// expect something in dataConstants otherwise something is wrong.
        ////////////////////////////////////////////////////////////////////////////////

        char* dataConstants;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief stores the logical value
        ///
        /// the logical value of the node either true or false
        ////////////////////////////////////////////////////////////////////////////////

        bool logicalValue;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief stores the operator id
        ///
        /// the logical operator (if any) used for comparison. Valid operators are
        /// ==, !=, <, <=, >, >=, not
        ////////////////////////////////////////////////////////////////////////////////

        LogicalType logicalOperator;
    };

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief HpdfBinaryTreeNode
    ///
    /// The simple tree structure based on the node objects defined above.
    ////////////////////////////////////////////////////////////////////////////////

    class HpdfBinaryTreeNode : boost::noncopyable {

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Constructor
        ////////////////////////////////////////////////////////////////////////////////

        explicit
        HpdfBinaryTreeNode (HpdfBinaryTreeNode* pNode)
          : parentNode(pNode), leftChildNode(0), rightChildNode(0) {
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Destructor
        ///
        /// Deletes the child nodes associated with this node.
        ////////////////////////////////////////////////////////////////////////////////

        ~HpdfBinaryTreeNode(){

          // we have to delete the whole tree starting at the leaves
          if (leftChildNode != 0) {
            delete leftChildNode;
          }

          if (rightChildNode != 0) {
            delete rightChildNode;
          }

          // now delete yourself which should implicitly call the nodeObject
          // destructor for currentNode
        }

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief the data
        ////////////////////////////////////////////////////////////////////////////////

        HpdfBinaryTreeNodeObject currentNode;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief the parent
        ////////////////////////////////////////////////////////////////////////////////

        HpdfBinaryTreeNode* parentNode;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief sets the left child
        ////////////////////////////////////////////////////////////////////////////////

        HpdfBinaryTreeNode* leftChildNode;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief sets the right child
        ////////////////////////////////////////////////////////////////////////////////

         HpdfBinaryTreeNode* rightChildNode;
    };
  }
}

#endif
