////////////////////////////////////////////////////////////////////////////////
/// @brief High-Performance Database Framework made by triagens
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Dr. Oreste Costa-Panaia
/// @author Copyright 2008-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "DynamicSort.h"

#include <boost/version.hpp>

#if BOOST_VERSION >= 103900

#include <boost/spirit/include/classic.hpp>
#include <boost/spirit/include/classic_ast.hpp>

using namespace boost::spirit::classic;

#else

#include <boost/spirit/core.hpp>
#include <boost/spirit/tree/ast.hpp>

using namespace boost::spirit;

#endif

#include "Basics/Exceptions.h"
#include "Basics/StringUtils.h"
#include "Hpdf/AttributeDescriptor.h"
#include "Hpdf/SortedCondition.h"

using namespace triagens::basics;

namespace {

  ////////////////////////////////////////////////////////////////////////////////
  /// @brief DynamicSortProfile
  ////////////////////////////////////////////////////////////////////////////////

  struct DynamicSortProfile {
    int  dataType;
    int  dataOffset;
    int  ascending;
    int  language;
    int  encoding;
  };

  ////////////////////////////////////////////////////////////////////////////////
  /// @brief dynamic is less
  ////////////////////////////////////////////////////////////////////////////////

  template<typename A>
  inline bool dlIsLess (char* pl, char* pr, bool& asc) {
    A l = * reinterpret_cast<A*>(pl);
    A r = * reinterpret_cast<A*>(pr);

    if (l < r) {
      return true;
    }
    else if (l > r) {
      asc = !asc;
      return true;
    }

    return false;
  }

  ////////////////////////////////////////////////////////////////////////////////
  /// @brief dynamic compare
  ////////////////////////////////////////////////////////////////////////////////

  template<typename A>
  inline int dlCompare (char const* pl, char const* pr) {
    A l = * reinterpret_cast<A const*>(pl);
    A r = * reinterpret_cast<A const*>(pr);

    if (l < r) {
      return -1;
    }
    else if (l > r) {
      return 1;
    }

    return 0;
  }
}

namespace triagens {
  namespace hpdf {

    // -----------------------------------------------------------------------------
    // constants
    // -----------------------------------------------------------------------------


    // -----------------------------------------------------------------------------
    // static public methods
    // -----------------------------------------------------------------------------

    DynamicSort::DynamicSortDescription DynamicSort::lessDescriptor (const string& description, AttributeDescriptor const * desc) {

      // parse definition string
      vector<string> v;

      rule<> id_p = (range_p('A','Z') | range_p('a', 'z'))
                 >> *(range_p('A','Z') | range_p('a', 'z') | range_p('0', '9') | ch_p('_'));

      rule<> desc_p =
          longest_d[ id_p
                   | (as_lower_d["asc"]  >> '(' >> id_p >> ')')   // default language & encoding

                   | (as_lower_d["asc"]  >> '(' >> id_p >> ',' >> as_lower_d["enau"] >> ')') // english australian default encoding
                   | (as_lower_d["asc"]  >> '(' >> id_p >> ',' >> as_lower_d["dede"] >> ')') // german german default encoding
                   | (as_lower_d["asc"]  >> '(' >> id_p >> ',' >> as_lower_d["itit"] >> ')') // italian italy default encoding

                   | (as_lower_d["asc"] >> '(' >> id_p >> ',' >> as_lower_d["enau"] >> ',' >> as_lower_d["ISO88591"] >> ')')
                   | (as_lower_d["asc"] >> '(' >> id_p >> ',' >> as_lower_d["dede"] >> ',' >> as_lower_d["ISO88591"] >> ')')
                   | (as_lower_d["asc"] >> '(' >> id_p >> ',' >> as_lower_d["itit"] >> ',' >> as_lower_d["ISO88591"] >> ')')

                   | (as_lower_d["asc"]  >> '(' >> id_p >> ',' >> as_lower_d["enau"] >> ',' >> as_lower_d["utf8"] >> ')')
                   | (as_lower_d["asc"]  >> '(' >> id_p >> ',' >> as_lower_d["dede"] >> ',' >> as_lower_d["utf8"] >> ')')
                   | (as_lower_d["asc"]  >> '(' >> id_p >> ',' >> as_lower_d["itit"] >> ',' >> as_lower_d["utf8"] >> ')')

                   // descending

                   | (as_lower_d["desc"]  >> '(' >> id_p >> ')')   // default language & encoding

                   | (as_lower_d["desc"]  >> '(' >> id_p >> ',' >> as_lower_d["enau"] >> ')') // english australian default encoding
                   | (as_lower_d["desc"]  >> '(' >> id_p >> ',' >> as_lower_d["dede"] >> ')') // german german default encoding
                   | (as_lower_d["desc"]  >> '(' >> id_p >> ',' >> as_lower_d["itit"] >> ')') // italian italy default encoding

                   | (as_lower_d["desc"] >> '(' >> id_p >> ',' >> as_lower_d["enau"] >> ',' >> as_lower_d["ISO88591"] >> ')')
                   | (as_lower_d["desc"] >> '(' >> id_p >> ',' >> as_lower_d["dede"] >> ',' >> as_lower_d["ISO88591"] >> ')')
                   | (as_lower_d["desc"] >> '(' >> id_p >> ',' >> as_lower_d["itit"] >> ',' >> as_lower_d["ISO88591"] >> ')')

                   | (as_lower_d["desc"]  >> '(' >> id_p >> ',' >> as_lower_d["enau"] >> ',' >> as_lower_d["utf8"] >> ')')
                   | (as_lower_d["desc"]  >> '(' >> id_p >> ',' >> as_lower_d["dede"] >> ',' >> as_lower_d["utf8"] >> ')')
                   | (as_lower_d["desc"]  >> '(' >> id_p >> ',' >> as_lower_d["itit"] >> ',' >> as_lower_d["utf8"] >> ')')

                   ][push_back_a(v)];

      string rdesc = StringUtils::replace(description,"-","");

#if BOOST_VERSION < 103500
      bool ok = parse(rdesc.c_str(), desc_p % ',').full;
#else
      bool ok = parse(rdesc.c_str(), desc_p % ',', space_p).full;
#endif

      if (! ok) {
        THROW_PARSE_ERROR_D("malformed sort condition", description);
      }

      // generator TYPE and OFFSET list
      DynamicSortDescription result;
      DynamicSortProfile profile;

      // modified the simple parsing of the sorted condition with a mildly more complicated one
      for (vector<string>::iterator i = v.begin();  i != v.end();  i++) {
        const string& text = *i;

        // remove any spaces within the string -- helps simple parsing and make everything lower case
        string id = StringUtils::tolower(StringUtils::replace(text," ",""));

        // determine if the sort field is going up or down
        size_t tempInt = id.find("desc");
        profile.ascending = ((tempInt == id.npos) ? 0 : 1);

        // determine the language and encoding to use
        profile.encoding = ET_NONE; // default for string or other data types
        profile.language = LT_NONE; // default for string or other data types

        size_t lBrace = id.find('(');
        size_t rBrace = id.find(')');
        if (lBrace != id.npos) {
          // user entered "asc(...)" or "desc(...)"
          if (rBrace == id.npos || rBrace < lBrace) { // something terribly wrong
            THROW_PARSE_ERROR_D("malformed sort condition", description);
          }
          id = id.substr(lBrace + 1, rBrace - lBrace - 1);
          size_t num = StringUtils::numEntries(id);

          if (num >= 2) {
            profile.language = LangAttributeByName(StringUtils::entry(2,id));
          }
          if (num >= 3) {
            profile.encoding = EncodeAttributeByName(StringUtils::entry(3,id));
          }
        }

        // TODO this will not work well for long attribute lists
        bool found = false;

        for (AttributeDescriptor const * d = desc;  d->name != 0 && ! found;  ++d) {
          if (strcmp(d->name, StringUtils::entry(1,id,",").c_str()) == 0) {
            // TODO add missing types
            // TODO work with string pools
            switch (d->type) {
              case AT_DOUBLE:           profile.dataType = (int) SortedCondition::DL_DOUBLE;  break;
              case AT_FLOAT:            profile.dataType = (int) SortedCondition::DL_FLOAT;   break;
              case AT_INTEGER:          profile.dataType = (int) SortedCondition::DL_INT32;   break;
              case AT_UNSIGNED_INTEGER: profile.dataType = (int) SortedCondition::DL_UINT32;  break;
              case AT_SMALL_INTEGER:    profile.dataType = (int) SortedCondition::DL_INT16;   break;

              case AT_STRING: {

                switch (d->subtype) {
                  case ST_PLAIN:
                    profile.dataType = (int) SortedCondition::DL_STRING;
                    break;

                  case ST_POOL:
                    profile.dataType = (int) SortedCondition::DL_STRING_POOL;
                    break;

                  case ST_SHARED:
                    profile.dataType = (int) SortedCondition::DL_STRING_SHARED;
                    break;

                  default:
                    THROW_INTERNAL_ERROR("unsupported subtype");
                }

                break;
              }

              default:
                THROW_INTERNAL_ERROR("unsupported");
            }

            profile.dataOffset = (int) d->offset;

            result.push_back(profile.dataType);
            result.push_back(profile.dataOffset);
            result.push_back(profile.ascending);
            result.push_back(profile.language);
            result.push_back(profile.encoding);

            found = true;
          }
        }

        if (! found) {
          THROW_PARSE_ERROR_D("unknown sort attribute", id);
        }
      }

      return result;
    }

    // -----------------------------------------------------------------------------
    // less-than for two rows
    // -----------------------------------------------------------------------------

    bool DynamicSort::dynamicLess (DynamicSortDescription const& desc, void const* left, void const* right) {
      for (DynamicSortDescription::const_iterator i = desc.begin();  i != desc.end(); ++i) {
        int type = *i;
        int off  = *++i;
        bool asc = *++i == 0 ? true : false;
        int language = *++i;
        int encoding = *++i;
        char* pl = ((char*) left) + off;
        char* pr = ((char*) right) + off;

        switch (type) {
          case DL_INT16:  { if (dlIsLess<int16_t>(pl, pr, asc)) {return asc;} break; }
          case DL_INT32:  { if (dlIsLess<int32_t>(pl, pr, asc)) {return asc;} break; }
          case DL_INT64:  { if (dlIsLess<int64_t>(pl, pr, asc)) {return asc;} break; }
          case DL_UINT16: { if (dlIsLess<uint16_t>(pl, pr, asc)) {return asc;} break; }
          case DL_UINT32: { if (dlIsLess<uint32_t>(pl, pr, asc)) {return asc;} break; }
          case DL_UINT64: { if (dlIsLess<uint64_t>(pl, pr, asc)) {return asc;} break; }
          case DL_DOUBLE: { if (dlIsLess<double>(pl, pr, asc)) {return asc;} break; }
          case DL_FLOAT:  { if (dlIsLess<float>(pl, pr, asc)) {return asc;} break; }

          // added the language and encoding for string comparisons, we are assuming that
          // we will never code for more than 64K of languages or encodings

          case DL_STRING: {
            switch ((language << 16) + (encoding)) {
              case ((LT_NONE << 16) + ET_NONE): {
                int a = ::strcmp(*reinterpret_cast<char**>(pl), *reinterpret_cast<char**>(pr));

                if (a < 0 ) {
                  return asc;
                }
                else if (a > 0) {
                  return !asc;
                }

                break;
              }


            }

            break;
          }


          default:
            throw "unknown sort instruction";
        }
      }

      return false;
    }

    // -----------------------------------------------------------------------------
    // compare for two rows
    // -----------------------------------------------------------------------------

    int DynamicSort::dynamicCompare (DynamicSortDescription const & desc, void const * left, void const * right) {
      for (DynamicSortDescription::const_iterator i = desc.begin();  i != desc.end(); ++i) {
        int type = *i;
        int off  = *++i;
        bool asc = *++i == 0 ? true : false;
        int language = *++i;
        int encoding = *++i;

        char const* pl = (reinterpret_cast<char const*>(left)) + off;
        char const* pr = (reinterpret_cast<char const*>(right)) + off;

        switch (type) {
          case DL_INT16:  { int a = dlCompare<int16_t>(pl, pr); if (a != 0) {return asc ? a : -a;} break; }
          case DL_INT32:  { int a = dlCompare<int32_t>(pl, pr); if (a != 0) {return asc ? a : -a;} break; }
          case DL_INT64:  { int a = dlCompare<int64_t>(pl, pr); if (a != 0) {return asc ? a : -a;} break; }
          case DL_UINT16: { int a = dlCompare<uint16_t>(pl, pr); if (a != 0) {return asc ? a : -a;} break; }
          case DL_UINT32: { int a = dlCompare<uint32_t>(pl, pr); if (a != 0) {return asc ? a : -a;} break; }
          case DL_UINT64: { int a = dlCompare<uint64_t>(pl, pr); if (a != 0) {return asc ? a : -a;} break; }
          case DL_DOUBLE: { int a = dlCompare<double>(pl, pr); if (a != 0) {return asc ? a : -a;} break; }
          case DL_FLOAT:  { int a = dlCompare<float>(pl, pr); if (a != 0) {return asc ? a : -a;} break; }

          // added the language and encoding for string comparisons, we are assuming that
          // we will never code for more than 64K of languages or encodings

          case DL_STRING: {
            switch ((language << 16) + (encoding)) {
              case ((LT_NONE << 16) + ET_NONE): {
                int a = ::strcmp(*(char**)pl, *(char**)pr);

                if (a != 0) {
                  return asc ? a : -a;
                }

                break;
              }


            }

            break;
          }


          default:
            throw "unknown sort instruction";
        }
      }

      return 0;
    }
  }
}
