////////////////////////////////////////////////////////////////////////////////
/// @brief High-Performance Database Framework made by triagens
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Copyright 2008-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#ifndef TRIAGENS_HPDF_COMMON_H
#define TRIAGENS_HPDF_COMMON_H 1

#include <Basics/Common.h>

#define __MAYBE_UNUSED__ __attribute__ ((__unused__))

namespace triagens {
  namespace hpdf {

    // -----------------------------------------------------------------------------
    // Object and attribute identifier
    // -----------------------------------------------------------------------------

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief type of an object identifier
    ///
    /// Each object is identified by a number of type oid_t.
    ////////////////////////////////////////////////////////////////////////////////

    typedef uint64_t oid_t;

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief type of an attribute identifier
    ///
    /// Each attribute of an object is identified by a number of type aid_t.
    ////////////////////////////////////////////////////////////////////////////////

    typedef uint16_t aid_t;

    // -----------------------------------------------------------------------------
    // StorageType
    // -----------------------------------------------------------------------------

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief string pool storage type
    ////////////////////////////////////////////////////////////////////////////////

    enum StorageType {
      ST_PLAIN = 1,
      ST_POOL = 2,
      ST_SHARED = 3
    };

    // -----------------------------------------------------------------------------
    // TypeAttribute
    // -----------------------------------------------------------------------------

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief possible attribute types
    ////////////////////////////////////////////////////////////////////////////////

    enum TypeAttribute {
      AT_UNKNOWN                = 0,
      AT_BOOLEAN                = 1,
      AT_POINTER                = 2,

      AT_BIG_INTEGER            = 10,
      AT_INTEGER                = 11,
      AT_SMALL_INTEGER          = 12,

      AT_UNSIGNED_BIG_INTEGER   = 13,
      AT_UNSIGNED_INTEGER       = 14,
      AT_UNSIGNED_SMALL_INTEGER = 15,

      AT_DOUBLE                 = 20,
      AT_FLOAT                  = 21,

      AT_DATETIME               = 30,
      AT_DATE                   = 31,

      AT_STRING                 = 40,
      AT_STRING_POINTER         = 41,

      AT_BLOB                   = 42
    };

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief name of an attribute type
    ///
    /// @param[in]
    ///    type           the type attrubute
    ///
    /// @return
    ///    the type attribute as string
    ////////////////////////////////////////////////////////////////////////////////

    std::string NameTypeAttribute (TypeAttribute type);

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief size of an attribute type
    ///
    /// @param[in]
    ///    type           the type attrubute
    ///
    /// @return
    ///    the size of the attribute
    ////////////////////////////////////////////////////////////////////////////////

    int32_t TypeAttributeSize (TypeAttribute type);

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief value of an attribute type
    ///
    /// @param[in]
    ///    type           the type attrubute
    ///
    /// @return
    ///    the type attribute as integer
    ////////////////////////////////////////////////////////////////////////////////

    uint32_t TypeAttributeValue (TypeAttribute type);

    // -----------------------------------------------------------------------------
    // LangAttribute
    // -----------------------------------------------------------------------------

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief possible languages
    ////////////////////////////////////////////////////////////////////////////////

    enum LangAttribute {
      LT_UNKNOWN = -1,
      LT_NONE    = 0,
      LT_DEDE    = 100,
      LT_ENAU    = 200,
      LT_ITIT    = 300
    };

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief name of a language
    ///
    /// @param[in]
    ///    type           the language attrubute
    ///
    /// @return
    ///    the language attribute as string
    ////////////////////////////////////////////////////////////////////////////////

    std::string NameLangAttribute (LangAttribute type);

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief language by name
    ///
    /// @param[in]
    ///    type           the language name in lower case
    ///
    /// @return
    ///    the language attribute
    ///
    /// @warning we are assuming that string "type" is send to this function
    /// in lower case.
    ////////////////////////////////////////////////////////////////////////////////

    LangAttribute LangAttributeByName (std::string const& type);

    // -----------------------------------------------------------------------------
    // EncodeAttribute
    // -----------------------------------------------------------------------------

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief possible encodings
    ////////////////////////////////////////////////////////////////////////////////

    enum EncodeAttribute {
      ET_UNKNOWN  = -1,
      ET_NONE     = 0,
      ET_ASCII    = 0,
      ET_ISO88591 = 100,
      ET_LATIN1   = 100,
      ET_UTF8     = 200
    };

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief name of an encoding
    ///
    /// @param[in]
    ///    type           the encoding
    ///
    /// @return
    ///    the encoding as string
    ////////////////////////////////////////////////////////////////////////////////

    std::string NameEncodeAttribute (EncodeAttribute type);

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief encoding by name
    ///
    /// @param[in]
    ///    type           the encoding name in lower case
    ///
    /// @return
    ///    the encoding
    ///
    /// @warning we are assuming that string "type" is send to this function
    /// in lower case.
    ////////////////////////////////////////////////////////////////////////////////

    EncodeAttribute EncodeAttributeByName (std::string const& type);

    // -----------------------------------------------------------------------------
    // IndexType
    // -----------------------------------------------------------------------------

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief possible index types
    ////////////////////////////////////////////////////////////////////////////////

    enum IndexType {
      IX_UNKNOWN    = -1,
      IX_NO_INDEX   = 0,
      IX_HASH_INDEX = 1,
      IX_PREFIX_INDEX = 2,
      IX_PQUE_INDEX = 3
    };

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief name of an index type
    ///
    /// @param[in]
    ///    type           the index type
    ///
    /// @return
    ///    the index type as string
    ////////////////////////////////////////////////////////////////////////////////

    std::string NameIndexType (IndexType type);


    // -----------------------------------------------------------------------------
    // Priority Queue Algorithms which can be used
    // -----------------------------------------------------------------------------

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief possible priority queue algorithms which can be used
    ////////////////////////////////////////////////////////////////////////////////

    enum PriorityQueueType {
      PQ_UNKNOWN          = -1,
      PQ_NONE             = 0,
      PQ_STANDARD_LIBRARY = 1
    };

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief name of an priority queue type
    ///
    /// @param[in]
    ///    type           the priority queue type
    ///
    /// @return
    ///    the priority queue type as string
    ////////////////////////////////////////////////////////////////////////////////

    std::string NamePriorityQueueType (PriorityQueueType type);

    // -----------------------------------------------------------------------------
    // RowDeletionType
    // -----------------------------------------------------------------------------

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief possible row deletion methods to use with paging
    ////////////////////////////////////////////////////////////////////////////////

    enum RowDeletionType {
      RD_UNKNOWN    = -1,
      RD_STANDARD   = 0,
      RD_UNSTABLE   = 1
    };

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief name of an row deletion method type
    ///
    /// @param[in]
    ///    type           the row deletion type
    ///
    /// @return
    ///    the row deletion type as string
    ////////////////////////////////////////////////////////////////////////////////

    std::string NameRowDeletionType (RowDeletionType type);

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief row deletion type by name
    ///
    /// @param[in]
    ///    type           the row deletion type in lower case
    ///
    /// @return
    ///    the row deletion type
    ///
    /// @warning we are assuming that string "type" is send to this function
    /// in lower case.
    ////////////////////////////////////////////////////////////////////////////////

    RowDeletionType RowDeletionByName (std::string const& type);
  }
}

#endif
