////////////////////////////////////////////////////////////////////////////////
/// @brief general server
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Achim Brandt
/// @author Copyright 2009-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#ifndef FYN_GENERALSERVER_GENERAL_SERVER_H
#define FYN_GENERALSERVER_GENERAL_SERVER_H 1

#include <Basics/Common.h>

#include <Basics/Exceptions.h>
#include <Basics/Logger.h>
#include <Rest/Handler.h>
#include <Rest/ListenTask.h>
#include <Rest/Scheduler.h>
#include <Rest/SocketTask.h>

#include "GeneralServer/GeneralCommTask.h"
#include "GeneralServer/GeneralListenTask.h"


#include "GeneralServer/SpecificCommTask.h"


namespace triagens {
  namespace rest {
    class Dispatcher;
    class Scheduler;

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief general server
    ////////////////////////////////////////////////////////////////////////////////

    template<typename S, typename HF, typename CT>
    class GeneralServer : private TaskManager {
      public:


        ////////////////////////////////////////////////////////////////////////////////
        /// @brief constructs a new general server
        ////////////////////////////////////////////////////////////////////////////////

        explicit
        GeneralServer (Scheduler* scheduler)
          : scheduler(scheduler), handlerFactory(0) {
        }


        ////////////////////////////////////////////////////////////////////////////////
        /// @brief destructs a general server
        ////////////////////////////////////////////////////////////////////////////////

        virtual ~GeneralServer () {
          if (handlerFactory != 0) {
            delete handlerFactory;
          }
        }

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief sets a handler factory
        ///
        /// Note that the general server claims the ownership of the factory.
        ////////////////////////////////////////////////////////////////////////////////

        void setHandlerFactory (HF* factory) {
          handlerFactory = factory;
        }

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief creates a new request
        ////////////////////////////////////////////////////////////////////////////////

        typename HF::GeneralRequest * createRequest (char const* ptr, size_t length) {
          return handlerFactory == 0 ? 0 : handlerFactory->createRequest(ptr, length);
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief creates a new request
        ////////////////////////////////////////////////////////////////////////////////

        typename HF::GeneralHandler * createHandler (typename HF::GeneralRequest* request) {
          return handlerFactory == 0 ? 0 : handlerFactory->createHandler(request);
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief returns header and body size restrictions
        ////////////////////////////////////////////////////////////////////////////////

        pair<size_t, size_t> sizeRestrictions () {
          static size_t m = (size_t) -1;

          if (handlerFactory == 0) {
            return make_pair(m, m);
          }
          else {
            return handlerFactory->sizeRestrictions();
          }
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief starts listening for general connections on given port
        ///
        /// The token is passed to the listen task which sends it back when accepting
        /// a connection.
        ////////////////////////////////////////////////////////////////////////////////

        bool addPort (int port, bool reuseAddress)  {
          return addPort("", port, reuseAddress);
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief starts listening for general connections on given port and address
        ///
        /// The token is passed to the listen task which sends it back when accepting
        /// a connection.
        ////////////////////////////////////////////////////////////////////////////////

        bool addPort (string const& address, int port, bool reuseAddress) {
          ListenTask* task = new GeneralListenTask<S>(dynamic_cast<S*>(this), address, port, reuseAddress);

          if (! task->isBound()) {
            deleteTask(task);
            return false;
          }

          scheduler->registerTask(task);

          return true;
        }

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief handles connection request
        ////////////////////////////////////////////////////////////////////////////////

        virtual void handleConnected (socket_t socket, ConnectionInfo& info) {


          SocketTask* task = new SpecificCommTask<S, HF, CT>(dynamic_cast<S*>(this), socket, info);


          scheduler->registerTask(task);
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief handles a connection close
        ////////////////////////////////////////////////////////////////////////////////

        void handleCommunicationClosed (Task* task)  {
          scheduler->destroyTask(task);
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief handles a connection failure
        ////////////////////////////////////////////////////////////////////////////////

        void handleCommunicationFailure (Task* task)  {
          scheduler->destroyTask(task);
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief handles a request
        ////////////////////////////////////////////////////////////////////////////////

        bool handleRequest (CT * task, typename HF::GeneralHandler * handler) {

          // execute handle and requeue
          bool done = false;

          while (! done) {


            Handler::status_e status = handleRequestDirectly(task, handler);

            if (status != Handler::HANDLER_REQUEUE) {
              done = true;
              delete handler;
            }
            else {
              continue;
            }


          }

          return true;
        }

      protected:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief handle request directly
        ////////////////////////////////////////////////////////////////////////////////

        Handler::status_e handleRequestDirectly (CT* task, typename HF::GeneralHandler * handler) {
          Handler::status_e status = Handler::HANDLER_FAILED;

          try {
            try {
              status = handler->execute();
            }
            catch (basics::TriagensError const& ex) {
              handler->handleError(ex);
            }
            catch (std::exception const& ex) {
              basics::InternalError err(ex);

              handler->handleError(err);
            }
            catch (...) {
              basics::InternalError err;
              handler->handleError(err);
            }

            if (status == Handler::HANDLER_REQUEUE) {
              return status;
            }

            typename HF::GeneralResponse * response = handler->getResponse();

            if (response == 0) {
              basics::InternalError err("no response received from handler");

              handler->handleError(err);
              response = handler->getResponse();
            }

            if (response != 0) {
              task->handleResponse(response);
            }
            else {
              LOGGER_ERROR << "cannot get any response in " << __FILE__ << "@" << __LINE__;
            }
          }
          catch (basics::TriagensError const& ex) {
            LOGGER_ERROR << "caught exception in " << __FILE__ << "@" << __LINE__ << ": " << DIAGNOSTIC_INFORMATION(ex);
          }
          catch (std::exception const& ex) {
            LOGGER_ERROR << "caught exception in " << __FILE__ << "@" << __LINE__ << ": " << ex.what();
          }
          catch (...) {
            LOGGER_ERROR << "caught exception in " << __FILE__ << "@" << __LINE__;
          }

          return status;
        }

      protected:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief the scheduler
        ////////////////////////////////////////////////////////////////////////////////

        Scheduler* scheduler;


        ////////////////////////////////////////////////////////////////////////////////
        /// @brief the handler factory
        ////////////////////////////////////////////////////////////////////////////////

        HF* handlerFactory;
    };
  }
}

#endif
