////////////////////////////////////////////////////////////////////////////////
/// @brief Thread
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Achim Brandt
/// @author Copyright 2008-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "Thread.h"

#include <errno.h>
#include <signal.h>

#include <Basics/ConditionLocker.h>
#include <Basics/Logger.h>

namespace triagens {
  namespace basics {

    // -----------------------------------------------------------------------------
    // static public methods
    // -----------------------------------------------------------------------------

    bool Thread::createKey (tlsKey& key, void (*destructor)(void*)) {
      return pthread_key_create(&key, destructor) == 0;
    }



    void* Thread::specific (tlsKey& key) {
      return pthread_getspecific(key);
    }



    bool Thread::setSpecific (tlsKey& key, void* value) {
      return pthread_setspecific(key, value) == 0;
    }



    Thread::pid_t Thread::currentProcessId () {
      return ::getpid();
    }



    Thread::pid_t Thread::currentThreadProcessId () {
#ifdef HAVE_GETTID
      return ::gettid();
#else
      return ::getpid();
#endif
    }



    Thread::tid_t Thread::currentThreadId () {
      return (intptr_t) pthread_self();
    }

    // -----------------------------------------------------------------------------
    // constructors and destructors
    // -----------------------------------------------------------------------------

    Thread::Thread (const string& name)
      : _name(name), _finishedCondition(0), _running(0) {
    }



    Thread::~Thread () {
      if (_running != 0) {
        LOGGER_WARNING << "forcefully shuting down thread '" << _name << "'";
        pthread_cancel(_thread);
        pthread_detach(_thread);
      }
      else {
        LOGGER_TRACE << "joining with stopped thread '" << _name << "'";
        pthread_join(_thread, 0);
      }
    }

    // -----------------------------------------------------------------------------
    // public methods
    // -----------------------------------------------------------------------------

    bool Thread::isRunning () {
      return _running != 0;
    }



    intptr_t Thread::threadId () {
      return (intptr_t) _thread;
    }



    bool Thread::start (ConditionVariable * finishedCondition) {
      _finishedCondition = finishedCondition;

      int rc = pthread_create(&_thread, 0, &startThread, this);

      if (rc != 0) {
        LOGGER_ERROR << "could not start thread '" << _name << "': " << strerror(errno);
        return false;
      }

      return true;
    }



    void Thread::stop () {
      if (_running != 0) {
        pthread_cancel(_thread);
      }
    }



    void Thread::join () {
      pthread_join(_thread, 0);
    }



    void Thread::sendSignal (int signal) {
      if (_running != 0) {
        int rc = pthread_kill(_thread, signal);

        if (rc != 0) {
          LOGGER_ERROR << "could not send signal to thread '" << _name << "': " << strerror(errno);
        }
      }
    }

    // -----------------------------------------------------------------------------
    // protected methods
    // -----------------------------------------------------------------------------

    void Thread::allowAsynchronousCancelation () {
      pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, 0);
    }

    // -----------------------------------------------------------------------------
    // private methods
    // -----------------------------------------------------------------------------

    void* Thread::startThread (void* arg) {
      sigset_t all;
      sigfillset(&all);

      pthread_sigmask(SIG_SETMASK, &all, 0);

      Thread * ptr = (Thread *) arg;
      ptr->runMe();
      return 0;
    }



    void Thread::runMe () {
      _running = 1;
      run();
      _running = 0;

      if (_finishedCondition != 0) {
        CONDITION_LOCKER(locker, *_finishedCondition);
        locker.broadcast();
      }
    }
  }
}
