////////////////////////////////////////////////////////////////////////////////
/// @brief collection of random functions and classes
///
/// @file
/// Thread-safe random generator
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Copyright 2009-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#ifndef TRIAGENS_BASICS_RANDOM_H
#define TRIAGENS_BASICS_RANDOM_H 1

#include <Basics/Common.h>

namespace triagens {
  namespace basics {

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief collection of random functions and classes
    ////////////////////////////////////////////////////////////////////////////////

    namespace Random {

      ////////////////////////////////////////////////////////////////////////////////
      /// @brief type of the random generator
      ////////////////////////////////////////////////////////////////////////////////

      enum random_e {
        RAND_MERSENNE = 1,
        RAND_RANDOM = 2,
        RAND_URANDOM = 3,
        RAND_COMBINED = 4
      };


      ////////////////////////////////////////////////////////////////////////////////
      /// @ingroup Utilities
      /// @brief randomize random
      ////////////////////////////////////////////////////////////////////////////////

      void seed ();

      ////////////////////////////////////////////////////////////////////////////////
      /// @ingroup Utilities
      /// @brief selects random generator
      ////////////////////////////////////////////////////////////////////////////////

      random_e selectVersion (random_e);

      ////////////////////////////////////////////////////////////////////////////////
      /// @ingroup Utilities
      /// @brief returns random generator version
      ////////////////////////////////////////////////////////////////////////////////

      random_e currentVersion ();

      ////////////////////////////////////////////////////////////////////////////////
      /// @brief interval inclusive both margins
      ////////////////////////////////////////////////////////////////////////////////

      int32_t interval (int32_t left, int32_t right);

      // -----------------------------------------------------------------------------
      // uniform integer generator
      // -----------------------------------------------------------------------------

      ////////////////////////////////////////////////////////////////////////////////
      /// @ingroup Utilities
      /// @brief constructor, the range includes both left and right
      ////////////////////////////////////////////////////////////////////////////////

      class UniformInteger : boost::noncopyable {
        public:

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief constructor, the range includes both margins
          ////////////////////////////////////////////////////////////////////////////////

          UniformInteger (int32_t left, int32_t right)
            : left(left), right(right) {
          }

        public:

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief returns a random integer between left and right inclusive
          ////////////////////////////////////////////////////////////////////////////////

          int32_t random ();

        private:
          int32_t left;
          int32_t right;
      };

      // -----------------------------------------------------------------------------
      // uniform character generator
      // -----------------------------------------------------------------------------

      ////////////////////////////////////////////////////////////////////////////////
      /// @ingroup Utilities
      /// @brief uniform character string
      ////////////////////////////////////////////////////////////////////////////////

      class UniformCharacter : boost::noncopyable {
        public:

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief constructor
          ////////////////////////////////////////////////////////////////////////////////

          explicit
          UniformCharacter (size_t length);

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief constructor
          ////////////////////////////////////////////////////////////////////////////////

          explicit
          UniformCharacter (string const& characters);

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief constructor
          ////////////////////////////////////////////////////////////////////////////////

          UniformCharacter (size_t length, string const& characters);

        public:

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief returns a random string of fixed length
          ////////////////////////////////////////////////////////////////////////////////

          string random ();

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief returns a random string of given length
          ////////////////////////////////////////////////////////////////////////////////

          string random (size_t length);

        private:
          size_t length;
          string const characters;
          UniformInteger generator;
      };
    }
  }
}

#endif
